/*******************************************************************************
 * Copyright (c) 2012 Obeo.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.compare.ide.internal.extension;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IRegistryEventListener;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.compare.extension.PostProcessorRegistry;

/**
 * This listener will allow us to be aware of contribution changes against the model resolver extension point.
 * 
 * @author <a href="mailto:cedric.notot@obeo.fr">Cedric Notot</a>
 */
public class PostProcessorRegistryListener implements IRegistryEventListener {
	/** Name of the extension point to parse for extensions. */
	public static final String POST_PROCESSOR_EXTENSION_POINT = "org.eclipse.emf.compare.postProcessor"; //$NON-NLS-1$

	/** Name of the extension point's "postProcessor" tag. */
	private static final String POST_PROCESSOR_TAG = "postProcessor"; //$NON-NLS-1$

	/** The post processor registry to which extension will be registered. */
	private final PostProcessorRegistry registry;

	/**
	 * Creates a new registry listener with the given post processor registry to which extension will be
	 * registered.
	 * 
	 * @param registry
	 *            the post processor registry to which extension will be registered.
	 */
	public PostProcessorRegistryListener(PostProcessorRegistry registry) {
		this.registry = registry;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.core.runtime.IRegistryEventListener#added(org.eclipse.core.runtime.IExtension[])
	 */
	public void added(IExtension[] extensions) {
		for (IExtension extension : extensions) {
			parseExtension(extension);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.core.runtime.IRegistryEventListener#added(org.eclipse.core.runtime.IExtensionPoint[])
	 */
	public void added(IExtensionPoint[] extensionPoints) {
		// no need to listen to this event
	}

	/**
	 * Though this listener reacts to the extension point changes, there could have been contributions before
	 * it's been registered. This will parse these initial contributions.
	 */
	public void parseInitialContributions() {
		final IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();

		for (IExtension extension : extensionRegistry.getExtensionPoint(POST_PROCESSOR_EXTENSION_POINT)
				.getExtensions()) {
			parseExtension(extension);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.core.runtime.IRegistryEventListener#removed(org.eclipse.core.runtime.IExtension[])
	 */
	public void removed(IExtension[] extensions) {
		for (IExtension extension : extensions) {
			final IConfigurationElement[] configElements = extension.getConfigurationElements();
			for (IConfigurationElement elem : configElements) {
				if (POST_PROCESSOR_TAG.equals(elem.getName())) {
					final String postProcessorClassName = elem
							.getAttribute(PostProcessorIDEDescriptor.POST_PROCESSOR_CLASS_ATTRIBUTE);
					registry.removePostProcessor(postProcessorClassName);
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.core.runtime.IRegistryEventListener#removed(org.eclipse.core.runtime.IExtensionPoint[])
	 */
	public void removed(IExtensionPoint[] extensionPoints) {
		// no need to listen to this event
	}

	/**
	 * Parses a single extension contribution.
	 * 
	 * @param extension
	 *            Parses the given extension and adds its contribution to the registry.
	 */
	private void parseExtension(IExtension extension) {
		final IConfigurationElement[] configElements = extension.getConfigurationElements();
		for (IConfigurationElement element : configElements) {
			if (POST_PROCESSOR_TAG.equals(element.getName())) {
				registry.addPostProcessor(new PostProcessorIDEDescriptor(element));
			}
		}
	}
}
