/*******************************************************************************
 * Copyright (c) 2011-2012 Dennis Wagelaar, Vrije Universiteit Brussel.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Dennis Wagelaar, Vrije Universiteit Brussel
 *******************************************************************************/
package org.eclipse.m2m.atl.emftvm.jit;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.m2m.atl.emftvm.CodeBlock;
import org.eclipse.m2m.atl.emftvm.EmftvmPackage;
import org.eclipse.m2m.atl.emftvm.ExecEnv;
import org.eclipse.m2m.atl.emftvm.Field;
import org.eclipse.m2m.atl.emftvm.Model;
import org.eclipse.m2m.atl.emftvm.Operation;
import org.eclipse.m2m.atl.emftvm.Rule;
import org.eclipse.m2m.atl.emftvm.RuleMode;
import org.eclipse.m2m.atl.emftvm.util.DuplicateEntryException;
import org.eclipse.m2m.atl.emftvm.util.EMFTVMUtil;
import org.eclipse.m2m.atl.emftvm.util.LazyBagOnCollection;
import org.eclipse.m2m.atl.emftvm.util.LazyList;
import org.eclipse.m2m.atl.emftvm.util.LazyListOnList;
import org.eclipse.m2m.atl.emftvm.util.LazySetOnSet;
import org.eclipse.m2m.atl.emftvm.util.NativeTypes;
import org.eclipse.m2m.atl.emftvm.util.StackFrame;
import org.eclipse.m2m.atl.emftvm.util.Tuple;
import org.eclipse.m2m.atl.emftvm.util.VMException;

/**
 * Interface for code generated by the {@link JITCodeBlock} compiler.
 * @author <a href="mailto:dennis.wagelaar@vub.ac.be">Dennis Wagelaar</a>
 */
public abstract class JITCodeBlock {

	/**
	 * Singleton instance of the {@link ExecEnv} {@link EClass}.
	 */
	protected static final EClass EXEC_ENV = EmftvmPackage.eINSTANCE.getExecEnv();

	private static final Object[] EMPTY = new Object[0];

	/**
	 * The {@link CodeBlock} that this {@link JITCodeBlock} represents.
	 */
	protected final CodeBlock cb;

	/**
	 * The nested {@link CodeBlock}s of <code>cb</code>.
	 */
	protected final EList<CodeBlock> nested;

	/**
	 * The EMF context type for this {@link CodeBlock}, or <code>null</code>.
	 */
	protected final EClass eContext;

	/**
	 * The Java context type for this {@link CodeBlock}, or <code>null</code>.
	 */
	protected final Class<?> context;

	/**
	 * Creates a new {@link JITCodeBlock}.
	 * @param cb the {@link CodeBlock} that this {@link JITCodeBlock} represents
	 */
	public JITCodeBlock(final CodeBlock cb) {
		super();
		this.cb = cb;
		this.nested = cb.getNested();
		final Operation op = cb.getOperation();
		final EClassifier ec;
		if (op != null) {
			ec = op.getEContext();
		} else {
			final Field f = cb.getField();
			if (f != null) {
				ec = f.getEContext();
			} else {
				ec = null;
			}
		}
		if (ec instanceof EClass) {
			this.eContext = (EClass)ec;
		} else {
			this.eContext = null;
		}
		if (ec != null) {
			this.context = ec.getInstanceClass();
		} else {
			this.context = null;
		}
	}

	/**
	 * Executes this {@link JITCodeBlock}, using the parameters stored in <code>frame</code>.
	 * @param frame the {@link StackFrame} to use during execution.
	 * @return the <code>frame</code>, containing any return value on top.
	 */
	public abstract Object execute(StackFrame frame);

	/**
	 * Implements the SET instruction.
	 * @param o the object to set the value for
	 * @param v the value to set
	 * @param cb the original non-JIT'ed code block
	 * @param frame the current stack frame
	 * @param propname the name of the property to set
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	protected static void set(final Object o, final Object v, final CodeBlock cb, final StackFrame frame, final String propname) 
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final Rule rule = cb.getRule();
		
		if (o instanceof EObject) {
			final EObject eo = (EObject)o;
			final EClass type = eo.eClass();
			final boolean queueSet = env.getCurrentPhase() == RuleMode.AUTOMATIC_SINGLE && env.getInoutModelOf(eo) != null;
			Field field = null;
			if (rule != null) {
				field = rule.findField(type, propname);
			}
			if (field == null) {
				field = env.findField(type, propname);
				if (field != null) {
					if (queueSet) {
						env.queueForSet(field, o, v, frame);
					} else {
						if (env.getInputModelOf(eo) != null) {
							throw new IllegalArgumentException(
									String.format("Cannot set properties of %s, as it is contained in an input model",
											EMFTVMUtil.toPrettyString(eo, env)));
						}
						if (env.getOutputModelOf(eo) != null) {
							throw new IllegalArgumentException(String.format(
									"Setting transient field %s of %s, which cannot be read back as %1s is contained in an output model",
									propname, EMFTVMUtil.toPrettyString(eo, env)));
						}
						field.setValue(o, v);
					}
					return;
				}
			} else {
				// Treat rule fields as local variables
				field.setValue(o, v);
				return;
			}
			final EStructuralFeature sf = type.getEStructuralFeature(propname);
			if (sf != null) {
				if (queueSet) {
					env.queueForSet(sf, eo, v, frame);
				} else {
					EMFTVMUtil.set(env, eo, sf, v);
				}
				return;
			}
			final Resource resource = eo.eResource();
			if (EMFTVMUtil.XMI_ID_FEATURE.equals(propname) && resource instanceof XMIResource) { //$NON-NLS-1$
				if (queueSet) {
					env.queueXmiIDForSet(eo, v, frame);
				} else {
					((XMIResource)resource).setID(eo, v.toString());
				}
				return;
			}
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}

		// o is a regular Java object
		final Class<?> type = o == null ? Void.TYPE : o.getClass();
		Field field = null;
		if (rule != null) {
			field = rule.findField(type, propname);
		}
		if (field == null) {
			field = env.findField(type, propname);
		}
		if (field != null) {
			field.setValue(o, v);
			return;
		}
		try {
			final java.lang.reflect.Field f = type.getField(propname);
			f.set(o, v);
		} catch (NoSuchFieldException e) {
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}
	}

	/**
	 * Implements the GET instruction.
	 * @param o the object to get the value for
	 * @param cb the original non-JIT'ed code block
	 * @param frame the current stack frame
	 * @param propname the name of the property to set
	 * @return the property value
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	@SuppressWarnings("unchecked")
	protected static Object get(final Object o, final CodeBlock cb, final StackFrame frame, final String propname) 
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final Rule rule = cb.getRule();

		if (o instanceof EObject) {
			final EObject eo = (EObject)o;
			final EClass type = eo.eClass();
			Field field = null;
			if (rule != null) {
				field = rule.findField(type, propname);
			}
			if (field == null) {
				field = env.findField(type, propname);
				if (field != null) {
					if (env.getOutputModelOf(eo) != null) {
						throw new IllegalArgumentException(String.format(
								"Cannot read properties of %s, as it is contained in an output model", EMFTVMUtil.toPrettyString(eo, env)));
					}
					return field.getValue(o, frame);
				}
			} else {
				// Treat rule fields as local variables
				return field.getValue(o, frame);
			}
			final EStructuralFeature sf = type.getEStructuralFeature(propname);
			if (sf != null) {
				return EMFTVMUtil.get(env, eo, sf);
			}
			final Resource resource = eo.eResource();
			if (EMFTVMUtil.XMI_ID_FEATURE.equals(propname) && resource instanceof XMIResource) { //$NON-NLS-1$
				return ((XMIResource)resource).getID(eo);
			}
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}

		// o is a regular Java object
		final Class<?> type = o == null ? Void.TYPE : o.getClass();
		Field field = null;
		if (rule != null) {
			field = rule.findField(type, propname);
		}
		if (field == null) {
			field = env.findField(type, propname);
		}
		if (field != null) {
			return field.getValue(o, frame);
		}

		if (o instanceof Tuple && ((Tuple) o).asMap().containsKey(propname)) {
			return ((Tuple) o).get(propname);
		}

		try {
			final java.lang.reflect.Field f = type.getField(propname);
			final Object result = f.get(o);
			if (result instanceof List<?>) {
				return new LazyListOnList<Object>((List<Object>)result);
			} else if (result instanceof java.util.Set<?>) {
				return new LazySetOnSet<Object>((java.util.Set<Object>)result);
			} else if (result instanceof Collection<?>) {
				return new LazyBagOnCollection<Object>((Collection<Object>)result);
			} else {
				return result;
			}
		} catch (NoSuchFieldException e) {
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}
	}

	/**
	 * Implements the GET instruction. Assumes this code block is not contained
	 * in a rule, or the rule does not contain a {@link Field} with the given
	 * <code>propname</code>.
	 * @param o the object to get the value for
	 * @param cb the original non-JIT'ed code block
	 * @param frame the current stack frame
	 * @param propname the name of the property to set
	 * @return the property value
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	@SuppressWarnings("unchecked")
	protected static Object get(final Object o, final StackFrame frame, final String propname) 
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		if (o instanceof EObject) {
			final EObject eo = (EObject)o;
			final EClass type = eo.eClass();
			final Field field = env.findField(type, propname);
			if (field != null) {
				if (env.getOutputModelOf(eo) != null) {
					throw new IllegalArgumentException(String.format("Cannot read properties of %s, as it is contained in an output model",
							EMFTVMUtil.toPrettyString(eo, env)));
				}
				return field.getValue(o, frame);
			}
			final EStructuralFeature sf = type.getEStructuralFeature(propname);
			if (sf != null) {
				return EMFTVMUtil.get(env, eo, sf);
			}
			final Resource resource = eo.eResource();
			if (EMFTVMUtil.XMI_ID_FEATURE.equals(propname) && resource instanceof XMIResource) { //$NON-NLS-1$
				return ((XMIResource)resource).getID(eo);
			}
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}

		// o is a regular Java object
		final Class<?> type = o == null ? Void.TYPE : o.getClass();
		final Field field = env.findField(type, propname);
		if (field != null) {
			return field.getValue(o, frame);
		}

		if (o instanceof Tuple && ((Tuple) o).asMap().containsKey(propname)) {
			return ((Tuple) o).get(propname);
		}

		try {
			final java.lang.reflect.Field f = type.getField(propname);
			final Object result = f.get(o);
			if (result instanceof List<?>) {
				return new LazyListOnList<Object>((List<Object>)result);
			} else if (result instanceof java.util.Set<?>) {
				return new LazySetOnSet<Object>((java.util.Set<Object>)result);
			} else if (result instanceof Collection<?>) {
				return new LazyBagOnCollection<Object>((Collection<Object>)result);
			} else {
				return result;
			}
		} catch (NoSuchFieldException e) {
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}
	}

	/**
	 * Implements the GET instruction. Assumes there is no {@link Field} with
	 * the given <code>propname</code>.
	 * @param o the object to get the value for
	 * @param env the execution environment
	 * @param propname the name of the property to set
	 * @return the property value
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	@SuppressWarnings("unchecked")
	protected static Object get(final Object o, final ExecEnv env, final String propname) 
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		if (o instanceof EObject) {
			final EObject eo = (EObject)o;
			final EClass type = eo.eClass();
			final EStructuralFeature sf = type.getEStructuralFeature(propname);
			if (sf != null) {
				return EMFTVMUtil.get(env, eo, sf);
			}
			final Resource resource = eo.eResource();
			if (EMFTVMUtil.XMI_ID_FEATURE.equals(propname) && resource instanceof XMIResource) { //$NON-NLS-1$
				return ((XMIResource)resource).getID(eo);
			}
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		} else if (o instanceof Tuple && ((Tuple) o).asMap().containsKey(propname)) {
			return ((Tuple) o).get(propname);
		}

		// o is a regular Java object
		final Class<?> type = o == null ? Void.TYPE : o.getClass();
		try {
			final java.lang.reflect.Field f = type.getField(propname);
			final Object result = f.get(o);
			if (result instanceof List<?>) {
				return new LazyListOnList<Object>((List<Object>)result);
			} else if (result instanceof java.util.Set<?>) {
				return new LazySetOnSet<Object>((java.util.Set<Object>)result);
			} else if (result instanceof Collection<?>) {
				return new LazyBagOnCollection<Object>((Collection<Object>)result);
			} else {
				return result;
			}
		} catch (NoSuchFieldException e) {
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}
	}

	/**
	 * Implements the GET_TRANS instruction.
	 * @param o the object to get the value for
	 * @param cb the original non-JIT'ed code block
	 * @param frame the current stack frame
	 * @param propname the name of the property to set
	 * @return the property values
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	protected static Collection<Object> getTrans(final Object o, final CodeBlock cb, final StackFrame frame, final String propname)
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final Rule rule = cb.getRule();

		if (o instanceof EObject) {
			final EObject eo = (EObject)o;
			final EClass type = eo.eClass();
			Field field = null;
			if (rule != null) {
				field = rule.findField(type, propname);
			}
			if (field == null) {
				field = env.findField(type, propname);
				if (field != null) {
					if (env.getOutputModelOf(eo) != null) {
						throw new IllegalArgumentException(String.format(
								"Cannot read properties of %s, as it is contained in an output model", EMFTVMUtil.toPrettyString(eo, env)));
					}
					return EMFTVMUtil.getTrans(o, field, frame, new LazyList<Object>());
				}
			} else {
				// Treat rule fields as local variables
				return EMFTVMUtil.getTrans(o, field, frame, new LazyList<Object>());
			}
			final EStructuralFeature sf = type.getEStructuralFeature(propname);
			if (sf == null) {
				throw new NoSuchFieldException(String.format("Field %s::%s not found", EMFTVMUtil.toPrettyString(type, env), propname));
			}
			return EMFTVMUtil.getTrans(eo, sf, env, new LazyList<Object>());
		} else {
			final Class<?> type = o.getClass();
			Field field = null;
			if (rule != null) {
				field = rule.findField(type, propname);
			}
			if (field == null) {
				field = env.findField(type, propname);
			}
			if (field != null) {
				return EMFTVMUtil.getTrans(o, field, frame, new LazyList<Object>());
			} else {
				final java.lang.reflect.Field f = type.getField(propname);
				return EMFTVMUtil.getTrans(o, f, new LazyList<Object>());
			}
		}
	}

	/**
	 * Implements the GET_SUPER instruction.
	 * @param o the source object
	 * @param context the current field context type
	 * @param propname
	 * @param frame
	 * @return the property value
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	@SuppressWarnings("unchecked")
	protected static Object getSuper(final Object o, final EClass context, final String propname, final StackFrame frame) 
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final List<EClass> superTypes = context.getESuperTypes();
		final java.util.Set<Object> superFs = new LinkedHashSet<Object>();

		if (o instanceof EObject) {
			// o may have EStructuralFeatures
			for (EClass superType : superTypes) {
				Object superF = env.findField(superType, propname);
				if (superF != null) {
					superFs.add(superF);
				} else {
					superF = superType.getEStructuralFeature(propname);
					if (superF != null) {
						superFs.add(superF);
					} else if (superType.getInstanceClass() != null) {
						try {
							superF = superType.getInstanceClass().getField(propname);
							assert superF != null;
							superFs.add(superF);
						} catch (NoSuchFieldException e) {
							// not found - skip
						}
					}
				}
			}
		} else {
			// o is a regular Java object - may be null
			for (EClass superType : superTypes) {
				Object superF = env.findField(superType, propname);
				if (superF != null) {
					superFs.add(superF);
				} else if (superType.getInstanceClass() != null) {
					try {
						superF = superType.getInstanceClass().getField(propname);
						assert superF != null;
						superFs.add(superF);
					} catch (NoSuchFieldException e) {
						// not found - skip
					}
				}
			}
		}

		if (superFs.size() > 1) {
			throw new DuplicateEntryException(String.format(
					"More than one super-field found for context %s: %s",
					context, superFs));
		}
		if (!superFs.isEmpty()) {
			final Object superF = superFs.iterator().next();
			if (superF instanceof Field) {
				final Field field = (Field) superF;
				if (o instanceof EObject) {
					final EObject eo = (EObject) o;
					if (field.getRule() == null && env.getOutputModelOf(eo) != null) {
						throw new IllegalArgumentException(String.format(
								"Cannot read properties of %s, as it is contained in an output model", EMFTVMUtil.toPrettyString(eo, env)));
					}
				}
				return field.getValue(o, frame);
			} else if (superF instanceof EStructuralFeature) {
				return EMFTVMUtil.get(env, (EObject)o, (EStructuralFeature)superF);
			} else {
				final Object result = ((java.lang.reflect.Field)superF).get(o);
				if (result instanceof List<?>) {
					return new LazyListOnList<Object>((List<Object>)result);
				} else if (result instanceof java.util.Set<?>) {
					return new LazySetOnSet<Object>((java.util.Set<Object>)result);
				} else if (result instanceof Collection<?>) {
					return new LazyBagOnCollection<Object>((Collection<Object>)result);
				} else {
					return result;
				}
			}
		}

		throw new NoSuchFieldException(String.format("Super-field of %s::%s not found", 
				EMFTVMUtil.toPrettyString(context, env), propname));
	}

	/**
	 * Implements the GET_SUPER instruction.
	 * @param o the source object
	 * @param context the current field context type
	 * @param propname
	 * @param frame
	 * @return the property value
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	@SuppressWarnings("unchecked")
	protected static Object getSuper(final Object o, final Class<?> context, final String propname, final StackFrame frame) 
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final Class<?> superType = context.getSuperclass();
		final java.util.Set<Object> superFs = new LinkedHashSet<Object>();

		if (o instanceof EObject) {
			// o may have EStructuralFeatures
			Object superF = env.findField(superType, propname);
			if (superF != null) {
				superFs.add(superF);
			} else {
				try {
					superF = superType.getField(propname);
					assert superF != null;
					superFs.add(superF);
				} catch (NoSuchFieldException e) {
					// not found - skip
				}
			}
		} else {
			// o is a regular Java object - may be null
			Object superF = env.findField(superType, propname);
			if (superF != null) {
				superFs.add(superF);
			} else {
				try {
					superF = superType.getField(propname);
					assert superF != null;
					superFs.add(superF);
				} catch (NoSuchFieldException e) {
					// not found - skip
				}
			}
		}

		if (superFs.size() > 1) {
			throw new DuplicateEntryException(String.format(
					"More than one super-field found for context %s: %s",
					context, superFs));
		}
		if (!superFs.isEmpty()) {
			final Object superF = superFs.iterator().next();
			if (superF instanceof Field) {
				return ((Field)superF).getValue(o, frame);
			} else if (superF instanceof EStructuralFeature) {
				return EMFTVMUtil.get(env, (EObject)o, (EStructuralFeature)superF);
			} else {
				final Object result = ((java.lang.reflect.Field)superF).get(o);
				if (result instanceof List<?>) {
					return new LazyListOnList<Object>((List<Object>)result);
				} else if (result instanceof java.util.Set<?>) {
					return new LazySetOnSet<Object>((java.util.Set<Object>)result);
				} else if (result instanceof Collection<?>) {
					return new LazyBagOnCollection<Object>((Collection<Object>)result);
				} else {
					return result;
				}
			}
		}

		throw new NoSuchFieldException(String.format("Super-field of %s::%s not found", 
				EMFTVMUtil.toPrettyString(context, env), propname));
	}

	/**
	 * Implements the SET_STATIC instruction.
	 * @param o the object type to set the value for
	 * @param v the value to set
	 * @param cb the original non-JIT'ed code block
	 * @param env the execution environment
	 * @param propname the name of the property to set
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	protected static void setStatic(final Object o, final Object v, final CodeBlock cb, final ExecEnv env, final String propname)
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final Object ort = EMFTVMUtil.getRegistryType(o);
		final Rule rule = cb.getRule();

		if (ort instanceof EClass) {
			final EClass type = (EClass)ort;
			Field field = null;
			if (rule != null) {
				field = rule.findStaticField(type, propname);
			}
			if (field == null) {
				field = env.findStaticField(type, propname);
			}
			if (field != null) {
				field.setStaticValue(v);
			} else {
				throw new NoSuchFieldException(String.format("Field %s::%s not found", 
						EMFTVMUtil.toPrettyString(type, env), propname));
			}
		} else if (ort instanceof Class<?>) {
			final Class<?> type = (Class<?>)ort;
			Field field = null;
			if (rule != null) {
				field = rule.findStaticField(type, propname);
			}
			if (field == null) {
				field = env.findStaticField(type, propname);
			}
			if (field != null) {
				field.setValue(ort, v);	
			} else {
				final java.lang.reflect.Field f = type.getField(propname);
				if (Modifier.isStatic(f.getModifiers())) {
					f.set(null, v);
				} else {
					throw new NoSuchFieldException(String.format("Field %s::%s not found", 
							EMFTVMUtil.toPrettyString(type, env), propname));
				}
			}
		} else {
			throw new IllegalArgumentException(String.format("%s is not a type", 
					EMFTVMUtil.toPrettyString(ort, env)));
		}
	}

	/**
	 * Implements the GET_STATIC instruction.
	 * @param o the object to get the value for
	 * @param cb the original non-JIT'ed code block
	 * @param frame the current stack frame
	 * @param propname the name of the property to set
	 * @return the property value
	 * @throws NoSuchFieldException 
	 * @throws IllegalAccessException 
	 * @throws IllegalArgumentException 
	 */
	protected static Object getStatic(final Object o, final CodeBlock cb, final StackFrame frame, final String propname)
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final Object ort = EMFTVMUtil.getRegistryType(o);
		final Rule rule = cb.getRule();

		if (ort instanceof EClass) {
			final EClass type = (EClass)ort;
			Field field = null;
			if (rule != null) {
				field = rule.findStaticField(type, propname);
			}
			if (field == null) {
				field = env.findStaticField(type, propname);
			}
			if (field != null) {
				return field.getStaticValue(frame);
			} else {
				throw new NoSuchFieldException(String.format("Field %s::%s not found", 
						EMFTVMUtil.toPrettyString(type, env), propname));
			}
		} else if (ort instanceof Class<?>) {
			final Class<?> type = (Class<?>)ort;
			Field field = null;
			if (rule != null) {
				field = rule.findStaticField(type, propname);
			}
			if (field == null) {
				field = env.findStaticField(type, propname);
			}
			if (field != null) {
				return field.getStaticValue(frame);
			} else {
				final java.lang.reflect.Field f = type.getField(propname);
				if (Modifier.isStatic(f.getModifiers())) {
					return f.get(null);
				} else {
					throw new NoSuchFieldException(String.format("Field %s::%s not found", 
							EMFTVMUtil.toPrettyString(type, env), propname));
				}
			}
		} else {
			throw new IllegalArgumentException(String.format("%s is not a type", ort));
		}
	}

	/**
	 * Implements the FINDTYPE_S instruction.
	 * @param modelName the name of the model in which type can be found
	 * @param typeName the type name
	 * @param env the execution environment
	 * @return the type
	 * @throws ClassNotFoundException
	 */
	protected static Object findTypeS(final String modelName, final String typeName, final ExecEnv env)
	throws ClassNotFoundException {
		return env.findType(modelName, typeName);
	}

	/**
	 * Creates a new instance of <code>type</code>.
	 * @param type the type to instantiate
	 * @param modelname the name of the model in which to create a new instance
	 * @param env the execution environment
	 * @return the new object
	 */
	protected static Object newInstance(final Object type, final String modelname, final ExecEnv env) {
		if (type instanceof EClass) {
			final EClass eType = (EClass)type;
			Model model = env.getOutputModels().get(modelname);
			if (model == null) {
				model = env.getInoutModels().get(modelname);
			}
			if (model == null) {
				throw new IllegalArgumentException(String.format("Inout/output model %s not found", modelname));
			}
			return model.newElement(eType);
		} else {
			try {
				return NativeTypes.newInstance((Class<?>)type);
			} catch (Exception e) {
				throw new IllegalArgumentException(e);
			}
		}
	}

	/**
	 * Implements the DELETE instruction.
	 * @param element the element to delete
	 * @param frame the current stack frame
	 */
	protected static void delete(final EObject element, final StackFrame frame) {
		final ExecEnv env = frame.getEnv();

		final Resource res = element.eResource();
		if (res == null) {
			throw new IllegalArgumentException(String.format(
					"Element %s is cannot be deleted; not contained in a model", 
					EMFTVMUtil.toPrettyString(element, env)));
		}
		final Model model = env.getInputModelOf(element);
		if (model != null) {
			throw new IllegalArgumentException(String.format(
					"Element %s is cannot be deleted; contained in input model %s", 
					EMFTVMUtil.toPrettyString(element, env), env.getModelID(model)));
		}
		env.queueForDelete(element, frame);
	}

	/**
	 * Invokes native Java <code>method</code> on <code>self</code> without arguments.
	 * 
	 * @param frame
	 *            the current stack frame
	 * @param self
	 *            the object on which to invoke the method
	 * @param opname
	 *            the method name
	 * @param method
	 *            the method
	 * @return the method result
	 */
	protected static Object invokeNative(final StackFrame frame, final Object self, final String opname, final Method method) {
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method);
		}
		throw new UnsupportedOperationException(String.format("%s::%s()",
				EMFTVMUtil.getTypeName(frame.getEnv(), EMFTVMUtil.getArgumentType(self)),
				opname));
	}

	/**
	 * Invokes native Java <code>method</code> on <code>self</code> with argument <code>arg</code>.
	 * 
	 * @param frame
	 *            the current stack frame
	 * @param self
	 *            the object on which to invoke the method
	 * @param opname
	 *            the method name
	 * @param arg
	 *            the method argument
	 * @param method
	 *            the method
	 * @return the method result
	 */
	protected static Object invokeNative(final StackFrame frame, final Object self, final String opname, final Object arg,
			final Method method) {
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method, arg);
		}
		throw new UnsupportedOperationException(String.format("%s::%s(%s)",
				EMFTVMUtil.getTypeName(frame.getEnv(), EMFTVMUtil.getArgumentType(self)),
				opname, EMFTVMUtil.getTypeName(frame.getEnv(), EMFTVMUtil.getArgumentType(arg))));
	}

	/**
	 * Invokes native Java <code>method</code> on <code>self</code> with arguments <code>args</code>.
	 * 
	 * @param frame
	 *            the current stack frame
	 * @param self
	 *            the object on which to invoke the method
	 * @param opname
	 *            the method name
	 * @param args
	 *            the method arguments
	 * @return the method result
	 */
	protected static Object invokeNative(final StackFrame frame, final Object self, final String opname, final Object[] args,
			final Method method) {
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method, args);
		}
		throw new UnsupportedOperationException(String.format("%s::%s(%s)",
				EMFTVMUtil.getTypeName(frame.getEnv(), EMFTVMUtil.getArgumentType(self)), opname,
				EMFTVMUtil.getTypeNames(frame.getEnv(), EMFTVMUtil.getArgumentTypes(args))));
	}

	/**
	 * Implements the INVOKE_SUPER instruction
	 * 
	 * @param self
	 * @param context
	 *            the context type for "super"
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeSuper(final Object self, final EClass context, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		final java.util.Set<Operation> ops = new LinkedHashSet<Operation>();
		final List<EClass> superTypes = context.getESuperTypes();
		final ExecEnv env = frame.getEnv();
		Operation superOp = null;

		for (EClass superType : superTypes) {
			superOp = env.findOperation(superType, opname);
			if (superOp != null) {
				ops.add(superOp);
			}
		}
		if (ops.size() > 1) {
			throw new DuplicateEntryException(String.format(
					"More than one super-operation found for context %s: %s",
					context, ops));
		}
		if (!ops.isEmpty()) {
			superOp = ops.iterator().next();
		}

		final Method method = EMFTVMUtil.findNativeSuperMethod(superOp, context.getInstanceClass(), opname);
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method);
		}
		if (superOp != null) {
			final CodeBlock body = superOp.getBody();
			return body.execute(frame.getSubFrame(body, self));
		}

		throw new UnsupportedOperationException(String.format("super %s::%s()", 
				EMFTVMUtil.getTypeName(env, context), 
				opname));
	}

	/**
	 * Implements the INVOKE_SUPER instruction
	 * @param self
	 * @param context the context type for "super"
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeSuper(final Object self, final Class<?> context, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		final ExecEnv env = frame.getEnv();
		final Operation superOp =  env.findOperation(context.getSuperclass(), opname);

		final Method method = EMFTVMUtil.findNativeSuperMethod(superOp, context, opname);
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method);
		}
		if (superOp != null) {
			final CodeBlock body = superOp.getBody();
			return body.execute(frame.getSubFrame(body, self));
		}

		throw new UnsupportedOperationException(String.format("super %s::%s()", 
				EMFTVMUtil.getTypeName(env, context), 
				opname));
	}

	/**
	 * Implements the INVOKE_SUPER instruction
	 * @param self
	 * @param arg
	 * @param context the context type for "super"
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeSuper(final Object self, final Object arg, final EClass context, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		final java.util.Set<Operation> ops = new LinkedHashSet<Operation>();
		final List<?> superTypes = context.getESuperTypes();
		final ExecEnv env = frame.getEnv();
		Operation superOp = null;

		for (Object superType : superTypes) {
			superOp = env.findOperation(superType, opname, EMFTVMUtil.getArgumentType(arg));
			if (superOp != null) {
				ops.add(superOp);
			}
		}
		if (ops.size() > 1) {
			throw new DuplicateEntryException(String.format(
					"More than one super-operation found for context %s: %s",
					context, ops));
		}
		if (!ops.isEmpty()) {
			superOp = ops.iterator().next();
		}

		final Method method = EMFTVMUtil.findNativeSuperMethod(superOp, context.getInstanceClass(), opname, arg);
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method, arg);
		}
		if (superOp != null) {
			final CodeBlock body = superOp.getBody();
			return body.execute(frame.getSubFrame(body, self, arg));
		}

		throw new UnsupportedOperationException(String.format("super %s::%s(%s)", 
				EMFTVMUtil.getTypeName(env, context), 
				opname, 
				EMFTVMUtil.getTypeName(env, EMFTVMUtil.getArgumentType(arg))));
	}

	/**
	 * Implements the INVOKE_SUPER instruction
	 * @param self
	 * @param arg
	 * @param context the context type for "super"
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeSuper(final Object self, final Object arg, final Class<?> context, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		final ExecEnv env = frame.getEnv();
		final Operation superOp = env.findOperation(context.getSuperclass(), opname, EMFTVMUtil.getArgumentType(arg));

		final Method method = EMFTVMUtil.findNativeSuperMethod(superOp, context, opname, arg);
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method, arg);
		}
		if (superOp != null) {
			final CodeBlock body = superOp.getBody();
			return body.execute(frame.getSubFrame(body, self, arg));
		}

		throw new UnsupportedOperationException(String.format("super %s::%s(%s)", 
				EMFTVMUtil.getTypeName(env, context), 
				opname, 
				EMFTVMUtil.getTypeName(env, EMFTVMUtil.getArgumentType(arg))));
	}

	/**
	 * Implements the INVOKE_SUPER instruction
	 * @param self
	 * @param args
	 * @param context the context type for "super"
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeSuper(final Object self, final Object[] args, final EClass context, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		final java.util.Set<Operation> ops = new LinkedHashSet<Operation>();
		final List<?> superTypes = context.getESuperTypes();
		final ExecEnv env = frame.getEnv();
		Operation superOp = null;

		for (Object superType : superTypes) {
			superOp = env.findOperation(superType, opname, EMFTVMUtil.getArgumentTypes(args));
			if (superOp != null) {
				ops.add(superOp);
			}
		}
		if (ops.size() > 1) {
			throw new DuplicateEntryException(String.format(
					"More than one super-operation found for context %s: %s",
					context, ops));
		}
		if (!ops.isEmpty()) {
			superOp = ops.iterator().next();
		}

		final Method method = EMFTVMUtil.findNativeSuperMethod(superOp, context.getInstanceClass(), opname, args);
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method, args);
		}
		if (superOp != null) {
			final CodeBlock body = superOp.getBody();
			return body.execute(frame.getSubFrame(body, self, args));
		}

		throw new UnsupportedOperationException(String.format("super %s::%s(%s)", 
				EMFTVMUtil.getTypeName(env, context), 
				opname, 
				EMFTVMUtil.getTypeNames(env, EMFTVMUtil.getArgumentTypes(args))));
	}

	/**
	 * Implements the INVOKE_SUPER instruction
	 * @param self
	 * @param args
	 * @param context the context type for "super"
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeSuper(final Object self, final Object[] args, final Class<?> context, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		final ExecEnv env = frame.getEnv();
		final Operation superOp = env.findOperation(context.getSuperclass(), opname, EMFTVMUtil.getArgumentTypes(args));

		final Method method = EMFTVMUtil.findNativeSuperMethod(superOp, context, opname, args);
		if (method != null) {
			return EMFTVMUtil.invokeNative(frame, self, method, args);
		}
		if (superOp != null) {
			final CodeBlock body = superOp.getBody();
			return body.execute(frame.getSubFrame(body, self, args));
		}

		throw new UnsupportedOperationException(String.format("super %s::%s(%s)", 
				EMFTVMUtil.getTypeName(env, context), 
				opname, 
				EMFTVMUtil.getTypeNames(env, EMFTVMUtil.getArgumentTypes(args))));
	}

	/**
	 * Implements the INVOKE_STATIC instruction
	 * @param type
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeStatic(final Object type, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (type == null) {
			throw new IllegalArgumentException(String.format("Cannot invoke static operation %s on null type", opname));
		}

		final ExecEnv env = frame.getEnv();
		if (type == env.eClass()) { // Lazy and called rule invocations are indistinguishable from static operations in ATL
			final Rule rule = env.findRule(opname);
			if (rule != null && rule.getMode() == RuleMode.MANUAL) {
				return matchOne(rule, frame);
			}
		}

		final Operation op = frame.getEnv().findStaticOperation(
				type, 
				opname);
		if (op != null) {
			final CodeBlock body = op.getBody();
			return body.execute(new StackFrame(frame, body));
		}
		if (type instanceof Class<?>) {
			return EMFTVMUtil.invokeNativeStatic(frame, (Class<?>)type, opname);
		}
		throw new UnsupportedOperationException(String.format("static %s::%s()", 
				EMFTVMUtil.getTypeName(env, type), 
				opname));
	}

	/**
	 * Implements the INVOKE_STATIC instruction
	 * @param type
	 * @param arg
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeStatic(final Object type, final Object arg, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (type == null) {
			throw new IllegalArgumentException(String.format("Cannot invoke static operation %s on null type", opname));
		}

		final ExecEnv env = frame.getEnv();
		if (type == env.eClass()) { // Lazy and called rule invocations are indistinguishable from static operations in ATL
			final Rule rule = env.findRule(opname);
			if (rule != null && rule.getMode() == RuleMode.MANUAL) {
				return matchOne(new Object[] { arg }, rule, frame);
			}
		}

		final Operation op = frame.getEnv().findStaticOperation(type, opname, EMFTVMUtil.getArgumentType(arg));
		if (op != null) {
			final CodeBlock body = op.getBody();
			return body.execute(frame.getSubFrame(body, arg));
		}
		if (type instanceof Class<?>) {
			return EMFTVMUtil.invokeNativeStatic(frame, (Class<?>)type, opname, arg);
		}
		throw new UnsupportedOperationException(String.format("static %s::%s(%s)", 
				EMFTVMUtil.getTypeName(env, type), 
				opname, 
				EMFTVMUtil.getTypeName(env, EMFTVMUtil.getArgumentType(arg))));
	}

	/**
	 * Implements the INVOKE_STATIC instruction
	 * @param type
	 * @param args
	 * @param frame
	 * @param opname
	 * @return the invocation result
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected static Object invokeStatic(final Object type, final Object[] args, final StackFrame frame, final String opname) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (type == null) {
			throw new IllegalArgumentException(String.format("Cannot invoke static operation %s on null type", opname));
		}

		final ExecEnv env = frame.getEnv();
		if (type == env.eClass()) { // Lazy and called rule invocations are indistinguishable from static operations in ATL
			final Rule rule = env.findRule(opname);
			if (rule != null && rule.getMode() == RuleMode.MANUAL) {
				return matchOne(args, rule, frame);
			}
		}

		final Operation op = frame.getEnv().findStaticOperation(type, opname, EMFTVMUtil.getArgumentTypes(args));
		if (op != null) {
			final CodeBlock body = op.getBody();
			return body.execute(frame.getSubFrame(body, args));
		}
		if (type instanceof Class<?>) {
			return EMFTVMUtil.invokeNativeStatic(frame, (Class<?>)type, opname, args);
		}
		throw new UnsupportedOperationException(String.format("static %s::%s(%s)", 
				EMFTVMUtil.getTypeName(env, type), 
				opname, 
				EMFTVMUtil.getTypeNames(env, EMFTVMUtil.getArgumentTypes(args))));
	}

	/**
	 * Executes <code>rule</code> with <code>args</code>.
	 * @param args the rule arguments
	 * @param rule the rule
	 * @param frame the current stack frame
	 */
	protected static Object matchOne(final Object[] args, final Rule rule, final StackFrame frame) {
		final int argcount = args.length;
		if (argcount != rule.getInputElements().size()) {
			throw new VMException(frame, String.format(
					"Rule %s has different amount of input elements than expected: %d instead of %d",
					rule.getName(), rule.getInputElements().size(), argcount));
		}
		return rule.matchManual(frame, args);
	}

	/**
	 * Executes <code>rule</code> without arguments.
	 * @param rule the rule
	 * @param frame the current stack frame
	 */
	protected static Object matchOne(final Rule rule, final StackFrame frame) {
		if (rule.getInputElements().size() != 0) {
			throw new VMException(frame, String.format(
					"Rule %s has different amount of input elements than expected: %d instead of %d",
					rule.getName(), rule.getInputElements().size(), 0));
		}
		return rule.matchManual(frame, EMPTY);
	}

	/**
	 * Executes <code>rule</code> with <code>args</code>.
	 * @param args the rule arguments
	 * @param frame the current stack frame
	 * @param rulename the rule name
	 */
	protected static Object matchOne(final Object[] args, final StackFrame frame, final String rulename) {
		final ExecEnv env = frame.getEnv();
		final Rule rule = env.findRule(rulename);
		if (rule == null) {
			throw new IllegalArgumentException(String.format("Rule %s not found", rulename));
		}
		final int argcount = args.length;
		if (argcount != rule.getInputElements().size()) {
			throw new VMException(frame, String.format(
					"Rule %s has different amount of input elements than expected: %d instead of %d",
					rule.getName(), rule.getInputElements().size(), argcount));
		}
		return rule.matchManual(frame, args);
	}

	/**
	 * Executes <code>rule</code> without arguments.
	 * @param frame the current stack frame
	 * @param rulename the rule name
	 */
	protected static Object matchOne(final StackFrame frame, final String rulename) {
		final ExecEnv env = frame.getEnv();
		final Rule rule = env.findRule(rulename);
		if (rule == null) {
			throw new IllegalArgumentException(String.format("Rule %s not found", rulename));
		}
		if (rule.getInputElements().size() != 0) {
			throw new VMException(frame, String.format(
					"Rule %s has different amount of input elements than expected: %d instead of %d",
					rule.getName(), rule.getInputElements().size(), 0));
		}
		return rule.matchManual(frame, EMPTY);
	}

	/**
	 * Adds <code>v</code> to <code>o.propname</code>. Implements the ADD and INSERT instructions.
	 * 
	 * @param o
	 *            object
	 * @param v
	 *            value
	 * @param index
	 *            the insertion index (-1 for end)
	 * @param propname
	 *            the property name
	 * @param cb
	 *            the original non-JIT'ed code block
	 * @param frame
	 *            the current stack frame
	 * @throws NoSuchFieldException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	protected static void add(final Object o, final Object v, final int index, final String propname, final CodeBlock cb,
			final StackFrame frame)
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final Rule rule = cb.getRule();

		if (o instanceof EObject) {
			final EObject eo = (EObject)o;
			final EClass type = eo.eClass();
			final boolean queueSet = env.getCurrentPhase() == RuleMode.AUTOMATIC_SINGLE && env.getInoutModelOf(eo) != null;
			Field field = null;
			if (rule != null) {
				field = rule.findField(type, propname);
			}
			if (field == null) {
				field = env.findField(type, propname);
				if (field != null) {
					if (queueSet) {
						env.queueForAdd(field, o, v, index, frame);
					} else {
						if (env.getInputModelOf(eo) != null) {
							throw new IllegalArgumentException(String.format(
									"Cannot add to properties of %s, as it is contained in an input model",
									EMFTVMUtil.toPrettyString(eo, env)));
						}
						if (env.getOutputModelOf(eo) != null) {
							throw new IllegalArgumentException(String.format(
									"Adding to transient field %s of %s, which cannot be read back as %1s is contained in an output model",
									propname, EMFTVMUtil.toPrettyString(eo, env)));
						}
						field.addValue(o, v, index, frame);
					}
					return;
				}
			} else {
				// Treat rule fields as local variables
				field.addValue(o, v, index, frame);
				return;
			}
			final EStructuralFeature sf = type.getEStructuralFeature(propname);
			if (sf != null) {
				if (queueSet) {
					env.queueForAdd(sf, eo, v, index, frame);
				} else {
					EMFTVMUtil.add(env, eo, sf, v, index);
				}
				return;
			}
			final Resource resource = eo.eResource();
			if (EMFTVMUtil.XMI_ID_FEATURE.equals(propname) && resource instanceof XMIResource) { //$NON-NLS-1$
				if (queueSet) {
					env.queueXmiIDForAdd(eo, v, index, frame);
				} else {
					if (((XMIResource) resource).getID(eo) != null) {
						throw new IllegalArgumentException(String.format(
								"Cannot add %s to field %s::%s: maximum multiplicity of 1 reached", v, EMFTVMUtil.toPrettyString(eo, env),
								propname));
					}
					if (index > 0) {
						throw new IndexOutOfBoundsException(String.valueOf(index));
					}
					((XMIResource) resource).setID(eo, v.toString());
				}
				return;
			}
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}

		// o is a regular Java object
		final Class<?> type = o == null ? Void.TYPE : o.getClass();
		Field field = null;
		if (rule != null) {
			field = rule.findField(type, propname);
		}
		if (field == null) {
			field = env.findField(type, propname);
		}
		if (field != null) {
			field.addValue(o, v, index, frame);
			return;
		}
		throw new NoSuchFieldException(String.format("Field %s::%s not found", EMFTVMUtil.toPrettyString(type, env), propname));
	}

	/**
	 * Implements the REMOVE instruction.
	 * 
	 * @param o
	 *            object
	 * @param v
	 *            value
	 * @param propname
	 *            the property name
	 * @param cb
	 *            the original non-JIT'ed code block
	 * @param frame
	 *            the current stack frame
	 * @throws NoSuchFieldException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	protected static void remove(final Object o, final Object v, final String propname, final CodeBlock cb, final StackFrame frame)
	throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
		final ExecEnv env = frame.getEnv();
		final Rule rule = cb.getRule();

		if (o instanceof EObject) {
			final EObject eo = (EObject)o;
			final EClass type = eo.eClass();
			final boolean queueSet = env.getCurrentPhase() == RuleMode.AUTOMATIC_SINGLE && env.getInoutModelOf(eo) != null;
			Field field = null;
			if (rule != null) {
				field = rule.findField(type, propname);
			}
			if (field == null) {
				field = env.findField(type, propname);
				if (field != null) {
					if (queueSet) {
						env.queueForRemove(field, o, v, frame);
					} else {
						if (env.getInputModelOf(eo) != null) {
							throw new IllegalArgumentException(String.format(
									"Cannot remove from properties of %s, as it is contained in an input model",
									EMFTVMUtil.toPrettyString(eo, env)));
						}
						if (env.getOutputModelOf(eo) != null) {
							throw new IllegalArgumentException(
									String.format(
											"Removing from transient field %s of %s, which cannot be read back as %1s is contained in an output model",
											propname, EMFTVMUtil.toPrettyString(eo, env)));
						}
						field.removeValue(o, v, frame);
					}
					return;
				}
			} else {
				// Treat rule fields as local variables
				field.removeValue(o, v, frame);
				return;
			}
			final EStructuralFeature sf = type.getEStructuralFeature(propname);
			if (sf != null) {
				if (queueSet) {
					env.queueForRemove(sf, eo, v, frame);
				} else {
					EMFTVMUtil.remove(env, eo, sf, v);
				}
				return;
			}
			final Resource resource = eo.eResource();
			if (EMFTVMUtil.XMI_ID_FEATURE.equals(propname) && resource instanceof XMIResource) { //$NON-NLS-1$
				if (queueSet) {
					env.queueXmiIDForRemove(eo, v, frame);
				} else {
					final XMIResource xmiRes = (XMIResource) resource;
					final Object xmiID = xmiRes.getID(eo);
					if (xmiID == null ? v == null : xmiID.equals(v)) {
						xmiRes.setID(eo, null);
					}
				}
				return;
			}
			throw new NoSuchFieldException(String.format("Field %s::%s not found", 
					EMFTVMUtil.toPrettyString(type, env), propname));
		}

		// o is a regular Java object
		final Class<?> type = o == null ? Void.TYPE : o.getClass();
		Field field = null;
		if (rule != null) {
			field = rule.findField(type, propname);
		}
		if (field == null) {
			field = env.findField(type, propname);
		}
		if (field != null) {
			field.removeValue(o, v, frame);
			return;
		}
		throw new NoSuchFieldException(String.format("Field %s::%s not found", EMFTVMUtil.toPrettyString(type, env), propname));
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Boolean> asList(boolean[] array) {
		final List<Boolean> list = new ArrayList<Boolean>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Character> asList(char[] array) {
		final List<Character> list = new ArrayList<Character>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Byte> asList(byte[] array) {
		final List<Byte> list = new ArrayList<Byte>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Short> asList(short[] array) {
		final List<Short> list = new ArrayList<Short>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Integer> asList(int[] array) {
		final List<Integer> list = new ArrayList<Integer>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Long> asList(long[] array) {
		final List<Long> list = new ArrayList<Long>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Float> asList(float[] array) {
		final List<Float> list = new ArrayList<Float>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

	/**
	 * Converts <code>array</code> to a {@link List}.
	 * @param array the array to convert
	 * @return the {@link List} containing the <code>array</code> values
	 */
	protected static List<Double> asList(double[] array) {
		final List<Double> list = new ArrayList<Double>(array.length);
		for (int i = 0; i < array.length; i++) {
			list.add(array[i]);
		}
		return list;
	}

}
