
package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link NetworkSpec}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNetworkSpec.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNetworkSpec implements NetworkSpec {
  private final String name;
  private final @Nullable Map<String, String> labels;
  private final Driver driverConfiguration;
  private final @Nullable Boolean ipv6Enabled;
  private final @Nullable Boolean internal;
  private final @Nullable Boolean attachable;
  private final @Nullable IpamOptions ipamOptions;

  private ImmutableNetworkSpec(
      String name,
      @Nullable Map<String, String> labels,
      Driver driverConfiguration,
      @Nullable Boolean ipv6Enabled,
      @Nullable Boolean internal,
      @Nullable Boolean attachable,
      @Nullable IpamOptions ipamOptions) {
    this.name = name;
    this.labels = labels;
    this.driverConfiguration = driverConfiguration;
    this.ipv6Enabled = ipv6Enabled;
    this.internal = internal;
    this.attachable = attachable;
    this.ipamOptions = ipamOptions;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code driverConfiguration} attribute
   */
  @JsonProperty("DriverConfiguration")
  @Override
  public Driver driverConfiguration() {
    return driverConfiguration;
  }

  /**
   * @return The value of the {@code ipv6Enabled} attribute
   */
  @JsonProperty("IPv6Enabled")
  @Override
  public @Nullable Boolean ipv6Enabled() {
    return ipv6Enabled;
  }

  /**
   * @return The value of the {@code internal} attribute
   */
  @JsonProperty("Internal")
  @Override
  public @Nullable Boolean internal() {
    return internal;
  }

  /**
   * @return The value of the {@code attachable} attribute
   */
  @JsonProperty("Attachable")
  @Override
  public @Nullable Boolean attachable() {
    return attachable;
  }

  /**
   * @return The value of the {@code ipamOptions} attribute
   */
  @JsonProperty("IPAMOptions")
  @Override
  public @Nullable IpamOptions ipamOptions() {
    return ipamOptions;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkSpec#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkSpec withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableNetworkSpec(
        newValue,
        this.labels,
        this.driverConfiguration,
        this.ipv6Enabled,
        this.internal,
        this.attachable,
        this.ipamOptions);
  }

  /**
   * Copy the current immutable object by replacing the {@link NetworkSpec#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the labels map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableNetworkSpec withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableNetworkSpec(
        this.name,
        newValue,
        this.driverConfiguration,
        this.ipv6Enabled,
        this.internal,
        this.attachable,
        this.ipamOptions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkSpec#driverConfiguration() driverConfiguration} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for driverConfiguration
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkSpec withDriverConfiguration(Driver value) {
    if (this.driverConfiguration == value) return this;
    Driver newValue = Objects.requireNonNull(value, "driverConfiguration");
    return new ImmutableNetworkSpec(
        this.name,
        this.labels,
        newValue,
        this.ipv6Enabled,
        this.internal,
        this.attachable,
        this.ipamOptions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkSpec#ipv6Enabled() ipv6Enabled} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for ipv6Enabled (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkSpec withIpv6Enabled(@Nullable Boolean value) {
    if (Objects.equals(this.ipv6Enabled, value)) return this;
    return new ImmutableNetworkSpec(
        this.name,
        this.labels,
        this.driverConfiguration,
        value,
        this.internal,
        this.attachable,
        this.ipamOptions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkSpec#internal() internal} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for internal (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkSpec withInternal(@Nullable Boolean value) {
    if (Objects.equals(this.internal, value)) return this;
    return new ImmutableNetworkSpec(
        this.name,
        this.labels,
        this.driverConfiguration,
        this.ipv6Enabled,
        value,
        this.attachable,
        this.ipamOptions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkSpec#attachable() attachable} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for attachable (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkSpec withAttachable(@Nullable Boolean value) {
    if (Objects.equals(this.attachable, value)) return this;
    return new ImmutableNetworkSpec(
        this.name,
        this.labels,
        this.driverConfiguration,
        this.ipv6Enabled,
        this.internal,
        value,
        this.ipamOptions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkSpec#ipamOptions() ipamOptions} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for ipamOptions (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkSpec withIpamOptions(@Nullable IpamOptions value) {
    if (this.ipamOptions == value) return this;
    return new ImmutableNetworkSpec(
        this.name,
        this.labels,
        this.driverConfiguration,
        this.ipv6Enabled,
        this.internal,
        this.attachable,
        value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNetworkSpec} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNetworkSpec
        && equalsByValue((ImmutableNetworkSpec) another);
  }

  private boolean equalsByValue(ImmutableNetworkSpec another) {
    return name.equals(another.name)
        && Objects.equals(labels, another.labels)
        && driverConfiguration.equals(another.driverConfiguration)
        && Objects.equals(ipv6Enabled, another.ipv6Enabled)
        && Objects.equals(internal, another.internal)
        && Objects.equals(attachable, another.attachable)
        && Objects.equals(ipamOptions, another.ipamOptions);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code labels}, {@code driverConfiguration}, {@code ipv6Enabled}, {@code internal}, {@code attachable}, {@code ipamOptions}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + name.hashCode();
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + driverConfiguration.hashCode();
    h += (h << 5) + Objects.hashCode(ipv6Enabled);
    h += (h << 5) + Objects.hashCode(internal);
    h += (h << 5) + Objects.hashCode(attachable);
    h += (h << 5) + Objects.hashCode(ipamOptions);
    return h;
  }

  /**
   * Prints the immutable value {@code NetworkSpec} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NetworkSpec{"
        + "name=" + name
        + ", labels=" + labels
        + ", driverConfiguration=" + driverConfiguration
        + ", ipv6Enabled=" + ipv6Enabled
        + ", internal=" + internal
        + ", attachable=" + attachable
        + ", ipamOptions=" + ipamOptions
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NetworkSpec} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NetworkSpec instance
   */
  public static ImmutableNetworkSpec copyOf(NetworkSpec instance) {
    if (instance instanceof ImmutableNetworkSpec) {
      return (ImmutableNetworkSpec) instance;
    }
    return ImmutableNetworkSpec.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNetworkSpec ImmutableNetworkSpec}.
   * <pre>
   * ImmutableNetworkSpec.builder()
   *    .name(String) // required {@link NetworkSpec#name() name}
   *    .labels(Map&lt;String, String&gt; | null) // nullable {@link NetworkSpec#labels() labels}
   *    .driverConfiguration(org.mandas.docker.client.messages.swarm.Driver) // required {@link NetworkSpec#driverConfiguration() driverConfiguration}
   *    .ipv6Enabled(Boolean | null) // nullable {@link NetworkSpec#ipv6Enabled() ipv6Enabled}
   *    .internal(Boolean | null) // nullable {@link NetworkSpec#internal() internal}
   *    .attachable(Boolean | null) // nullable {@link NetworkSpec#attachable() attachable}
   *    .ipamOptions(org.mandas.docker.client.messages.swarm.IpamOptions | null) // nullable {@link NetworkSpec#ipamOptions() ipamOptions}
   *    .build();
   * </pre>
   * @return A new ImmutableNetworkSpec builder
   */
  public static ImmutableNetworkSpec.Builder builder() {
    return new ImmutableNetworkSpec.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNetworkSpec ImmutableNetworkSpec}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_NAME = 0x1L;
    private static final long INIT_BIT_DRIVER_CONFIGURATION = 0x2L;
    private long initBits = 0x3L;

    private String name;
    private Map<String, String> labels = null;
    private Driver driverConfiguration;
    private Boolean ipv6Enabled;
    private Boolean internal;
    private Boolean attachable;
    private IpamOptions ipamOptions;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NetworkSpec} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NetworkSpec instance) {
      Objects.requireNonNull(instance, "instance");
      this.name(instance.name());
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      this.driverConfiguration(instance.driverConfiguration());
      @Nullable Boolean ipv6EnabledValue = instance.ipv6Enabled();
      if (ipv6EnabledValue != null) {
        ipv6Enabled(ipv6EnabledValue);
      }
      @Nullable Boolean internalValue = instance.internal();
      if (internalValue != null) {
        internal(internalValue);
      }
      @Nullable Boolean attachableValue = instance.attachable();
      if (attachableValue != null) {
        attachable(attachableValue);
      }
      @Nullable IpamOptions ipamOptionsValue = instance.ipamOptions();
      if (ipamOptionsValue != null) {
        ipamOptions(ipamOptionsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkSpec#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Put one entry to the {@link NetworkSpec#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link NetworkSpec#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link NetworkSpec#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link NetworkSpec#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkSpec#driverConfiguration() driverConfiguration} attribute.
     * @param driverConfiguration The value for driverConfiguration 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("DriverConfiguration")
    public final Builder driverConfiguration(Driver driverConfiguration) {
      this.driverConfiguration = Objects.requireNonNull(driverConfiguration, "driverConfiguration");
      initBits &= ~INIT_BIT_DRIVER_CONFIGURATION;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkSpec#ipv6Enabled() ipv6Enabled} attribute.
     * @param ipv6Enabled The value for ipv6Enabled (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("IPv6Enabled")
    public final Builder ipv6Enabled(@Nullable Boolean ipv6Enabled) {
      this.ipv6Enabled = ipv6Enabled;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkSpec#internal() internal} attribute.
     * @param internal The value for internal (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Internal")
    public final Builder internal(@Nullable Boolean internal) {
      this.internal = internal;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkSpec#attachable() attachable} attribute.
     * @param attachable The value for attachable (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Attachable")
    public final Builder attachable(@Nullable Boolean attachable) {
      this.attachable = attachable;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkSpec#ipamOptions() ipamOptions} attribute.
     * @param ipamOptions The value for ipamOptions (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("IPAMOptions")
    public final Builder ipamOptions(@Nullable IpamOptions ipamOptions) {
      this.ipamOptions = ipamOptions;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNetworkSpec ImmutableNetworkSpec}.
     * @return An immutable instance of NetworkSpec
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNetworkSpec build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNetworkSpec(
          name,
          labels == null ? null : createUnmodifiableMap(false, false, labels),
          driverConfiguration,
          ipv6Enabled,
          internal,
          attachable,
          ipamOptions);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      if ((initBits & INIT_BIT_DRIVER_CONFIGURATION) != 0) attributes.add("driverConfiguration");
      return "Cannot build NetworkSpec, some of required attributes are not set " + attributes;
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
