/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.container.orchestration.provider.impl;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.async.ResultCallback;
import com.github.dockerjava.api.command.CreateContainerCmd;
import com.github.dockerjava.api.command.PullImageCmd;
import com.github.dockerjava.api.command.PullImageResultCallback;
import com.github.dockerjava.api.model.AuthConfig;
import com.github.dockerjava.api.model.Bind;
import com.github.dockerjava.api.model.Container;
import com.github.dockerjava.api.model.ContainerPort;
import com.github.dockerjava.api.model.Device;
import com.github.dockerjava.api.model.ExposedPort;
import com.github.dockerjava.api.model.HostConfig;
import com.github.dockerjava.api.model.Image;
import com.github.dockerjava.api.model.LogConfig;
import com.github.dockerjava.api.model.Ports;
import com.github.dockerjava.api.model.PullResponseItem;
import com.github.dockerjava.api.model.Volume;
import com.github.dockerjava.core.DefaultDockerClientConfig;
import com.github.dockerjava.core.DockerClientConfig;
import com.github.dockerjava.core.DockerClientImpl;
import com.github.dockerjava.httpclient5.ApacheDockerHttpClient;
import com.github.dockerjava.transport.DockerHttpClient;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.eclipse.kura.KuraErrorCode;
import org.eclipse.kura.KuraException;
import org.eclipse.kura.configuration.ConfigurableComponent;
import org.eclipse.kura.container.orchestration.ContainerConfiguration;
import org.eclipse.kura.container.orchestration.ContainerInstanceDescriptor;
import org.eclipse.kura.container.orchestration.ContainerOrchestrationService;
import org.eclipse.kura.container.orchestration.ContainerState;
import org.eclipse.kura.container.orchestration.PasswordRegistryCredentials;
import org.eclipse.kura.container.orchestration.RegistryCredentials;
import org.eclipse.kura.container.orchestration.listener.ContainerOrchestrationServiceListener;
import org.eclipse.kura.container.orchestration.provider.impl.ContainerOrchestrationServiceOptions;
import org.eclipse.kura.crypto.CryptoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerOrchestrationServiceImpl
implements ConfigurableComponent,
ContainerOrchestrationService {
    private static final String PARAMETER_CANNOT_BE_NULL = "The provided parameter cannot be null";
    private static final String UNABLE_TO_CONNECT_TO_DOCKER_CLI = "Unable to connect to docker cli";
    private static final Logger logger = LoggerFactory.getLogger(ContainerOrchestrationServiceImpl.class);
    private static final String APP_ID = "org.eclipse.kura.container.orchestration.provider.ConfigurableDocker";
    private ContainerOrchestrationServiceOptions currentConfig;
    private final Set<ContainerOrchestrationServiceListener> dockerServiceListeners = new HashSet<ContainerOrchestrationServiceListener>();
    private final Set<FrameworkManagedContainer> frameworkManagedContainers = new HashSet<FrameworkManagedContainer>();
    private DockerClient dockerClient;
    private CryptoService cryptoService;

    public void setDockerClient(DockerClient dockerClient) {
        this.dockerClient = dockerClient;
    }

    public void setCryptoService(CryptoService cryptoService) {
        this.cryptoService = cryptoService;
    }

    public void activate(Map<String, Object> properties) {
        logger.info("Bundle {} is starting with config!", (Object)APP_ID);
        if (!Objects.isNull(properties)) {
            this.updated(properties);
        }
        logger.info("Bundle {} has started with config!", (Object)APP_ID);
    }

    public void deactivate() {
        logger.info("Bundle {} is stopping!", (Object)APP_ID);
        if (this.testConnection()) {
            this.disconnect();
        }
        logger.info("Bundle {} has stopped!", (Object)APP_ID);
    }

    public void updated(Map<String, Object> properties) {
        logger.info("Bundle {} is updating with config!", (Object)APP_ID);
        ContainerOrchestrationServiceOptions newProps = new ContainerOrchestrationServiceOptions(properties);
        if (!newProps.equals(this.currentConfig)) {
            this.currentConfig = newProps;
            logger.info("Connecting to docker ");
            if (!this.currentConfig.isEnabled()) {
                this.cleanUpDocker();
                return;
            }
            this.connect();
            if (!this.testConnection()) {
                logger.error("Could not connect to docker CLI.");
                return;
            }
            logger.info("Connection Successful");
        }
        logger.info("Bundle {} has updated with config!", (Object)APP_ID);
    }

    public List<String> listContainersIds() {
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
        List containers = (List)this.dockerClient.listContainersCmd().withShowAll(Boolean.valueOf(true)).exec();
        ArrayList<String> result = new ArrayList<String>();
        for (Container cont : containers) {
            result.add(cont.getId());
        }
        return result;
    }

    public List<ContainerInstanceDescriptor> listContainerDescriptors() {
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
        List containers = (List)this.dockerClient.listContainersCmd().withShowAll(Boolean.valueOf(true)).exec();
        ArrayList<ContainerInstanceDescriptor> result = new ArrayList<ContainerInstanceDescriptor>();
        containers.forEach(container -> {
            boolean bl = result.add(ContainerInstanceDescriptor.builder().setContainerName(this.getContainerName((Container)container)).setContainerImage(this.getContainerTag((Container)container)).setContainerImageTag(this.getContainerVersion((Container)container)).setContainerID(container.getId()).setInternalPorts(this.parseInternalPortsFromDockerPs(container.getPorts())).setExternalPorts(this.parseExternalPortsFromDockerPs(container.getPorts())).setContainerState(this.convertDockerStateToFrameworkState(container.getState())).setFrameworkManaged(this.isFrameworkManaged((Container)container)).build());
        });
        return result;
    }

    private Boolean isFrameworkManaged(Container container) {
        String containerName = this.getContainerName(container);
        return this.frameworkManagedContainers.stream().anyMatch(c -> ((FrameworkManagedContainer)c).name.equals(containerName));
    }

    private String getContainerName(Container container) {
        return container.getNames()[0].replace("/", "");
    }

    private String getContainerVersion(Container container) {
        String version = "";
        if (container.getImage().split(":").length > 1) {
            version = container.getImage().split(":")[1];
        }
        return version;
    }

    private String getContainerTag(Container container) {
        return container.getImage().split(":")[0];
    }

    private List<Integer> parseExternalPortsFromDockerPs(ContainerPort[] ports) {
        ContainerPort[] tempPorts;
        ArrayList<Integer> externalPorts = new ArrayList<Integer>();
        ContainerPort[] containerPortArray = tempPorts = ports;
        int n = tempPorts.length;
        int n2 = 0;
        while (n2 < n) {
            String ipFormatTest;
            ContainerPort tempPort = containerPortArray[n2];
            if (tempPort.getIp() != null && (ipFormatTest = tempPort.getIp()) != null && (ipFormatTest.equals("::") || ipFormatTest.equals("0.0.0.0"))) {
                externalPorts.add(tempPort.getPublicPort());
            }
            ++n2;
        }
        return externalPorts;
    }

    private List<Integer> parseInternalPortsFromDockerPs(ContainerPort[] ports) {
        ContainerPort[] tempPorts;
        ArrayList<Integer> internalPorts = new ArrayList<Integer>();
        ContainerPort[] containerPortArray = tempPorts = ports;
        int n = tempPorts.length;
        int n2 = 0;
        while (n2 < n) {
            String ipFormatTest;
            ContainerPort tempPort = containerPortArray[n2];
            if (tempPort.getIp() != null && (ipFormatTest = tempPort.getIp()) != null && (ipFormatTest.equals("::") || ipFormatTest.equals("0.0.0.0"))) {
                internalPorts.add(tempPort.getPrivatePort());
            }
            ++n2;
        }
        return internalPorts;
    }

    private ContainerState convertDockerStateToFrameworkState(String dockerState) {
        switch (dockerState.trim()) {
            case "created": {
                return ContainerState.INSTALLED;
            }
            case "restarting": {
                return ContainerState.INSTALLED;
            }
            case "running": {
                return ContainerState.ACTIVE;
            }
            case "paused": {
                return ContainerState.STOPPING;
            }
            case "exited": {
                return ContainerState.STOPPING;
            }
            case "dead": {
                return ContainerState.FAILED;
            }
        }
        return ContainerState.INSTALLED;
    }

    public Optional<String> getContainerIdByName(String name) {
        this.checkRequestEnv(name);
        List containers = (List)this.dockerClient.listContainersCmd().withShowAll(Boolean.valueOf(true)).exec();
        for (Container cont : containers) {
            String[] containerNames;
            String[] stringArray = containerNames = cont.getNames();
            int n = containerNames.length;
            int n2 = 0;
            while (n2 < n) {
                String containerName = stringArray[n2];
                if (containerName.equals("/" + name)) {
                    return Optional.of(cont.getId());
                }
                ++n2;
            }
        }
        return Optional.empty();
    }

    public void startContainer(String id) throws KuraException {
        this.checkRequestEnv(id);
        try {
            this.dockerClient.startContainerCmd(id).exec();
        }
        catch (Exception exception) {
            logger.error("Could not start container {}. It could be already running or not exist at all", (Object)id);
            throw new KuraException(KuraErrorCode.OS_COMMAND_ERROR);
        }
    }

    public String startContainer(ContainerConfiguration container) throws KuraException, InterruptedException {
        String containerId;
        this.checkRequestEnv(container);
        logger.info("Starting {} Microservice", (Object)container.getContainerName());
        Optional<ContainerInstanceDescriptor> existingInstance = this.listContainerDescriptors().stream().filter(c -> c.getContainerName().equals(container.getContainerName())).findAny();
        if (existingInstance.isPresent() && existingInstance.get().getContainerState() == ContainerState.ACTIVE) {
            logger.info("Found already existing running container");
            containerId = existingInstance.get().getContainerId();
        } else if (!existingInstance.isPresent()) {
            logger.info("Creating new container instance");
            this.pullImage(container.getContainerImage(), container.getContainerImageTag(), container.getImageDownloadTimeoutSeconds(), container.getRegistryCredentials());
            containerId = this.createContainer(container);
            this.startContainer(containerId);
        } else {
            logger.info("Found already exisiting not running container, starting it..");
            containerId = existingInstance.get().getContainerId();
            this.startContainer(containerId);
        }
        logger.info("Container Started Successfully");
        if (container.isFrameworkManaged()) {
            this.frameworkManagedContainers.add(new FrameworkManagedContainer(container.getContainerName(), containerId));
        }
        return containerId;
    }

    public void stopContainer(String id) throws KuraException {
        this.checkRequestEnv(id);
        try {
            this.dockerClient.stopContainerCmd(id).exec();
        }
        catch (Exception exception) {
            logger.error("Could not stop container {}.", (Object)id);
            throw new KuraException(KuraErrorCode.OS_COMMAND_ERROR);
        }
    }

    public void deleteContainer(String id) throws KuraException {
        this.checkRequestEnv(id);
        try {
            this.dockerClient.removeContainerCmd(id).exec();
            this.frameworkManagedContainers.removeIf(c -> id.equals(((FrameworkManagedContainer)c).id));
        }
        catch (Exception exception) {
            logger.error("Could not remove container {}.", (Object)id);
            throw new KuraException(KuraErrorCode.OS_COMMAND_ERROR);
        }
    }

    private void checkRequestEnv(Object parameter) {
        if (Objects.isNull(parameter)) {
            throw new IllegalArgumentException(PARAMETER_CANNOT_BE_NULL);
        }
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
    }

    public void pullImage(String imageName, String imageTag, int timeOutSeconds, Optional<RegistryCredentials> registryCredentials) throws KuraException, InterruptedException {
        if (Objects.isNull(imageName) || Objects.isNull(imageTag) || timeOutSeconds < 0 || Objects.isNull(registryCredentials)) {
            throw new IllegalArgumentException("Parameters cannot be null or negative");
        }
        boolean imageAvailableLocally = this.doesImageExist(imageName, imageTag);
        if (!imageAvailableLocally) {
            try {
                this.imagePullHelper(imageName, imageTag, timeOutSeconds, registryCredentials);
            }
            catch (InterruptedException e) {
                throw e;
            }
            catch (Exception exception) {
                throw new KuraException(KuraErrorCode.IO_ERROR, new Object[]{"Unable to pull container"});
            }
        }
    }

    public void registerListener(ContainerOrchestrationServiceListener dockerListener) {
        this.dockerServiceListeners.add(dockerListener);
    }

    public void unregisterListener(ContainerOrchestrationServiceListener dockerListener) {
        this.dockerServiceListeners.remove(dockerListener);
    }

    private void imagePullHelper(final String imageName, final String imageTag, int timeOutSeconds, Optional<RegistryCredentials> repositoryCredentials) throws InterruptedException, KuraException {
        logger.info("Attempting to pull image: {}.", (Object)imageName);
        PullImageCmd pullRequest = this.dockerClient.pullImageCmd(imageName).withTag(imageTag);
        if (repositoryCredentials.isPresent()) {
            this.doAuthenticate(repositoryCredentials.get(), pullRequest);
        }
        (pullRequest.exec((ResultCallback)new PullImageResultCallback(){

            public void onNext(PullResponseItem item) {
                super.onNext(item);
                ContainerOrchestrationServiceImpl.this.createLoggerMessageForContainerPull(item, imageName, imageTag);
            }
        })).awaitCompletion(timeOutSeconds, TimeUnit.SECONDS);
    }

    private void doAuthenticate(RegistryCredentials repositoryCredentials, PullImageCmd pullRequest) throws KuraException {
        if (!(repositoryCredentials instanceof PasswordRegistryCredentials)) {
            throw new KuraException(KuraErrorCode.BAD_REQUEST);
        }
        PasswordRegistryCredentials repositoryPasswordCredentials = (PasswordRegistryCredentials)repositoryCredentials;
        AuthConfig authConfig = new AuthConfig().withUsername(repositoryPasswordCredentials.getUsername()).withPassword(new String(this.cryptoService.decryptAes(repositoryPasswordCredentials.getPassword().getPassword())));
        Optional url = repositoryPasswordCredentials.getUrl();
        if (url.isPresent()) {
            logger.info("Attempting to sign into repo: {}", url.get());
            authConfig = authConfig.withRegistryAddress((String)url.get());
        }
        pullRequest.withAuthConfig(authConfig);
    }

    private void createLoggerMessageForContainerPull(PullResponseItem item, String imageName, String imageTag) {
        if (logger.isDebugEnabled()) {
            logger.debug("Pulling {}:{} Layer {}, State: {}", new Object[]{imageName, imageTag, item.getId(), item.getStatus()});
        }
        if (item.isErrorIndicated()) {
            logger.error("Unable To Pull image {}:{} because : {}", new Object[]{item.getErrorDetail(), imageName, imageTag});
        }
        if (item.isPullSuccessIndicated()) {
            logger.info("Image pull of {}:{}, Layer: {}, was succsessful", new Object[]{imageName, imageTag, item.getId()});
        }
    }

    private String createContainer(ContainerConfiguration containerDescription) throws KuraException {
        if (!this.testConnection()) {
            throw new IllegalStateException("failed to reach docker engine");
        }
        if (containerDescription == null || containerDescription.getContainerImage() == null || containerDescription.getContainerImageTag() == null) {
            throw new IllegalStateException("failed to create container, null containerImage passed");
        }
        String containerImageFullString = String.format("%s:%s", containerDescription.getContainerImage(), containerDescription.getContainerImageTag());
        CreateContainerCmd commandBuilder = null;
        try {
            commandBuilder = this.dockerClient.createContainerCmd(containerImageFullString);
            if (containerDescription.getContainerName() != null) {
                commandBuilder = commandBuilder.withName(containerDescription.getContainerName());
            }
            HostConfig configuration = new HostConfig();
            commandBuilder = this.containerEnviromentVariablesHandler(containerDescription, commandBuilder);
            configuration = this.containerVolumeMangamentHandler(containerDescription, configuration);
            configuration = this.containerDevicesHandler(containerDescription, configuration);
            configuration = this.containerPortManagementHandler(containerDescription, configuration);
            configuration = this.containerLogConfigurationHandler(containerDescription, configuration);
            if (containerDescription.isContainerPrivileged()) {
                configuration = configuration.withPrivileged(Boolean.valueOf(containerDescription.isContainerPrivileged()));
            }
            String string = commandBuilder.withHostConfig(configuration).exec().getId();
            return string;
        }
        catch (Exception e) {
            logger.error("failed to create container", (Throwable)e);
            throw new KuraException(KuraErrorCode.PROCESS_EXECUTION_ERROR);
        }
        finally {
            if (!Objects.isNull(commandBuilder)) {
                commandBuilder.close();
            }
        }
    }

    private HostConfig containerLogConfigurationHandler(ContainerConfiguration containerDescription, HostConfig configuration) {
        LogConfig.LoggingType lt;
        switch (containerDescription.getContainerLoggingType().toUpperCase().trim()) {
            case "NONE": {
                lt = LogConfig.LoggingType.NONE;
                break;
            }
            case "LOCAL": {
                lt = LogConfig.LoggingType.LOCAL;
                break;
            }
            case "ETWLOGS": {
                lt = LogConfig.LoggingType.ETWLOGS;
                break;
            }
            case "JSON_FILE": {
                lt = LogConfig.LoggingType.JSON_FILE;
                break;
            }
            case "SYSLOG": {
                lt = LogConfig.LoggingType.SYSLOG;
                break;
            }
            case "JOURNALD": {
                lt = LogConfig.LoggingType.JOURNALD;
                break;
            }
            case "GELF": {
                lt = LogConfig.LoggingType.GELF;
                break;
            }
            case "FLUENTD": {
                lt = LogConfig.LoggingType.FLUENTD;
                break;
            }
            case "AWSLOGS": {
                lt = LogConfig.LoggingType.AWSLOGS;
                break;
            }
            case "DB": {
                lt = LogConfig.LoggingType.DB;
                break;
            }
            case "SPLUNK": {
                lt = LogConfig.LoggingType.SPLUNK;
                break;
            }
            case "GCPLOGS": {
                lt = LogConfig.LoggingType.GCPLOGS;
                break;
            }
            case "LOKI": {
                lt = LogConfig.LoggingType.LOKI;
                break;
            }
            default: {
                lt = LogConfig.LoggingType.DEFAULT;
            }
        }
        LogConfig lc = new LogConfig(lt, containerDescription.getLoggerParameters());
        configuration.withLogConfig(lc);
        return configuration;
    }

    private HostConfig containerPortManagementHandler(ContainerConfiguration containerDescription, HostConfig commandBuilder) {
        if (containerDescription.getContainerPortsInternal() != null && containerDescription.getContainerPortsExternal() != null && containerDescription.getContainerPortsExternal().size() == containerDescription.getContainerPortsInternal().size()) {
            LinkedList<ExposedPort> exposedPorts = new LinkedList<ExposedPort>();
            Ports portbindings = new Ports();
            int index = 0;
            while (index < containerDescription.getContainerPortsInternal().size()) {
                ExposedPort tempExposedPort = new ExposedPort(((Integer)containerDescription.getContainerPortsInternal().get(index)).intValue());
                exposedPorts.add(tempExposedPort);
                portbindings.bind(tempExposedPort, Ports.Binding.bindPort((int)((Integer)containerDescription.getContainerPortsExternal().get(index))));
                ++index;
            }
            commandBuilder.withPortBindings(portbindings);
        } else {
            logger.error("portsExternal and portsInternal must be int[] of the same size or they do not exist: {}", (Object)containerDescription.getContainerName());
        }
        return commandBuilder;
    }

    private CreateContainerCmd containerEnviromentVariablesHandler(ContainerConfiguration containerDescription, CreateContainerCmd commandBuilder) {
        if (containerDescription.getContainerEnvVars().isEmpty()) {
            return commandBuilder;
        }
        if (containerDescription.getContainerEnvVars() != null && !containerDescription.getContainerEnvVars().isEmpty()) {
            LinkedList<String> formattedEnvVars = new LinkedList<String>();
            for (String env : containerDescription.getContainerEnvVars()) {
                if (env.trim().isEmpty()) continue;
                formattedEnvVars.add(env.trim());
            }
            commandBuilder = commandBuilder.withEnv(formattedEnvVars);
        }
        return commandBuilder;
    }

    private HostConfig containerVolumeMangamentHandler(ContainerConfiguration containerDescription, HostConfig hostConfiguration) {
        if (containerDescription.getContainerVolumes().isEmpty()) {
            return hostConfiguration;
        }
        LinkedList<Bind> bindsToAdd = new LinkedList<Bind>();
        if (containerDescription.getContainerVolumes() != null && !containerDescription.getContainerVolumes().isEmpty()) {
            for (Map.Entry element : containerDescription.getContainerVolumes().entrySet()) {
                if (((String)element.getKey()).isEmpty() || ((String)element.getValue()).isEmpty()) continue;
                Volume tempVolume = new Volume((String)element.getValue());
                Bind tempBind = new Bind((String)element.getKey(), tempVolume);
                bindsToAdd.add(tempBind);
            }
            hostConfiguration = hostConfiguration.withBinds(bindsToAdd);
        }
        return hostConfiguration;
    }

    private HostConfig containerDevicesHandler(ContainerConfiguration containerDescription, HostConfig hostConfiguration) {
        if (containerDescription.getContainerDevices().isEmpty()) {
            return hostConfiguration;
        }
        if (containerDescription.getContainerDevices() != null && !containerDescription.getContainerDevices().isEmpty()) {
            LinkedList<Device> deviceList = new LinkedList<Device>();
            for (String deviceString : containerDescription.getContainerDevices()) {
                deviceList.add(Device.parse((String)deviceString));
            }
            if (!deviceList.isEmpty()) {
                hostConfiguration = hostConfiguration.withDevices(deviceList);
            }
        }
        return hostConfiguration;
    }

    private boolean doesImageExist(String imageName, String imageTag) {
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
        List images = (List)this.dockerClient.listImagesCmd().exec();
        for (Image image : images) {
            if (image.getRepoTags() == null) continue;
            String[] stringArray = image.getRepoTags();
            int n = stringArray.length;
            int n2 = 0;
            while (n2 < n) {
                String tags = stringArray[n2];
                if (tags.contains(String.valueOf(imageName) + ":" + imageTag)) {
                    return true;
                }
                ++n2;
            }
        }
        return false;
    }

    private void cleanUpDocker() {
        if (this.testConnection()) {
            this.dockerServiceListeners.forEach(ContainerOrchestrationServiceListener::onDisabled);
            this.disconnect();
        }
    }

    private boolean connect() {
        if (this.currentConfig.getHostUrl() == null) {
            return false;
        }
        DefaultDockerClientConfig config = DefaultDockerClientConfig.createDefaultConfigBuilder().withDockerHost(this.currentConfig.getHostUrl()).build();
        ApacheDockerHttpClient httpClient = new ApacheDockerHttpClient.Builder().dockerHost(config.getDockerHost()).build();
        this.dockerClient = DockerClientImpl.getInstance((DockerClientConfig)config, (DockerHttpClient)httpClient);
        boolean connected = this.testConnection();
        if (connected) {
            this.dockerServiceListeners.forEach(ContainerOrchestrationServiceListener::onConnect);
        }
        return connected;
    }

    private void disconnect() {
        if (this.testConnection()) {
            try {
                this.dockerServiceListeners.forEach(ContainerOrchestrationServiceListener::onDisconnect);
                this.dockerClient.close();
            }
            catch (IOException e) {
                logger.error("Error disconnecting", (Throwable)e);
            }
        }
    }

    private boolean testConnection() {
        boolean canConnect = false;
        try {
            this.dockerClient.pingCmd().exec();
            canConnect = true;
        }
        catch (Exception exception) {
            canConnect = false;
        }
        return canConnect;
    }

    private static class FrameworkManagedContainer {
        private final String name;
        private final String id;

        public FrameworkManagedContainer(String name, String id) {
            this.name = name;
            this.id = id;
        }

        public int hashCode() {
            return Objects.hash(this.id, this.name);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            FrameworkManagedContainer other = (FrameworkManagedContainer)obj;
            return Objects.equals(this.id, other.id) && Objects.equals(this.name, other.name);
        }
    }
}

