/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.resource;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.jvmmodel.JvmModelInferrerHelper;
import fr.inria.diverse.melange.jvmmodel.MelangeTypesBuilder;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.Transformation;
import fr.inria.diverse.melange.processors.ExactTypeInferrer;
import fr.inria.diverse.melange.processors.ImportDslProcessor;
import fr.inria.diverse.melange.processors.LanguageProcessor;
import fr.inria.diverse.melange.processors.MelangeProcessor;
import fr.inria.diverse.melange.processors.TypingInferrer;
import fr.inria.diverse.melange.processors.WildcardAspectResolver;
import fr.inria.diverse.melange.utils.EPackageProvider;
import java.util.List;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.xml.namespace.XMLNamespacePackage;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.resource.DerivedStateAwareResource;
import org.eclipse.xtext.util.internal.Stopwatches;
import org.eclipse.xtext.xbase.jvmmodel.JvmModelAssociator;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A specialized {@link JvmModelAssociator} that is basically the entry point
 * for all the Melange process.
 * <br>
 * We first retrieve the just-parsed {@link DerivedStateAwareResource},
 * initialize the stopwatches, apply a set of pre-processors that transforms
 * the parsed AST before code generation and set the context of inferrers
 * before calling them.
 */
@SuppressWarnings("all")
public class MelangeDerivedStateComputer extends JvmModelAssociator {
  @Inject
  private MelangeTypesBuilder builder;
  
  @Inject
  private JvmModelInferrerHelper helper;
  
  @Inject
  private EPackageProvider provider;
  
  private List<MelangeProcessor> processors = CollectionLiterals.<MelangeProcessor>newArrayList();
  
  private final static Logger log = Logger.getLogger(MelangeDerivedStateComputer.class);
  
  /**
   * The parameters of this constructor define the list of the processors and
   * the order in which they'll be applied.
   * Yes, that does not make sense, but we don't have Guice Multibindings here.
   */
  @Inject
  public MelangeDerivedStateComputer(final ImportDslProcessor d, final WildcardAspectResolver r, final LanguageProcessor l, final ExactTypeInferrer e, final TypingInferrer t) {
    this.processors.add(d);
    this.processors.add(r);
    this.processors.add(e);
    this.processors.add(l);
    this.processors.add(t);
  }
  
  /**
   * Resets the current state of the {@link EPackageProvider} to discard
   * obsolete {@link EPackage}/{@link GenModel} cache. Then, call each
   * {@link MelangeProcessor} in the right order to give them a chance to
   * process the AST. Finally, simply delegate to the super installDerivedState.
   */
  @Override
  public void installDerivedState(final DerivedStateAwareResource resource, final boolean preLinkingPhase) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("installDerivedState() from [Thread ");
    long _id = Thread.currentThread().getId();
    _builder.append(_id);
    _builder.append("]");
    MelangeDerivedStateComputer.log.debug(_builder);
    final Stopwatches.StoppedTask task = Stopwatches.forTask("installing derived state");
    task.start();
    final XMLNamespacePackage testsBug = XMLNamespacePackage.eINSTANCE;
    testsBug.hashCode();
    Stopwatches.setEnabled(true);
    this.provider.resetFor(resource);
    EObject _head = IterableExtensions.<EObject>head(resource.getContents());
    final ModelTypingSpace root = ((ModelTypingSpace) _head);
    if ((root != null)) {
      try {
        final Consumer<MelangeProcessor> _function = new Consumer<MelangeProcessor>() {
          @Override
          public void accept(final MelangeProcessor p) {
            final Stopwatches.StoppedTask pTask = Stopwatches.forTask(p.getClass().getSimpleName());
            try {
              pTask.start();
              p.preProcess(root, preLinkingPhase);
            } catch (final Throwable _t) {
              if (_t instanceof Exception) {
                final Exception e = (Exception)_t;
                String _simpleName = p.getClass().getSimpleName();
                String _plus = (" exception  in " + _simpleName);
                String _plus_1 = (_plus + " while processing ");
                URI _uRI = resource.getURI();
                String _plus_2 = (_plus_1 + _uRI);
                String _plus_3 = (_plus_2 + " (may be temporary, try a clean all) ");
                MelangeDerivedStateComputer.log.error(_plus_3, e);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            } finally {
              pTask.stop();
            }
          }
        };
        this.processors.forEach(_function);
        this.builder.setContext(resource.getResourceSet());
        this.helper.setContext(resource.getResourceSet());
        task.stop();
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          URI _uRI = resource.getURI();
          String _plus = (" exception  while processing " + _uRI);
          String _plus_1 = (_plus + " (may be temporary, try a clean all) ");
          MelangeDerivedStateComputer.log.error(_plus_1, e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      } finally {
        boolean _containsTransformations = this.containsTransformations(root);
        if (_containsTransformations) {
          super.installDerivedState(resource, preLinkingPhase);
        } else {
          super.installDerivedState(resource, true);
        }
      }
    }
  }
  
  /**
   * Gives an opportunity to the {@link MelangeProcessor}s to discard
   * their modification of the AST of {@code resource}, and prints the
   * stopwatches collected along the way.
   */
  @Override
  public void discardDerivedState(final DerivedStateAwareResource resource) {
    EObject _head = IterableExtensions.<EObject>head(resource.getContents());
    final ModelTypingSpace root = ((ModelTypingSpace) _head);
    if ((root != null)) {
      final Consumer<MelangeProcessor> _function = new Consumer<MelangeProcessor>() {
        @Override
        public void accept(final MelangeProcessor it) {
          it.postProcess(root);
        }
      };
      this.processors.forEach(_function);
    }
    super.discardDerivedState(resource);
    MelangeDerivedStateComputer.log.debug(Stopwatches.getPrintableStopwatchData());
    Stopwatches.resetAll();
  }
  
  public boolean containsTransformations(final ModelTypingSpace root) {
    boolean _isEmpty = IterableExtensions.isEmpty(Iterables.<Transformation>filter(root.getElements(), Transformation.class));
    return (!_isEmpty);
  }
  
  /**
   * Should be invoked right before code generation to complete
   * the derived state model
   */
  public void inferFullDerivedState(final DerivedStateAwareResource resource) {
    super.installDerivedState(resource, false);
  }
}
