/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.metamodel.generator;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import opsemanticsview.OperationalSemanticsView;
import org.eclipse.emf.codegen.ecore.genmodel.GenModelPackage;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EGenericType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gemoc.trace.commons.EcoreCraftingUtil;
import org.eclipse.gemoc.trace.commons.ExecutionMetamodelTraceability;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class TraceMMGeneratorStates {
  private final OperationalSemanticsView mmext;

  private final TraceMMExplorer traceMMExplorer;

  private final String languageName;

  private final boolean gemoc;

  private final EPackage tracemmresult;

  private final TraceMMGenerationTraceability traceability;

  private final Map<EClass, EClass> runtimeToTraced = new HashMap<EClass, EClass>();

  private final Set<EClass> allRuntimeClasses = new HashSet<EClass>();

  private final Set<EClass> allStaticClasses = new HashSet<EClass>();

  private final Set<EClass> allNewEClasses;

  private final Set<EClass> multipleOrig = new HashSet<EClass>();

  public TraceMMGeneratorStates(final OperationalSemanticsView mmext, final TraceMMGenerationTraceability traceability, final TraceMMExplorer traceMMExplorer, final String languageName, final EPackage tracemmresult, final boolean gemoc) {
    this.mmext = mmext;
    this.allNewEClasses = IterableExtensions.<EClass>toSet(Iterables.<EClass>filter(IteratorExtensions.<EObject>toSet(mmext.eAllContents()), EClass.class));
    this.traceability = traceability;
    this.traceMMExplorer = traceMMExplorer;
    this.languageName = languageName;
    this.tracemmresult = tracemmresult;
    this.gemoc = gemoc;
  }

  private void cleanup() {
    final Set<EClass> allCreatedEClasses = IteratorExtensions.<EClass>toSet(Iterators.<EClass>filter(this.tracemmresult.eAllContents(), EClass.class));
    for (final EClass c : allCreatedEClasses) {
      this.cleanupAnnotations(c);
    }
    Iterable<EReference> _filter = Iterables.<EReference>filter(this.runtimeToTraced.values(), EReference.class);
    for (final EReference r : _filter) {
      r.setEOpposite(null);
    }
  }

  public void process() {
    this.handleTraceClasses();
    this.cleanup();
  }

  private void cleanupAnnotations(final EClass eClass) {
    final EAnnotation traceabilityAnnotation = ExecutionMetamodelTraceability.getTraceabilityAnnotation(eClass);
    eClass.getEAnnotations().clear();
    boolean _notEquals = (!Objects.equal(traceabilityAnnotation, null));
    if (_notEquals) {
      eClass.getEAnnotations().add(traceabilityAnnotation);
    }
  }

  private EPackage obtainTracedPackage(final EPackage runtimePackage) {
    EPackage result = this.traceMMExplorer.statesPackage;
    boolean _notEquals = (!Objects.equal(runtimePackage, null));
    if (_notEquals) {
      final EPackage tracedSuperPackage = this.obtainTracedPackage(runtimePackage.getESuperPackage());
      final String tracedPackageName = TraceMMStrings.package_createTracedPackage(runtimePackage);
      final Function1<EPackage, Boolean> _function = (EPackage p) -> {
        return Boolean.valueOf(p.getName().equals(tracedPackageName));
      };
      result = IterableExtensions.<EPackage>findFirst(tracedSuperPackage.getESubpackages(), _function);
      boolean _equals = Objects.equal(result, null);
      if (_equals) {
        result = EcoreFactory.eINSTANCE.createEPackage();
        result.setName(tracedPackageName);
        String _name = result.getName();
        String _plus = ((this.languageName + "_") + _name);
        result.setNsURI(_plus);
        result.setNsPrefix("");
        tracedSuperPackage.getESubpackages().add(result);
      }
    }
    return result;
  }

  private String computeTraceabilityAnnotationValue(final EClass extendedClass) {
    String traceabilityAnnotationValue = null;
    final Function1<EStructuralFeature, Boolean> _function = (EStructuralFeature f) -> {
      return Boolean.valueOf(this.mmext.getDynamicProperties().contains(f));
    };
    final Set<EStructuralFeature> dynamicProperties = IterableExtensions.<EStructuralFeature>toSet(IterableExtensions.<EStructuralFeature>filter(extendedClass.getEStructuralFeatures(), _function));
    boolean _isEmpty = dynamicProperties.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final EStructuralFeature mutableProperty = ((EStructuralFeature[])Conversions.unwrapArray(dynamicProperties, EStructuralFeature.class))[0];
      final String mutablePropertyTraceabilityValue = ExecutionMetamodelTraceability.getTraceabilityAnnotationValue(mutableProperty);
      boolean _notEquals = (!Objects.equal(mutablePropertyTraceabilityValue, null));
      if (_notEquals) {
        final int classSubstringStartIndex = mutablePropertyTraceabilityValue.lastIndexOf("/");
        traceabilityAnnotationValue = mutablePropertyTraceabilityValue.substring(0, classSubstringStartIndex);
      }
    }
    return traceabilityAnnotationValue;
  }

  private boolean isInPackage(final EPackage c, final EPackage p) {
    if ((((!Objects.equal(c, null)) && (!Objects.equal(p, null))) && Objects.equal(c, p))) {
      return true;
    } else {
      EPackage _eSuperPackage = c.getESuperPackage();
      boolean _notEquals = (!Objects.equal(_eSuperPackage, null));
      if (_notEquals) {
        return this.isInPackage(c.getESuperPackage(), p);
      } else {
        return false;
      }
    }
  }

  private Set<EClass> getSubTypesOf(final EClass c) {
    final HashSet<EClass> result = new HashSet<EClass>();
    Iterable<EClass> _filter = Iterables.<EClass>filter(IteratorExtensions.<EObject>toSet(this.mmext.getExecutionMetamodel().eAllContents()), EClass.class);
    for (final EClass someEClass : _filter) {
      boolean _contains = someEClass.getESuperTypes().contains(c);
      if (_contains) {
        result.add(someEClass);
      }
    }
    return result;
  }

  private void getAllInheritance(final Set<EClass> result, final EClass c) {
    boolean _contains = result.contains(c);
    boolean _not = (!_contains);
    if (_not) {
      result.add(c);
      EList<EClass> _eSuperTypes = c.getESuperTypes();
      for (final EClass sup : _eSuperTypes) {
        this.getAllInheritance(result, sup);
      }
      Set<EClass> _subTypesOf = this.getSubTypesOf(c);
      for (final EClass sub : _subTypesOf) {
        this.getAllInheritance(result, sub);
      }
    }
  }

  private Set<EClass> getAllInheritance(final EClass c) {
    final HashSet<EClass> result = new HashSet<EClass>();
    this.getAllInheritance(result, c);
    return result;
  }

  private void handleTraceClasses() {
    EList<EStructuralFeature> _dynamicProperties = this.mmext.getDynamicProperties();
    for (final EStructuralFeature dp : _dynamicProperties) {
      {
        final EClass extendedExistingClass = dp.getEContainingClass();
        this.allRuntimeClasses.add(extendedExistingClass);
        final Set<EClass> allInheritance = this.getAllInheritance(extendedExistingClass);
        this.allRuntimeClasses.addAll(allInheritance);
      }
    }
    final HashMap<EClass, EClass> baseClassToNewEClass = new HashMap<EClass, EClass>();
    for (final EClass c : this.allNewEClasses) {
      final Function1<EClass, Boolean> _function = (EClass cls) -> {
        String _name = cls.getName();
        String _name_1 = c.getName();
        return Boolean.valueOf(Objects.equal(_name, _name_1));
      };
      baseClassToNewEClass.put(IterableExtensions.<EClass>findFirst(Iterables.<EClass>filter(IteratorExtensions.<EObject>toSet(this.mmext.getExecutionMetamodel().eAllContents()), EClass.class), _function), c);
    }
    for (final EClass c_1 : this.allNewEClasses) {
      {
        final Function1<EClass, Boolean> _function_1 = (EClass cls) -> {
          String _name = cls.getName();
          String _name_1 = c_1.getName();
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        };
        final Set<EClass> allInheritance = this.getAllInheritance(IterableExtensions.<EClass>findFirst(Iterables.<EClass>filter(IteratorExtensions.<EObject>toSet(this.mmext.getExecutionMetamodel().eAllContents()), EClass.class), _function_1));
        final Function1<EClass, EClass> _function_2 = (EClass cls) -> {
          EClass _xblockexpression = null;
          {
            final EClass newEClass = baseClassToNewEClass.get(cls);
            EClass _xifexpression = null;
            boolean _equals = Objects.equal(newEClass, null);
            if (_equals) {
              _xifexpression = cls;
            } else {
              _xifexpression = newEClass;
            }
            _xblockexpression = _xifexpression;
          }
          return _xblockexpression;
        };
        Iterables.<EClass>addAll(this.allRuntimeClasses, IterableExtensions.<EClass, EClass>map(allInheritance, _function_2));
      }
    }
    final Function1<EClass, Boolean> _function_1 = (EClass c_2) -> {
      boolean _contains = this.allRuntimeClasses.contains(c_2);
      return Boolean.valueOf((!_contains));
    };
    Iterables.<EClass>addAll(this.allStaticClasses, IterableExtensions.<EClass>filter(Iterables.<EClass>filter(IteratorExtensions.<EObject>toSet(this.mmext.getExecutionMetamodel().eAllContents()), EClass.class), _function_1));
    for (final EClass rc : this.allRuntimeClasses) {
      {
        final Function1<EClass, Boolean> _function_2 = (EClass c_2) -> {
          return Boolean.valueOf(((!c_2.isAbstract()) && this.allRuntimeClasses.contains(c_2)));
        };
        final Set<EClass> concreteSuperTypes = IterableExtensions.<EClass>toSet(IterableExtensions.<EClass>filter(rc.getEAllSuperTypes(), _function_2));
        this.multipleOrig.addAll(concreteSuperTypes);
      }
    }
    final ArrayList<EClass> tracedClasses = new ArrayList<EClass>();
    final List<EClass> runtimeClasses = IterableExtensions.<EClass>toList(this.allRuntimeClasses);
    final Function1<EClass, String> _function_2 = (EClass it) -> {
      return it.getName();
    };
    final List<EClass> runtimeClassesSorted = IterableExtensions.<EClass, String>sortBy(runtimeClasses, _function_2);
    for (final EClass runtimeClass : runtimeClassesSorted) {
      {
        final EClass tracedClass = this.handleTraceClass(runtimeClass);
        tracedClasses.add(tracedClass);
      }
    }
  }

  private EClass handleTraceClass(final EClass runtimeClass) {
    boolean _contains = this.allRuntimeClasses.contains(runtimeClass);
    boolean _not = (!_contains);
    if (_not) {
      return runtimeClass;
    }
    boolean _containsKey = this.runtimeToTraced.containsKey(runtimeClass);
    boolean _not_1 = (!_containsKey);
    if (_not_1) {
      final EClass tracedClass = EcoreFactory.eINSTANCE.createEClass();
      tracedClass.setName(TraceMMStrings.class_createTraceClassName(runtimeClass));
      tracedClass.setAbstract((runtimeClass.isAbstract() || runtimeClass.isInterface()));
      this.runtimeToTraced.put(runtimeClass, tracedClass);
      this.traceability.putTracedClasses(runtimeClass, tracedClass);
      final Function1<EClass, Boolean> _function = (EClass t) -> {
        return Boolean.valueOf(this.allRuntimeClasses.contains(t));
      };
      Iterable<EClass> _filter = IterableExtensions.<EClass>filter(runtimeClass.getESuperTypes(), _function);
      for (final EClass superType : _filter) {
        {
          final EClass tracedSuperType = this.handleTraceClass(superType);
          tracedClass.getESuperTypes().add(tracedSuperType);
        }
      }
      boolean _contains_1 = this.allNewEClasses.contains(runtimeClass);
      final boolean notNewClass = (!_contains_1);
      boolean _isAbstract = tracedClass.isAbstract();
      final boolean notAbstract = (!_isAbstract);
      boolean _isEmpty = tracedClass.getESuperTypes().isEmpty();
      if (_isEmpty) {
        final EGenericType tracedObjectGenericSuperType = EcoreFactory.eINSTANCE.createEGenericType();
        tracedObjectGenericSuperType.setEClassifier(this.traceMMExplorer.specificTracedObjectClass);
        final EGenericType dimensionClassTracedObjectTypeBinding = EcoreFactory.eINSTANCE.createEGenericType();
        tracedObjectGenericSuperType.getETypeArguments().add(dimensionClassTracedObjectTypeBinding);
        dimensionClassTracedObjectTypeBinding.setEClassifier(this.traceMMExplorer.specificDimensionClass);
        dimensionClassTracedObjectTypeBinding.getETypeArguments().add(EcoreFactory.eINSTANCE.createEGenericType());
        tracedClass.getEGenericSuperTypes().add(tracedObjectGenericSuperType);
      }
      final EPackage tracedPackage = this.obtainTracedPackage(runtimeClass.getEPackage());
      tracedPackage.getEClassifiers().add(tracedClass);
      final Function1<EStructuralFeature, Boolean> _function_1 = (EStructuralFeature f) -> {
        return Boolean.valueOf(this.mmext.getDynamicProperties().contains(f));
      };
      final Iterable<EStructuralFeature> dynamicProperties = IterableExtensions.<EStructuralFeature>filter(runtimeClass.getEStructuralFeatures(), _function_1);
      if ((notNewClass && (!IterableExtensions.isEmpty(dynamicProperties)))) {
        final String traceabilityAnnotationValue = this.computeTraceabilityAnnotationValue(runtimeClass);
        boolean _notEquals = (!Objects.equal(traceabilityAnnotationValue, null));
        if (_notEquals) {
          ExecutionMetamodelTraceability.createTraceabilityAnnotation(tracedClass, traceabilityAnnotationValue);
        }
      }
      final Function1<EClass, Boolean> _function_2 = (EClass c) -> {
        return Boolean.valueOf(((!this.allRuntimeClasses.contains(c)) || c.isAbstract()));
      };
      final boolean onlyAbstractSuperTypes = IterableExtensions.<EClass>forall(runtimeClass.getEAllSuperTypes(), _function_2);
      if (((notNewClass && notAbstract) && onlyAbstractSuperTypes)) {
        String _xifexpression = null;
        boolean _contains_2 = this.multipleOrig.contains(runtimeClass);
        if (_contains_2) {
          _xifexpression = TraceMMStrings.ref_OriginalObject_MultipleInheritance(runtimeClass);
        } else {
          _xifexpression = TraceMMStrings.ref_OriginalObject;
        }
        final String refName = _xifexpression;
        final EReference ref = EcoreCraftingUtil.addReferenceToClass(tracedClass, refName, runtimeClass);
        this.traceability.addRefs_originalObject(tracedClass, ref);
      }
      Set<EStructuralFeature> runtimeProperties = new HashSet<EStructuralFeature>();
      boolean _contains_3 = this.allNewEClasses.contains(runtimeClass);
      if (_contains_3) {
        runtimeProperties.addAll(runtimeClass.getEStructuralFeatures());
      } else {
        boolean _isEmpty_1 = IterableExtensions.isEmpty(dynamicProperties);
        boolean _not_2 = (!_isEmpty_1);
        if (_not_2) {
          Iterables.<EStructuralFeature>addAll(runtimeProperties, dynamicProperties);
        }
      }
      boolean _isEmpty_2 = runtimeProperties.isEmpty();
      boolean _not_3 = (!_isEmpty_2);
      if (_not_3) {
        this.traceability.addRuntimeClass(runtimeClass);
      }
      final ArrayList<String> dimensionsGetters = new ArrayList<String>();
      for (final EStructuralFeature runtimeProperty : runtimeProperties) {
        {
          this.traceability.addMutableProperty(runtimeClass, runtimeProperty);
          final EClass valueClass = EcoreFactory.eINSTANCE.createEClass();
          valueClass.setName(TraceMMStrings.class_createStateClassName(runtimeClass, runtimeProperty));
          EStructuralFeature _copy = EcoreUtil.<EStructuralFeature>copy(runtimeProperty);
          final EStructuralFeature copiedProperty = ((EStructuralFeature) _copy);
          if ((copiedProperty instanceof EReference)) {
            ((EReference)copiedProperty).setContainment(false);
            ((EReference)copiedProperty).setEOpposite(null);
            EClassifier _eType = runtimeProperty.getEType();
            ((EReference)copiedProperty).setEType(this.handleTraceClass(((EClass) _eType)));
            ((EReference)copiedProperty).setDerived(false);
            ((EReference)copiedProperty).setChangeable(true);
            ((EReference)copiedProperty).setVolatile(false);
            final EGenericType valueGenericSuperType = EcoreFactory.eINSTANCE.createEGenericType();
            valueGenericSuperType.setEClassifier(this.traceMMExplorer.specificReferenceValueClass);
            final EGenericType valueTypeBinding = EcoreFactory.eINSTANCE.createEGenericType();
            valueGenericSuperType.getETypeArguments().add(valueTypeBinding);
            valueTypeBinding.setEClassifier(((EReference)copiedProperty).getEType());
            valueClass.getEGenericSuperTypes().add(valueGenericSuperType);
          } else {
            valueClass.getESuperTypes().add(this.traceMMExplorer.specificAttributeValueClass);
          }
          valueClass.getEStructuralFeatures().add(copiedProperty);
          this.traceMMExplorer.statesPackage.getEClassifiers().add(valueClass);
          this.traceability.putMutablePropertyToValueProperty(runtimeProperty, copiedProperty);
          ExecutionMetamodelTraceability.createTraceabilityAnnotation(valueClass, 
            ExecutionMetamodelTraceability.getTraceabilityAnnotationValue(runtimeProperty));
          final EClass dimensionClass = EcoreFactory.eINSTANCE.createEClass();
          dimensionClass.setName(TraceMMStrings.class_createDimensionClassName(runtimeClass, runtimeProperty));
          final EGenericType dimensionGenericSuperType = EcoreFactory.eINSTANCE.createEGenericType();
          dimensionGenericSuperType.setEClassifier(this.traceMMExplorer.specificDimensionClass);
          final EGenericType dimensionTypeBinding = EcoreFactory.eINSTANCE.createEGenericType();
          dimensionGenericSuperType.getETypeArguments().add(dimensionTypeBinding);
          dimensionTypeBinding.setEClassifier(valueClass);
          dimensionClass.getEGenericSuperTypes().add(dimensionGenericSuperType);
          this.traceMMExplorer.statesPackage.getEClassifiers().add(dimensionClass);
          final EReference dimensionRef = EcoreCraftingUtil.addReferenceToClass(tracedClass, StringExtensions.toFirstLower(dimensionClass.getName()), dimensionClass);
          dimensionRef.setContainment(true);
          dimensionRef.setLowerBound(0);
          dimensionRef.setUpperBound(1);
          dimensionsGetters.add(EcoreCraftingUtil.stringGetter(dimensionRef));
          this.traceability.putDimensionClass(runtimeProperty, dimensionClass);
          this.traceability.putDimensionRef(runtimeProperty, dimensionRef);
          this.traceability.putValueClass(runtimeProperty, valueClass);
        }
      }
      final EOperation getDimensionsInternal = EcoreFactory.eINSTANCE.createEOperation();
      final EAnnotation getDimensionsAnnotation = EcoreFactory.eINSTANCE.createEAnnotation();
      getDimensionsInternal.getEAnnotations().add(getDimensionsAnnotation);
      getDimensionsInternal.setName("getDimensionsInternal");
      getDimensionsInternal.setLowerBound(0);
      getDimensionsInternal.setUpperBound((-1));
      final EGenericType dimensionGenericSuperType = EcoreFactory.eINSTANCE.createEGenericType();
      dimensionGenericSuperType.setEClassifier(this.traceMMExplorer.specificDimensionClass);
      final EGenericType dimensionTypeBinding = EcoreFactory.eINSTANCE.createEGenericType();
      dimensionGenericSuperType.getETypeArguments().add(dimensionTypeBinding);
      getDimensionsInternal.setEGenericType(dimensionGenericSuperType);
      getDimensionsAnnotation.setSource(GenModelPackage.eNS_URI);
      EMap<String, String> _details = getDimensionsAnnotation.getDetails();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("final EList<SpecificDimension<?>> result = new org.eclipse.emf.ecore.util.BasicInternalEList<SpecificDimension<?>>(Object.class);");
      _builder.newLine();
      _builder.append("result.addAll(super.getDimensionsInternal());");
      _builder.newLine();
      {
        for(final String getter : dimensionsGetters) {
          _builder.append("result.add(");
          _builder.append(getter);
          _builder.append(");");
          _builder.newLineIfNotEmpty();
        }
      }
      _builder.append("return result;");
      _builder.newLine();
      _details.put("body", _builder.toString());
      tracedClass.getEOperations().add(getDimensionsInternal);
      return tracedClass;
    } else {
      return this.runtimeToTraced.get(runtimeClass);
    }
  }
}
