/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.gemoc.generator.util;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Predicate;
import opsemanticsview.OperationalSemanticsView;
import opsemanticsview.Rule;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gemoc.trace.commons.EcoreCraftingUtil;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Given a set of references to classes and properties from the execution metamodel,
 * will filter out all other elements from the execution extension transient model.
 * 
 * For now it is implemented through fully qualified name comparisons.
 */
@SuppressWarnings("all")
public class ExtensionFilter {
  public static class CallPath {
    public boolean isContainedInSet = false;
  }

  private final Set<EClass> chosenClasses;

  private final Set<? extends EStructuralFeature> chosenProperties;

  private final OperationalSemanticsView executionExtension;

  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PRIVATE_SETTER })
  private boolean didFilterSomething = false;

  private final Set<EClass> retainedClasses = new HashSet<EClass>();

  private final Set<EStructuralFeature> retainedProperties = new HashSet<EStructuralFeature>();

  private final Set<Rule> retainedRules = new HashSet<Rule>();

  public ExtensionFilter(final OperationalSemanticsView executionExtension, final Set<EClass> chosenClasses, final Set<? extends EStructuralFeature> chosenProperties) {
    boolean _notEquals = (!Objects.equal(chosenClasses, null));
    if (_notEquals) {
      this.chosenClasses = chosenClasses;
    } else {
      this.chosenClasses = Collections.<EClass>unmodifiableSet(CollectionLiterals.<EClass>newHashSet());
    }
    boolean _notEquals_1 = (!Objects.equal(chosenProperties, null));
    if (_notEquals_1) {
      this.chosenProperties = chosenProperties;
    } else {
      this.chosenProperties = Collections.<EStructuralFeature>unmodifiableSet(CollectionLiterals.<EStructuralFeature>newHashSet());
    }
    this.executionExtension = executionExtension;
  }

  public void execute() {
    if (((!this.chosenClasses.isEmpty()) || (!this.chosenProperties.isEmpty()))) {
      final Function1<EClass, String> _function = (EClass c) -> {
        return EcoreCraftingUtil.getFQN(c, ".");
      };
      final Set<String> chosenClassesFQNs = IterableExtensions.<String>toSet(IterableExtensions.<EClass, String>map(this.chosenClasses, _function));
      final Function1<EStructuralFeature, String> _function_1 = (EStructuralFeature p) -> {
        String _fQN = EcoreCraftingUtil.getFQN(p.getEContainingClass(), ".");
        String _plus = (_fQN + ".");
        String _name = p.getName();
        return (_plus + _name);
      };
      final Set<String> chosenPropertiesFQNs = IterableExtensions.<String>toSet(IterableExtensions.map(this.chosenProperties, _function_1));
      EList<EClass> _dynamicClasses = this.executionExtension.getDynamicClasses();
      for (final EClass element : _dynamicClasses) {
        {
          final String fqn = EcoreCraftingUtil.getFQN(element, ".");
          boolean _contains = chosenClassesFQNs.contains(fqn);
          if (_contains) {
            this.retainedClasses.add(element);
            CollectionExtensions.<EClass>addAll(this.retainedClasses, element);
            this.retainedProperties.addAll(element.getEStructuralFeatures());
          }
        }
      }
      EList<EStructuralFeature> _dynamicProperties = this.executionExtension.getDynamicProperties();
      for (final EStructuralFeature element_1 : _dynamicProperties) {
        {
          String _fQN = EcoreCraftingUtil.getFQN(element_1.getEContainingClass(), ".");
          String _plus = (_fQN + ".");
          String _name = element_1.getName();
          final String fqn = (_plus + _name);
          boolean _contains = chosenPropertiesFQNs.contains(fqn);
          if (_contains) {
            this.retainedProperties.add(element_1);
            this.retainedClasses.add(element_1.getEContainingClass());
            CollectionExtensions.<EClass>addAll(this.retainedClasses, element_1.getEContainingClass());
            if ((element_1 instanceof EReference)) {
              this.retainedClasses.add(((EReference)element_1).getEReferenceType());
              CollectionExtensions.<EClass>addAll(this.retainedClasses, ((EReference)element_1).getEReferenceType());
            }
          }
        }
      }
      EList<Rule> _rules = this.executionExtension.getRules();
      for (final Rule element_2 : _rules) {
        boolean _isStepRule = element_2.isStepRule();
        if (_isStepRule) {
          this.retainedRules.add(element_2);
          final Function1<EParameter, EClassifier> _function_2 = (EParameter p) -> {
            return p.getEType();
          };
          Iterable<EClass> _filter = Iterables.<EClass>filter(ListExtensions.<EParameter, EClassifier>map(element_2.getOperation().getEParameters(), _function_2), EClass.class);
          for (final EClass paramClass : _filter) {
            this.retainedClasses.add(paramClass);
          }
          EClassifier _eType = element_2.getOperation().getEType();
          if ((_eType instanceof EClass)) {
            EClassifier _eType_1 = element_2.getOperation().getEType();
            this.retainedClasses.add(((EClass) _eType_1));
          }
          this.retainedClasses.add(element_2.getContainingClass());
        }
      }
      for (final EClass c1 : this.retainedClasses) {
        final Function1<EClass, Boolean> _function_3 = (EClass c) -> {
          return Boolean.valueOf((!Objects.equal(c, c1)));
        };
        Iterable<EClass> _filter_1 = IterableExtensions.<EClass>filter(this.retainedClasses, _function_3);
        for (final EClass c2 : _filter_1) {
          boolean _contains = c1.getEAllSuperTypes().contains(c2);
          if (_contains) {
            boolean _hasSuperTypePathContainedIn = ExtensionFilter.hasSuperTypePathContainedIn(c1, c2, this.retainedClasses);
            boolean _not = (!_hasSuperTypePathContainedIn);
            if (_not) {
              c1.getESuperTypes().add(c2);
            }
          }
        }
      }
      for (final Rule r1 : this.retainedRules) {
        final Function1<Rule, Boolean> _function_4 = (Rule r) -> {
          return Boolean.valueOf((!Objects.equal(r, r1)));
        };
        Iterable<Rule> _filter_2 = IterableExtensions.<Rule>filter(this.retainedRules, _function_4);
        for (final Rule r2 : _filter_2) {
          HashSet<Rule> _hashSet = new HashSet<Rule>();
          boolean _callsIndirectly = this.callsIndirectly(r1, r2, _hashSet);
          if (_callsIndirectly) {
            r1.getCalledRules().add(r2);
          }
        }
      }
      for (final Rule r : this.retainedRules) {
        {
          final Predicate<Rule> _function_5 = (Rule r2_1) -> {
            boolean _contains_1 = this.retainedRules.contains(r);
            return (!_contains_1);
          };
          r.getCalledRules().removeIf(_function_5);
          final Predicate<Rule> _function_6 = (Rule r2_1) -> {
            boolean _contains_1 = this.retainedRules.contains(r);
            return (!_contains_1);
          };
          r.getOverridenBy().removeIf(_function_6);
          boolean _contains_1 = this.retainedRules.contains(r.getOverrides());
          boolean _not_1 = (!_contains_1);
          if (_not_1) {
            r.setOverrides(null);
          }
        }
      }
      Set<EObject> _set = IteratorExtensions.<EObject>toSet(this.executionExtension.eAllContents());
      for (final EObject element_3 : _set) {
        this.tryRemove(element_3);
      }
    }
  }

  public boolean callsIndirectly(final Rule origin, final Rule destination, final Set<Rule> visited) {
    boolean _contains = visited.contains(origin);
    boolean _not = (!_contains);
    if (_not) {
      boolean _equals = Objects.equal(origin, destination);
      if (_equals) {
        return true;
      }
      EList<Rule> _calledRules = origin.getCalledRules();
      for (final Rule r : _calledRules) {
        {
          final boolean result = this.callsIndirectly(r, destination, visited);
          if (result) {
            return true;
          }
        }
      }
    }
    return false;
  }

  public static Set<ExtensionFilter.CallPath> findCallPaths(final Rule origin, final Rule destination, final Set<Rule> mustBeContainedIn, final Set<Rule> visited, final boolean containedSoFar) {
    final HashSet<ExtensionFilter.CallPath> result = new HashSet<ExtensionFilter.CallPath>();
    boolean _contains = visited.contains(origin);
    boolean _not = (!_contains);
    if (_not) {
      visited.add(origin);
      boolean _equals = Objects.equal(origin, destination);
      if (_equals) {
        final ExtensionFilter.CallPath path = new ExtensionFilter.CallPath();
        path.isContainedInSet = containedSoFar;
        result.add(path);
      }
      EList<Rule> _calledRules = origin.getCalledRules();
      for (final Rule s : _calledRules) {
        {
          final boolean containedSoFarNext = (containedSoFar && mustBeContainedIn.contains(s));
          final Set<ExtensionFilter.CallPath> interResult = ExtensionFilter.findCallPaths(s, destination, mustBeContainedIn, visited, containedSoFarNext);
          result.addAll(interResult);
        }
      }
    }
    return result;
  }

  private static boolean hasSuperTypePathContainedIn(final EClass origin, final EClass destination, final Set<EClass> containedIn) {
    boolean _equals = Objects.equal(origin, destination);
    if (_equals) {
      return true;
    }
    final Function1<EClass, Boolean> _function = (EClass s) -> {
      return Boolean.valueOf(containedIn.contains(s));
    };
    Iterable<EClass> _filter = IterableExtensions.<EClass>filter(origin.getESuperTypes(), _function);
    for (final EClass s : _filter) {
      {
        final boolean result = ExtensionFilter.hasSuperTypePathContainedIn(s, destination, containedIn);
        if (result) {
          return true;
        }
      }
    }
    return false;
  }

  private Set<EObject> tryRemove(final EObject element) {
    return null;
  }

  @Pure
  public boolean isDidFilterSomething() {
    return this.didFilterSomething;
  }

  private void setDidFilterSomething(final boolean didFilterSomething) {
    this.didFilterSomething = didFilterSomething;
  }
}
