# ROOMLanguage
The Real Time Object Oriented Modeling (ROOM)

eTrice comprises several models:

- the ROOM model (*.room) -- defines model classes and the logical structure of the model
- the Config model (*.config) -- defines configuration values for attributes
- the Physical model (*.etphys) -- defines the structure and properties of the physical system
- the Mapping model (*.etmap) -- defines a mapping from logical elements to physical elements


In the following diagram the models and their relations are depicted. The meaning of the arrows is: uses/references.

![Model overview](../images/080-models.jpg)

| Features |     |
| -------- | --- |
| Contains: | [LogicalModel][], [PhysicalModel][], [MappingModel][], [ConfigModel][] |

## ConfigModel
The ConfigModel describes the Attribute configuration of ActorInstances and PortInstances. 

The scope of this model is the configuration of Attributes of the LogicalModel.
Thus it provides enhanced capabilities for assigning default values to Attributes, which are:

- type safe value assignment
- setting on class level
- setting on instance level

Values defined for class attributes are used for all instances unless there is an instance value configured for the same attribute.
The configuration is available for actors and ports, thus ActorClasses/ActorRefs and ProtocolClasses/Ports.

```etconfig
ConfigModel ExampleConfig {
	import Example.*

	ActorClassConfig ActorClass1 {
		Attr attribute1 = 4
	}

	ActorInstanceConfig LogSys/subsysRef/actor1 {
		Attr attribute1 = 7
	}
}
```


| Features |     |
| -------- | --- |
| Uses: | [Attribute][] |



---


## LogicalModel
The LogicalModel describes the logical structure and behavior of a ROOM application

The ROOM model defines DataTypes, ProtocolClasses, ActorClasses, SubSystemClasses and LogicalSystems.
Thereby the three latter form a hierarchy. The LogicalSystem is the top level element of the structure. 
It contains references to SubSystemClass elements. The SubSystemClass in turn contains 
references to ActorClass elements which again contain (recursively) references to 
ActorClass elements. The complete structural hierarchy implies a tree which has the 
LogicalSystem as root and where each reference stands for a new node with possibly further 
branches.


| Features |     |
| -------- | --- |
| Contains: | [LogicalSystem][], [SubSystemClass][], [ActorClass][], [ProtocolClass][], [DataType][], [AnnotationType][] |



---


### ActorClass
An actor is the basic structural building block for building systems with ROOM

An ActorClass consists of three main parts:

- **Interface** (external interface) specifies the communication to 'outside' actors and consists of Ports.
- **Structure** (internal interface) contains Ports, Attributes and ActorRefs. These elements are accessible from the Behavior part of the actor (in contrary to the external interface above). An ActorClass can be composed of other actors again by declaring ActorRefs. Also this part declares the connection of ports in form of Bindings and LayerConnections.
- **Behavior** is described by the StateMachine. It can receive and send messages from the ports, declared in the Structure above. The Attributes can be used to store data during an state transition. Furthermore it is possible to declare Operations. They can be used to define reusable logic, that is invoked during a state transition.


![ActorClass](../images/040-ActorClass.png)

```room
 ActorClass ExampleActorClass {
	 Interface {
		 Port port1: ProtocolClass1
		 Port port4: ProtocolClass1
	 }
	 Structure {
		 external Port port1
		 conjugated Port port2: ProtocolClass1
		 conjugated Port port3: ProtocolClass1
		 
		 ActorRef ActorRef_A: ActorClass2
		 ActorRef ActorRef_B: ActorClass3
		 
		 Binding port2 and ActorRef_A.port5
		 // ...
	 }
	 Behavior {
		 // ...
	 }
 }
```


| Features |     |
| -------- | --- |
| Contains: | [ExecutionType][], [ActorRef][], [Port][], [SAP][], [SPP][], [ServiceImplementation][], [Binding][], [LayerConnection][], [Attribute][], [Operation][], [StateMachine][], [Annotation][] |
| Uses: | [Inheritance][] |

| Feature Usage |     |
| ------------- | --- |
| Typecasts: | [ActorRef][] |
| Is contained in: | [LogicalModel][] |
| Is edited by: | [GraphicalStructureEditor][] |

**Example**:


---


### ActorRef
An ActorRef is an instance of an ActorClass

- ActorClass: The type of the ActorRef
- Multiplicity: The number of instances. A number greater than one can be seen as an array of instances
- Reference Type: Can be fixed or optional. Fixed requires an integer multiplicity and results in an static instantiation with an fixed number of instances during runtime . Optional denotes an dynamic instantiation, where ActorRefs can be created in arbitrary number during runtime. In this case, the multiplicity has to be set to '*'

| Properties | Values |
| ---------- | ------ |
| multiplicity | *1..n*, <em>*</em> |

| Features |     |
| -------- | --- |
| Is of type: | [ActorClass][] |
| Uses: | [Replication][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][], [SubSystemClass][] |
| Is edited by: | [GraphicalStructureEditor][], [ActorRefPropertyDialog][] |

**Example**:

```room
SubSystemClass SubSystemExample {
	ActorRef mainActor : ActorClassExample
	
	LogicalThread default_thread
}

ActorClass ActorClassExample {
	Structure {
		ActorRef sender : Sender
		ActorRef receiver : Receiver
		
		Binding receiver.port and sender.port
	}
}

ActorClass ActorClassExampleReplicated {
	Structure {
		ActorRef sender[3]: Sender
		ActorRef receiver[3] : Receiver
		
		Binding receiver.port and sender.port
		/* Equivalent to:
		 *  Binding receiver[1].port and sender[1].port
		 *  Binding receiver[2].port and sender[2].port
		 * ....
		 */		
	}
}
```

![ActorRef instance diagram](../images/300-ActorRefInstanceDiagram.jpg)

Instance hierarchy of ActorRef Example (*System(System)* not shown in code snippet)


---


### Annotation
An Annotation can be attached to a ROOM class to specify the properties of its AnnotationType

It refers to an AnnotationType and may have to pass key value pairs. Its notation is similar to Java:

```room
@AnnotationType1
@AnnotationType2(key1="STRING", key2=3, ...)
```

See section Annotations for further reading.


| Features |     |
| -------- | --- |
| Is of type: | [AnnotationType][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [LogicalSystem][], [ActorClass][], [SubSystemClass][], [ProtocolClass][], [DataClass][] |

**Example**:

```room
import etrice.api.annotations.BehaviorManual

ActorClass ComponentAbstraction {
	Interface {
		conjugated Port port1: Protocol1
	}
	Structure {
		external Port port1
	}
	Behavior {
		// custom/external state machine implementation
		@BehaviorManual
	}
}
```

---


### AnnotationType
AnnotationTypes can be used to tag ROOM classes for further custom processing

They provide the ability to associate custom properties to ROOM classes, that adjust or toggle features, like generation or the runtime behavior.
eTrice comes with predefined annotations, which can be found in Annotations.room within the eTrice modellib.

See section Annotations for further reading.



| Feature Usage |     |
| ------------- | --- |
| Typecasts: | [Annotation][] |
| Is contained in: | [LogicalModel][] |


---


### Attribute
An Attribute is a member variable of a class

An Attribute can be be used to store arbitrary data. There are two common conceptual purpose of use:


- model current system state (state machine variable)
- store reference to more fine-grained components (e.g. c pointer to handle)

Attributes can be defined in several ROOM classes.

| Properties | Values |
| ---------- | ------ |
| defaultValueLiteral | *target code* |
| multiplicity | *1..n* |
| ref |  |

| Features |     |
| -------- | --- |
| Is of type: | [DataType][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][], [ProtocolClass][], [DataClass][] |
| Is used by: | [ConfigModel][] |

**Example**:

```room
import etrice.api.types.*

DataClass SimpleDataClass {
	Attribute attribute1: int16
	Attribute attribute2: uint32
}

ActorClass ActorClassWithAttributes {
	Structure {
		/** attribute of a PrimitiveType */
		Attribute attribute1: int32
		/** attribute of a DataClass */
		Attribute attribute2: SimpleDataClass
	}
}

ActorClass ActorClassWithAttributes2 {
	Structure {
		/** attribute with multiplicity */
		Attribute arrayAttribute[8] : uint32
		/** attribute as a reference (void pointer) */
		Attribute refAttribue : voidType ref
	}
}

ActorClass ActorClassWithAttributeInitialization {
	Structure {
		Attribute attribute1: uint32 = "3"
		Attribute attribute2: SimpleDataClass = "{1, 2}"
		Attribute arrayAttribute[8] : uint32 = "0" // or {0,0,0, ...}
		Attribute refAttribue : voidType ref = "NULL" // set reference in constructor or in state machine
	}
}
```

---


### Binding
A Binding connects two Ports with each other

In essence, a binding is a abstraction for an underlying communication channel whose function is to convey messages from one port to the other.
The precise semantics of these channels are not defined in the Binding. Instead, they are determined by the ProtocolClasses that are associated with the Ports at the end of the Binding.

```room
ActorClass ExampleActorClass {
	Structure {
		conjugated Port sender: ProtocolClass1
		ActorRef actorRef: ActorClass2
		
		Binding sender and actorRef.receiver
	}
}
```


| Features |     |
| -------- | --- |
| Uses: | [Port][]: endpoint1, [Port][]: endpoint2 |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][], [SubSystemClass][] |
| Is edited by: | [GraphicalStructureEditor][] |


---


### CPBranchTransition
a choice point branch transition is an outgoing transition from a choice point and is traversed if its conditions is evaluated to true

A choice point together with its outgoing transitions can be thought of as a if, else-if, else cascade in traditional
		programming languages. The choice point branch transition corresponds to an if clause while the final else is modeled as
		a continuation transition. Coming from a choice point, a choice point branch transition is traversed whenever its
		condition evaluates to true. No order of evaluations is guaranteed.


| Features |     |
| -------- | --- |
| Is a: | [Transition][] |



---


### ChoicePoint
a choice point is the state machine counterpart of a conditional statement

A ChoicePoint in ROOM has one incoming transition and an arbitrary number of outgoing transitions.
One of the outgoing transitions has no condition (a ContinuationTransition). It is the default branch that is taken
if none of the transition conditions evaluated to true. All other transitions going out from the ChoicePoint
are CPBranchTransitions which have a mandatory condition (keyword 'cond'). The evaluation order of the conditions
is not deterministic.


| Features |     |
| -------- | --- |
| Is a: | [StateGraphNode][] |
| Uses: | [ContinuationTransition][] |



---


### CommunicationType
The CommunicationType defines the communication semantics of a ProtocolClass

Since from ROOM models executable code can be generated, it is important to define the way the actors are executed and communicate with each other.
The combination of communication and execution is called the *execution model*. Therefore the ExecutionType of an actor and the CommunicationType of the ports has to be considered.

The CommunicationType of a ProtocolClass (and thus of a Port) specifies in which way the communication should happen:

- **message driven** -- asynchronous, non blocking, no return value: Usually the message driven communication is implemented with message queues. Message queues are inherently asynchronous and enable a very good decoupling of the communicating parties.
- **data driven** -- asynchronous, non blocking, no return value: In data driven communication sender and receiver often have a shared block of data. The sender writes the data and the receiver polls the data.
- _**function call** -- synchronous, blocking, return value: Regular function call as known in most programming languages._ (not supported yet)

CommunicationType relates with the [ExecutionType][] of an ActorClass, e.g. a data-driven port needs a cyclic thread, that polls the shared data.

| Properties | Values |
| ---------- | ------ |
| type | *eventdriven*, *datadriven*, *sync* |


| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ProtocolClass][] |
| Is used by: | [ExecutionType][] |

**Example**:

```room

import etrice.api.types.*

/** default is eventdriven */
ProtocolClass EventdrivenProtocolClass1 {
	// explicit: eventdriven ProtocolClass EventdrivenProtocolClass {
	incoming {
		/** message without data */
		Message msg1()
		/** message with data */
		Message msg2(int32)
	}
	outgoing {
		/** eventdriven ProtocolClass can have message into two directions */
		Message msg4()
	}
}

datadriven ProtocolClass DatadrivenProtocolClass {
	incoming {
		/** a datadriven message needs data */
		Message signal1 (int32)
	}
	// datadriven ProtocolClass can only have incoming messages (signals)
}

//  sync is not supported yet
//	sync ProtocolClass SyncProtcolClass { 
//		
//	}
```

---


### ContinuationTransition
the continuation transition is a transition with just an optional action code

A continuation transition is used as default branch of a choice point or as outgoing transition of an entry point


| Features |     |
| -------- | --- |
| Is a: | [Transition][] |

| Feature Usage |     |
| ------------- | --- |
| Is used by: | [ChoicePoint][] |


---


### DataClass
A DataClass is a composition of Attributes

Intended to model a type that primarily consists of data, which is usually grouped together in some manner. DataClasses roughly translate to Java classes without interaction or C <em>struct</em>s.

```room
DataClass TCPConnectionData {
	Attribute IPAddr: string
	Attribute TcpPort: int32
}
```


| Features |     |
| -------- | --- |
| Is a: | [DataType][] |
| Contains: | [Attribute][], [Operation][], [Annotation][] |
| Uses: | [Inheritance][] |


**Example**:

```room
DataClass SimpleDataClass {
	Attribute attribute1: uint16
	Attribute attribute2: uint32
}

DataClass DataClassExample {
	Attribute attribute1: uint32
	Attribute attribute2: SimpleDataClass
	Attribute attribute3: voidType ref
	
	Operation operation1(param1: uint32, param2: uint16): boolean '''
		return true;
	'''
}
```

---


### DataType
A DataType can take 4 forms and types data elements like an Attribute or Operation argument




| Feature Usage |     |
| ------------- | --- |
| Inheriting features: | [PrimitiveType][], [Enumeration][], [DataClass][], [ExternalType][] |
| Typecasts: | [Attribute][] |
| Is contained in: | [LogicalModel][] |
| Is used by: | [Operation][] |


---


### EntryPoint
an entry point is an explicit entry point in a sub state machine to which transitions in the parent state graph can connect

text


| Features |     |
| -------- | --- |
| Is a: | [TrPoint][] |



---


### Enumeration
An EnumerationType declares an enumeration similar to most well-known languages


| Properties | Values |
| ---------- | ------ |
| literals | *name* |

| Features |     |
| -------- | --- |
| Is a: | [DataType][] |


**Example**:

```room
Enumeration EOnOff {
	Off = 0, // explicit value=0
	On = 1 // explicit value=1 
}

Enumeration EDay {
	SUN,
	MON,
	TUE,
	WED,
	THU,
	FRI,
	SAT // implicit enumeration 0..6
}
```

---


### ExecutionType
Determines the execution type of an actor

Since from ROOM models executable code can be generated, it is important to define the way the actors are 
executed and communicate with each other. The combination of communication and execution is called the 
*execution model*. Therefore the ExecutionType of an actor and the CommunicationType of the ports has to be considered.

The ExecutionType of an ActorClass specifies in which way its instance (ActorRef) should be executed:

- **execution by receive event**: The message queue or the event dispatcher calls a **receive event** function of the message receiver and thereby executes the processing of the event.
- **polled execution**: The objects are processed by a cyclic **execute** call
- _**execution by function call**: The caller executes the called object via function call_ (not supported yet)
- **mixture**: An asynchronous execution combines an event dispatcher and a polled execution.


Thereby the ExecutionType determines the execution mode of the actor's logical thread:

![Thread of Control](../images/010-RoomIntroduction03.png)
		
The actual execution of the underlying physical thread can be specified in the PhysicalModel in conjunction with the MappingModel.

ExecutionType relates to the [CommunicationType][], e.g. if an actor uses data-driven ports, it should support an polled execution.

| Properties | Values |
| ---------- | ------ |
| mode | *eventdriven*, *datadriven*, *async*, *sync* |

| Features |     |
| -------- | --- |
| Uses: | [CommunicationType][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][] |

**Example**:

```room
/** default is eventdriven */
eventdriven ActorClass EventdrivenActor {
	// only event-driven Ports and ActorRefs allowed
}

datadriven ActorClass DatadrivenActor {
	// only data-driven Ports and ActorRefs allowed
}

async ActorClass MixedActor{
	// both data/event-driven Ports and ActorRefs allowed
}
```

---


### ExitPoint
an exit point is an explicit exit point in a sub state machine from which transitions in the parent state graph can start

text


| Features |     |
| -------- | --- |
| Is a: | [TrPoint][] |



---


### ExternalEndPort
A ExternalEndPort is an interface Port, that is made accessible to the internal interface of an ActorClass

```room
ActorClass ExternalEndPortExample {
	Interface {
		// externalEndPort is connect from 'outside' and thus needs a Binding from containing ActorClass
		Port externalEndPort : PSimpleProtocol
	}
	Structure {
		external Port externalEndPort
	}
	Behavior {
		// send/receive messages from externalEndPort
	}
}
```


| Features |     |
| -------- | --- |
| Is a: | [Port][] |



---


### ExternalType
An ExternalType is used to make an target language type accessible in ROOM


| Properties | Values |
| ---------- | ------ |
| targetName | *identifier name* |

| Features |     |
| -------- | --- |
| Is a: | [DataType][] |


**Example**:

```room
// Include is needed when used (e.g. in ActorClassWithExternalType)
ExternalType someStructType -> "struct FILE_HANDLE"

ActorClass ActorClassWithExternalType{
	Structure {
		usercode1 '''
			// #include <___.h> /* User includes here*/
		'''
		Attribute someHandle : someStructType ref // needs include
	}
	Behavior {
		Operation operation1(param1: charPtr) '''
			// external calls or casts may need includes
			write(someHandle, param1);
		'''
	}
}
```

---


### Inheritance
A class can specify a single super class and inherits elements from the super class hierarchy

When a ROOM class specifies a super class, it generally inherits all elements and properties.
In several cases, it is possible, to override these inherited elements. Generally, eTrice has two semantics of overriding: refinement and replacement.
Refinement is used in most cases (e.g. StateMachine) and realizes an extension of the overridden elements.
In this case, if a sub class overrides a piece of logic from a super class, it will always be executed subsequently to the inherited.
Contrary to this, replacement is applied to overridden Operations, similar to programming languages C++ and Java.

A formal definition of several variants of overriding is given below:

- early or late resolve - if element is overridden, which one should the super class use by default - own or override?
- replacing or refining - ignore inherited code or prepend inherited code automatically?
- (non-)accessible - if element is overridden, is super class' original accessible from sub class? E.g. super.foo()
- implicit or explicit - does it use a distinct model element or keyword?


Examples from programming languages:  
C++ virtual function and Java override ↔ accessible, explicit, late, replacing  
C++ function redefine ↔ accessible, implicit, early, replacing  
C++ destructor ↔ late, refining

eTrice override of model elements:  
Operations (C generation)  ↔ non-accessible, explicit, late, replacing  
Operations (Java generation) ↔ accessible, explicit, late, replacing  
State and Transitions ↔ non-accessible, explicit, late, refining  
ctor/dtor ↔ non-accessible, implicit, late, refining  
StateMachine ↔ non-accessible, implicit, late, refining  
UserCode ↔ non-accessible, implicit, late, refining



| Feature Usage |     |
| ------------- | --- |
| Is used by: | [ActorClass][], [StateMachine][], [StateGraphNode][], [State][], [SimpleState][], [RefinedState][], [Transition][], [ProtocolClass][], [DataClass][] |

**Example**:

```room
ActorClass ActorSubClass extends ActorBaseClass {
	// inherits all elements from super type hierarchy
}

ActorClass ActorBaseClass {
	Interface {
		Port port1 : ProtocolBaseClass
	}
	Structure {
		Attribute attribute1 : uint32
	}
	Behavior {
		Operation operation1() '''
			return;
		'''
	}
}

ProtocolClass ProtocolSubClass extends ProtocolBaseClass {
	// inherits all elements from super type hierarchy
}

ProtocolClass ProtocolBaseClass {
	incoming {
		Message message1()
	}
}

DataClass DataSubClass extends DataBaseClass {
	// inherits all elements from super type hierarchy
}

DataClass DataBaseClass {
	Attribute attribute1 : uint32
}
```

---


### InitialTransition
the initial transition is used to identify the initial state

The initial transition connects the initial point to a state. There can be at most one initial transition
		in a state machine. Under special circumstances the initial transition can be omitted.


| Features |     |
| -------- | --- |
| Is a: | [Transition][] |



---


### InternalEndPort
A InternalEndPort is an local Port, that is declared in the internal interface of an ActorClass

```room
ActorClass InternalEndPortExample {
	Structure {
		Port internalEndPort : PSimpleProtocol
		ActorRef actorRef1 : SimpleActorClass
		
		// internalEndPort lives 'local' and
		// thus needs a Binding to port of a ActorRef
		Binding internalEndPort and actorRef1.externalPort2 
	}
	Behavior {
		// send/receive messages from internalEndPorts
	}
}
```
![InternalEndPort](../images/300-InternalEndPort.png)


| Features |     |
| -------- | --- |
| Is a: | [Port][] |



---


### LayerConnection
A LayerConnection associates a SPP to an ActorRef, resulting in an connection of all SAPs on its instance hierarchy


- An actor class can define a Service Provision Point (SPP) to publish a specific service, defined by a protocol class
- An actor class can define a Service Access Point (SAP) if it needs a service, defined by a protocol class
- For a given actor hierarchy, a LayerConnection defines which SAP will be satisfied by (connected to) which SPP



| Features |     |
| -------- | --- |
| Uses: | [SAP][]: SAPoint, [SPP][]: SPPoint |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][], [SubSystemClass][] |
| Is edited by: | [GraphicalStructureEditor][] |


---


### LogicalSystem
The LogicalSystem is the topmost structural class. It assembles a distributed system by means of sub systems

It describes the logical topology of your distributed system and is composed of sub systems (SubSystemRefs). Thus it is the notationally root of every instance path or actor hierarchy.


| Features |     |
| -------- | --- |
| Contains: | [SubSystemRef][], [Annotation][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [LogicalModel][] |
| Is used by: | [MappingModel][] |


---


### Operation
An Operation is a member function of a class

Operations can be used to define a piece of reusable logic. The definition consists of:

- Arbitrary amount of arguments
- Return type
- User code body, which can access the structural part of the containing class (e.g. attributes)
- 'override' keyword, replaces the logic of the inherited operation having the same signature


| Properties | Values |
| ---------- | ------ |
| returnType | *DataType* |
| arguments | *name : DataType* |

| Features |     |
| -------- | --- |
| Uses: | [DataType][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][], [ProtocolClass][], [DataClass][] |

**Example**:

```room
import etrice.api.types.*

DataClass DataClassWithOperation {
	Attribute attribute1 : uint32
	
	Operation operation1(param1: uint32, param2: int32): boolean '''
		return attribute1 > (param1 - param2);
	'''
}

ActorClass ActorClassWithOperation {
	Structure {
		Attribute attribute1 : uint32
	}
	Behavior {
		Operation operation1(param1: uint32, param2: int32): boolean '''
			return attribute1 > (param1 - param2);
		'''
	}
}

ActorClass ActorClassWithOperation2 {
	Structure {
		usercode1 '''
			// #include <___.h> /* User includes here */
		'''
		Attribute someHandle : voidType ref
	}
	Behavior {
		Operation operation1(param1: charPtr) '''
			// external calls or casts may need includes
			write(someHandle, param1);
		'''
	}
}
```

---


### Port
A Port is an instance of a ProtocolClass and the interface for an ActorClass

Once a ProtocolClass has been created, it can be used to define actor interfaces. This is accomplished by means of Ports. 
A Port is a declaration that the set of messages defined by its ProtocolClass is now part of the actor's interface.
It provides strong decoupling of ActorClasses from each other, thus enabling easy testability, reusability and deployment of actors to different threads or nodes.

```room
ActorClass Example {
	Structure{
		Port port0 : ProtocolClass1
	}
	Behavior {
		// send/receive message from port0
	}
}
```

For communication between two actors to take place, a connection must be established between a port on one of the actors and a port on the other.
One condition is, that both Ports have compatible ProtocolClasses. In most cases the Ports simply refer to the same protocol.
In addition, a ProtocolClass has an imposed directionality - it defines one subset of messages as incoming and the complementary subset as outgoing.
Which subset is labeled as incoming and outgoing is arbitrary, it simply depends on the point of view, that was taken when defining.
Therefore Ports can be 'regular' and 'conjugated'. When two actors communicate by a connected pair of Ports, one Port has to be regular and the other conjugated.
The ProtocolClass' incoming messages are on one side received by the regular Port and on the other sent by the conjugated Port (outgoing message vice versa).

The necessity of port connections can be defined using the `mandatory` or `optional` keywords at the beginning of the port definition. By default, all ports are defined as `optional`, and no warning will be shown if they are not connected. If the `mandatory` keyword is used, warnings will be shown in the following cases:

-	For *internal* ports, a warning is shown under the ActorClass name.
-	For *relay* and *external* ports with a mandatory connection, the warning is shown where their ActorClass is referenced.
-	For *replicated ActorRef* and *Port* definitions, if the port is partially connected, the number of peers is shown as a warning.
-	For *abstract* classes, no warning will be shown. However, if an ActorClass extends an abstract class with a mandatory Port, a warning will still be shown.
-	If an `optional` relay port is connected to a port with `mandatory` connection necessity, a warning is shown. 

A connection of Ports is denoted by a Binding. 

| Properties | Values |
| ---------- | ------ |
| connection necessity | *optional*, *mandatory* |
| conjugated | *regular*, *conjugated* |
| multiplicity | *1..n*, <em>*</em> |

| Features |     |
| -------- | --- |
| Is of type: | [ProtocolClass][] |
| Uses: | [Replication][] |

| Feature Usage |     |
| ------------- | --- |
| Inheriting features: | [RelayPort][], [ExternalEndPort][], [InternalEndPort][] |
| Is contained in: | [ActorClass][] |
| Is edited by: | [GraphicalStructureEditor][], [PortPropertyDialog][] |
| Is used by: | [Binding][]: endpoint1, [Binding][]: endpoint2 |


---


### PortClass
A PortClass allows a specific implementation of a port class of a protocol

*This is an advanced feature which is highly target language specific.*

A ProtocolClass may define a port class which can be used to intercept messages within a port either on the sending or on the receiving side. On the sending side the messages are intercepted before they are passed to the message service. On the receiving side they are intercepted before they are passed to the receiving actor.

A PortClass is always either *regular*, i.e. it "lives" within a regular port, or *conjugated*. A PortClass may define local attributes, operations and/or handlers for incoming or outgoing messages. The message direction of handlers in PortClasses always matches the message direction of the ProtocolClass. It is not switched for the *conjugated* PortClass. Whether a message is passed from the message service into the handler and has to be forwarded to the actor or vise versa depends on the type of the PortClass (*regular*, *conjugated*) and the message direction in the ProtocolClass.

Note that the separate instances cannot directly access resources of other instances. This is also true for the regular and conjugated instances on the same binding. To ensure correct synchronization, regular and conjugated instance must only communicate using messages.

* **Port operations** defined on port classes may be called by an actor using the port. They can provide access to protocol functionalities which should be handled synchronously (i.e. a simple function call).
* **Handlers** *intercept* the message of the specified type. A handler may be defined on either side of the protocol (regular or conjugated). They are a very powerful construct which uses the language specific low level api. A handler can choose freely if a received message should be forwarded (to the actor or the message queue) or if it shall only be handled internally. Additionally, handlers may also generate new messages or change the way data is transported across the binding.
* **Attributes** may store additional data in the port. As the attributes are stored per port instance they scale with the number of connected peers for replicated ports.



| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ProtocolClass][] |

**Example**:

```room
// This example uses the port class interface provided by the c code generator
ProtocolClass PingPong {
	incoming {
		Message ping(uint8)
	}
	outgoing {
		Message pong(uint8)
	}
	regular PortClass {
		handle incoming ping '''
			uint8 msgDataOffset = MEM_CEIL(sizeof(etMessage));
			uint8 transitionData = (uint8)((char*)msg + msgDataOffset);;
			
			// do something here
						
			/* hand over the message to the actor:      */
			(*receiveMessageFunc)(actor, self, msg);
		'''
		handle outgoing pong '''
			uint8 transitionData = data__et;
			
			// do something with the data here
			
			// hand over data to message service
			etPort_sendMessage(self, PingPong_OUT_pong, sizeof(uint8), &transitionData);
		'''
        // NOTE: It is not necessary to define handlers for all messages of the protocol. If no handler is defined the default message handling is applied.
	}
	conjugated PortClass {
		handle incoming ping '''
			// The ping message is outgoing in the conjugated portclass
			uint8 transitionData = data__et;
			
			// do something with the data here
			
			// hand over data to message service if you want
			etPort_sendMessage(self, PingPong_IN_ping, sizeof(uint8), &transitionData);
		'''
		handle outgoing pong '''
			// The pong message is incoming in the conjugated portclass
			uint8 msgDataOffset = MEM_CEIL(sizeof(etMessage));
			uint8 transitionData = (uint8)((char*)msg + msgDataOffset);;
			
			// do something here
			
			/* hand over the message to the actor:      */
			(*receiveMessageFunc)(actor, self, msg);
		'''
	}		
}
```

---


### PrimitiveType
A PrimitiveType is an abstraction of a target language's basic type (e.g. integer or boolean)


| Properties | Values |
| ---------- | ------ |
| targetName | *identifier name* |

| Features |     |
| -------- | --- |
| Is a: | [DataType][] |


**Example**:

The eTrice built-in types can be found in the _org.eclipse.etrice.modellib_ project. In most cases the _Types.room_ is already included:

```room
// Follow import by Open Declaration (F3)
import etrice.api.types.*
```

---


### ProtocolClass
A ProtocolClass defines messages and is the interface specification for a Port

A ProtocolClass provides a reusable interface specification for ports. It defines a set of incoming and outgoing Messages that can be exchanged between two ports.
The exact semantics of a message is defined by the CommunicationType.
Protocol classes have only textual notation.

```room
ProtocolClass SimpleProtocolClass {
	incoming {
		Message msg1(int32}
		Message msg2()
	}
	outgoing {
		Message msg3(DataClass1}
		Message msg4()
	}
}
```


| Features |     |
| -------- | --- |
| Contains: | [CommunicationType][], [Attribute][], [Operation][], [Annotation][], [PortClass][] |
| Uses: | [Inheritance][] |

| Feature Usage |     |
| ------------- | --- |
| Typecasts: | [Port][], [SAP][], [SPP][] |
| Is contained in: | [LogicalModel][] |

**Example**:

```room
import etrice.api.types.*

// eventdriven ProtocolClass (asynchronous message passing, bidirectional)
eventdriven ProtocolClass ProtocolClassEvt {
	// ProtocolClass ProtocolClassEvt { // same like above because eventdriven is default 
	incoming {
		// incoming means incoming for a regular port and outgoing for a conjugated port
		Message message1() // message without data
		Message message2(int32) // message with simple data
		Message message3(DMessageData) // message with complex data (DataClass)

	}
	outgoing {
	// outgoing means outgoing for a regular port and incoming for a conjugated port
		Message message1(int32) // incoming and outgoing Messages can have the same name to enable symmetric protocols
	}
}

// DataClass for sending complex data via message
DataClass DMessageData {
	Attribute SomeData: int16
	Attribute SomeMoreData: int32
}

// datadriven ProtocolClass (asynchronous data flow, unidirectional)
datadriven ProtocolClass ProtocolClassData {
	incoming {
		// incoming means incoming for a regular port and outgoing for a conjugated port
		Message value1(int32) // a datadriven message (signal) always needs data
		Message value2(int16) // datadriven message with simple data
		Message value3(DMessageData) // datadriven message with complex data (DataClass)

	}
	// no outgoing messages for datadriven ports allowed 
}
```

---


### RefinedState
A RefinedState refines a State of one of the Actor's base class state machines

A State can be a plain State or a RefinedState.


| Features |     |
| -------- | --- |
| Is a: | [State][] |
| Uses: | [Inheritance][], [StateMachine][] |



---


### RelayPort
A RelayPort forwards its messages without exposing them to the internal interface of the ActorClass

```room
ActorClass RelayPortExample{
	Interface {
		Port relayPort : PSimpleProtocol
	}
	Structure {
		ActorRef actorRef1 : SimpleActorClass2
		
		// relayPort can be directed to port of an ActorRef
		Binding relayPort and actorRef1.externalPort
	}
	Behavior {
		// relayPort not available !
	}
}
```
![RelayPort](../images/300-RelayPort.png)


| Features |     |
| -------- | --- |
| Is a: | [Port][] |



---


### Replication
Replication is mechanism for multi-instantiation for ActorRefs and Ports

ActorRefs and Ports can be instantiated several times under the same name. The notation is similar to arrays in programming languages.

This possibility provides an elegant way of scaling of your system without redundancy.

```room
ActorRef sensor : Sensor 			// one instance
ActorRef sensor[1] : Sensor			// one instance
ActorRef sensorArray[5] : Sensor	// five instances  
```

Replication can also applied to Ports. One use case is to establish a communication with multiple actors through one port interface.
```room
Port service[5] : TimingService 	// five instances
Port service[*] : TimingService		// automatic, as many as needed
```



| Feature Usage |     |
| ------------- | --- |
| Is used by: | [ActorRef][], [Port][] |


---


### SAP
A Service Access Point is similar to a Port, but uses a LayerConnection for wiring


- An actor class can define a Service Provision Point (SPP) to publish a specific service, defined by a protocol class
- An actor class can define a Service Access Point (SAP) if it needs a service, defined by a protocol class
- For a given actor hierarchy, a LayerConnection defines which SAP will be satisfied by (connected to) which SPP



| Features |     |
| -------- | --- |
| Is of type: | [ProtocolClass][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][] |
| Is edited by: | [GraphicalStructureEditor][] |
| Is used by: | [LayerConnection][]: SAPoint |


---


### SPP
A Service Provision Point is the counterpart of a SAP

- An actor class can define a Service Provision Point (SPP) to publish a specific service, defined by a protocol class
- An actor class can define a Service Access Point (SAP) if it needs a service, defined by a protocol class
- For a given actor hierarchy, a LayerConnection defines which SAP will be satisfied by (connected to) which SPP


| Features |     |
| -------- | --- |
| Is of type: | [ProtocolClass][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][] |
| Is edited by: | [SPPPropertyDialog][] |
| Is used by: | [LayerConnection][]: SPPoint, [ServiceImplementation][] |


---


### ServiceImplementation
The implementation of an Service Provision Point (SPP)



| Features |     |
| -------- | --- |
| Uses: | [SPP][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][] |


---


### SimpleState
A State is a node in the state graph representation of the state machine

A State has optional 'entry' and 'exit' codes. The entry code is executed when the state is entered, the exit code is executed
when it is left. In the case of an data driven (also known as polled) state machine, there also is a 'do' action code.
The do code is executed for the active state in each polling cycle.
A state can have a sub state machine. Starting at the top level state machine the states with their optional sub state machines
form a tree which is called a 'hierarchical state machine'.
A state machine always is in exactly one state which can only be a leaf state, i.e. a state which has no sub state machine.


| Features |     |
| -------- | --- |
| Is a: | [State][] |
| Uses: | [Inheritance][], [StateMachine][] |



---


### State
A State can be a plain State or a RefinedState

A State can be a plain State or a RefinedState.


| Features |     |
| -------- | --- |
| Is a: | [StateGraphNode][] |
| Uses: | [Inheritance][] |

| Feature Usage |     |
| ------------- | --- |
| Inheriting features: | [SimpleState][], [RefinedState][] |


---


### StateGraphNode
A StateGraphNode is an abstract node of the state graph

A StateGraphNode can be a State, a TransitionPoint, a ChoicePoint or an InitialPoint.


| Features |     |
| -------- | --- |
| Uses: | [Inheritance][] |

| Feature Usage |     |
| ------------- | --- |
| Inheriting features: | [State][], [ChoicePoint][], [TrPoint][] |
| Is contained in: | [StateMachine][] |


---


### StateMachine
A StateMachine describes the state based, event driven behavior of an ActorClass

In ROOM each actor class can implement its behavior using a state machine. Events occurring at the end ports of an actor will
be forwarded to and processed by the state machine. Events possibly trigger state transitions.

ROOM state machines are hierarchical finite state machines. That means that each state in the state graph can contain another state graph.
This is possible to arbitrary depth.

A state graph consists of

* states
* transitions
* transition points
* choice points
* initial point

![PingPongReceiverFSM](../images/300-PingPongReceiverFSM.png)



| Features |     |
| -------- | --- |
| Contains: | [StateGraphNode][], [Transition][] |
| Uses: | [Inheritance][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [ActorClass][] |
| Is edited by: | [GraphicalBehaviorEditor][] |
| Is used by: | [SimpleState][], [RefinedState][] |


---


### SubSystemClass
A SubSystem is the topmost building block of the executable part of an system

It represents a class for an logical node in a distributed system. An instantiation translates to an executable application, that runs on a node or process.
A SubSystemClass is the structural starting point of an ROOM application. Thus it declares the topmost actor instances (ActorRefs).


| Features |     |
| -------- | --- |
| Contains: | [ActorRef][], [Binding][], [LayerConnection][], [Annotation][] |

| Feature Usage |     |
| ------------- | --- |
| Typecasts: | [SubSystemRef][] |
| Is contained in: | [LogicalModel][] |


---


### SubSystemRef
A Sub System Reference is an instance of an SubSystemClass

It represent a logical node in the structural view of a distributed system. An instantiation translates to an executable application, that runs on a node or process.
	
To be executable, a SubSystemRef has first to be mapped to a physical node, which defines the executional properties.
A physical node is denoted by a NodeClass and NodeRef in the PhysicalModel. The mapping is defined in the MappingModel.



| Features |     |
| -------- | --- |
| Is of type: | [SubSystemClass][] |

| Feature Usage |     |
| ------------- | --- |
| Is contained in: | [LogicalSystem][] |
| Is used by: | [MappingModel][] |


---


### TrPoint
a TrPoint can be an EntryPoint, an ExitPoint or a TransitionPoint

text


| Features |     |
| -------- | --- |
| Is a: | [StateGraphNode][] |

| Feature Usage |     |
| ------------- | --- |
| Inheriting features: | [TransitionPoint][], [EntryPoint][], [ExitPoint][] |


---


### Transition
A Transition is an edge in the state graph representation of the state machine

A transition connects StateGraphNodes in a state graph. A transition is allowed to connect a state or a transition point with itself.
Transition points can only be targets of transitions originating from the same transition point.
The initial point is the source of exactly one transition. In the textual model it is present only in an implicit way.


| Features |     |
| -------- | --- |
| Uses: | [Inheritance][] |

| Feature Usage |     |
| ------------- | --- |
| Inheriting features: | [InitialTransition][], [ContinuationTransition][], [CPBranchTransition][], [TriggeredTransition][] |
| Is contained in: | [StateMachine][] |


---


### TransitionPoint
a transition point is the starting point of transitions that trigger for any state of this state machine

text


| Features |     |
| -------- | --- |
| Is a: | [TrPoint][] |



---


### TriggeredTransition
a triggered transition is used in event driven state machines to trigger state transitions

text


| Features |     |
| -------- | --- |
| Is a: | [Transition][] |



---


## MappingModel
The MappingModel describes the mapping of elements of the LogicalModel to elements of the PhysicalModel

It enables the complete decoupling of the LogicalModel and the PhysicalModel, thus providing a maximum flexibility and reuse for the models.

The model starts with an import part, where you can import .room and .etphys models. They must contain at least one LogicalSystem and one PhysicalSystem.
A Mapping entry puts both in relation, meaning that all sub systems of the LogicalSystem will be distributed to the nodes of the PhysicalSystem.
This is carried out by a SubSystemMapping, that maps a SubSystemRef (logical node) to a NodeRef (physical node).
In the next step, ThreadMappings provide the same action for the logical and physical threads.

```etmap
MappingModel PingPongMapping {
	import PingPong_Model.LogSys
	import GenericPhysicalModel.PhysSys1

	Mapping LogSys -> PhysSys1 {
		SubSystemMapping subSystemRef -> nodeRef1 {
			ThreadMapping defaultThread -> PhysicalThread1
		}
	}

}
```


| Features |     |
| -------- | --- |
| Uses: | [LogicalSystem][], [SubSystemRef][], [PhysicalModel][] |



---


## PhysicalModel
The PhysicalModel defines the setup of your nodes with their attributes like threads and mode of execution

The model describes the physical view of your system:

```etphys
PhysicalSystem PhysSys1 {
	NodeRef nodeRef1 : NodeClass1
	NodeRef nodeRef2 : NodeClass2
}
```

The central element is a NodeClass, that models the executional aspects of a device (node).
At first, it can be associated with a RuntimeClass, which specifies if your device supports multiple threads.
'priomin' and 'priomax' define the range of priorities, that can be assigned to threads.

```etphys
NodeClass NodeClass1 {
	runtime = RuntimeClass1
	priomin = -10
	priomax = 10

	// Thread definitions ...
}

RuntimeClass RuntimeClass1 {
	model = multiThreaded // or singleThreaded
}
```

A thread has to specify the following properties:

- **execmode**: defines the execution type, see more at ExecutionType
	- blocked: message-driven only, thread wakes up if message arrives and is put to sleep after all action is done
	- polled: data-driven only, thread is executed cyclic. The 'interval' property is mandatory in this case.
	- mixed: combines both execution types

- **msgblocksize**: the size in bytes of a message
- **msgpoolsize**:  the amount of messages, that the thread's message queue can store

:::note
'msgblocksize' and 'msgpoolsize' also apply to the polled execution due the internal implementation via message passing.
 The size of the message queue can be calculated as follows: msgpoolsize * msgblocksize bytes 
:::

```etphys
DefaultThread ThreadMessaging {
	execmode = polled
	prio = 0
	stacksize = 1024
	msgblocksize = 32
	msgpoolsize = 10
}

Thread ThreadPolled {
	execmode = polled
	prio = 0
	interval = 100ms
	stacksize = 1024
	msgblocksize = 32
	msgpoolsize = 10
}
```

![Overview of PhysicalModel](../images/300-PhysicalModelOverview.png)



| Feature Usage |     |
| ------------- | --- |
| Is used by: | [MappingModel][] |


---


[AnnotationType]: #annotationtype
[Annotation]: #annotation
[Inheritance]: #inheritance
[PhysicalModel]: #physicalmodel
[MappingModel]: #mappingmodel
[ConfigModel]: #configmodel
[LogicalModel]: #logicalmodel
[LogicalSystem]: #logicalsystem
[ActorClass]: #actorclass
[SubSystemClass]: #subsystemclass
[StateMachine]: #statemachine
[StateGraphNode]: #stategraphnode
[State]: #state
[SimpleState]: #simplestate
[RefinedState]: #refinedstate
[ChoicePoint]: #choicepoint
[TrPoint]: #trpoint
[TransitionPoint]: #transitionpoint
[EntryPoint]: #entrypoint
[ExitPoint]: #exitpoint
[Transition]: #transition
[InitialTransition]: #initialtransition
[GuardedTransition]: #guardedtransition
[ContinuationTransition]: #continuationtransition
[CPBranchTransition]: #cpbranchtransition
[TriggeredTransition]: #triggeredtransition
[SubSystemRef]: #subsystemref
[Replication]: #replication
[ActorRef]: #actorref
[Binding]: #binding
[LayerConnection]: #layerconnection
[ExecutionType]: #executiontype
[CommunicationType]: #communicationtype
[ProtocolClass]: #protocolclass
[PortClass]: #portclass
[DataType]: #datatype
[PrimitiveType]: #primitivetype
[Enumeration]: #enumeration
[DataClass]: #dataclass
[ExternalType]: #externaltype
[Attribute]: #attribute
[Operation]: #operation
[Port]: #port
[RelayPort]: #relayport
[ExternalEndPort]: #externalendport
[InternalEndPort]: #internalendport
[SAP]: #sap
[ServiceImplementation]: #serviceimplementation
[SPP]: #spp

[GraphicalBehaviorEditor]: model-editors.md#graphicalbehavioreditor
[GraphicalStructureEditor]: model-editors.md#graphicalstructureeditor
[ActorRefPropertyDialog]: model-editors.md#actorrefpropertydialog
[PortPropertyDialog]: model-editors.md#portpropertydialog
[SPPPropertyDialog]: model-editors.md#spppropertydialog
