/*
 * Copyright (c) 2007, 2009, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

/*
 * @(#)RegistryUtils.java	1.1 05/03/17
 */
package com.sun.ts.lib.tests.jbi;

import java.rmi.*;
import java.rmi.registry.*;
import java.rmi.server.*;
import java.util.Arrays;

public class RegistryUtils {

    // These values are hard coded for now but they exist in the ts.jte file.
    // If we need to make them configurable we need to find a way to pass them
    // to the JBI components and make the values available to the life cycle
    // methods, perhaps a shared library would work, or a service unit (although
    // the service unit is after the fact).
    private final static String DEFAULT_HOST = "localhost";
    private final static int    DEFAULT_PORT = 1099;


    /**
     * Returns the remote object bound to the specified name.
     *
     * @param name The name of the remote object to lookup.
     * @returns The Remote object (stub) bound to the specified name.  if
     *          no such name to object mapping exists a NotBoundException
     *          will be thrown.
     * @throws Exception if an error occurs, the client will simply print
     *         the exception name so there is no need to catch the various
     *         child exceptions.
     */
    public static Remote lookup(String hostname, int port, String name) throws Exception {
	String url = "//" + hostname + ":" + port + "/" + name;
	Remote rem = null;
	try {
	    rem = Naming.lookup(url);
	} catch (Exception e) {
	    throw e;
	}
	return rem;
    }
    
    public static Remote lookup(String hostname, String name) throws Exception {
	return lookup(hostname, DEFAULT_PORT, name);
    }

    public static Remote lookup(String name) throws Exception {
	return lookup(DEFAULT_HOST, DEFAULT_PORT, name);
    }

    /**
     * Binds the specified object to the specified registry using the specified
     * name.  If the object name is already listed in the registry this method
     * checks the value of the forceRebind boolean.  If a force rebind is specified
     * then the specified name is rebound to the specified object replacing the
     * previous name-object pair.  If forceRebind is false the method simply returns.
     *
     * @param localRegistry The registry to bind to
     * @param name The name of the remote object
     * @param object The object to bind to the registry
     * @param forceRebind If the object exists in the registry and this parameter
     *        is set, we replace the existing name-object 
     */
    public static void bindObject(Registry localRegistry, String name,
	Remote object, boolean forceRebind)
	throws UnknownHostException, RemoteException, Exception {

	String[] roNames = localRegistry.list();

	for (int i = 0; i < roNames.length; i++) {
	    if (roNames[i].equalsIgnoreCase(name)) {
		if (forceRebind) {
		    System.out.println(name + " Forced To Re-Bind To Registry");
		    localRegistry.rebind(name, object);
		} else {
		    System.out.println(name + " Already Bound To Registry");
		}
		return;
	    }
	}
	localRegistry.bind(name, object);
	System.out.println(name + " Bound To Registry");
    }

    /**
     * Binds the specified object to the specified registry using the specified
     * name.  If the object name is already listed in the registry this method
     * returns without modifying the registry state.
     *
     * @param localRegistry The registry to bind to
     * @param name   The lookup name of the remote object
     * @param object The remote object to bind to the registry
     */
     public static void bindObject(Registry localRegistry, String name, Remote object)
	throws UnknownHostException, RemoteException, Exception {
	bindObject(localRegistry, name, object, false);
    }

    /**
     * Returns the registry running on the specified port.  If no registry is running
     * on the specified port, a new registry is created and returned.
     *
     * @param port RMI registry host
     * @param port RMI registry port
     * @return The registry stub
     */
    public static Registry getRunningRegistry(String host, int port) throws RemoteException {
	Registry registry = LocateRegistry.getRegistry(host, port);
	String[] boundNames = null;
	try {
	    boundNames = registry.list();
	} catch(Exception e) { // no registry, create one
	    //comment this out since we can't be sure that a registry doesn't already exist
            //within this VM.  Once we move to JDK 5, we will be able to create multiple
            //registries within a single VM.  For now, this code assumes the existence
            //of an external registry
            //registry = LocateRegistry.createRegistry(DEFAULT_PORT);
            throw new RemoteException("RegistryUtils - registry.list failed: port = " + port);
	}
	return registry;
    }
    
    /**
     * Returns the registry running on the specified port.  If no registry is running
     * a new one is created.
     *
     * @param port RMI registry port
     * @return The registry stub
     */
    public static Registry getRunningRegistry(int port) throws RemoteException {
	return (getRunningRegistry(DEFAULT_HOST, port));
    }

    /**
     * Returns the registry running on the specified port.  If no registry is running
     * a new one is created.
     *
     * @return The registry stub
     */
    public static Registry getRunningRegistry() throws RemoteException {
	return (getRunningRegistry(DEFAULT_HOST, DEFAULT_PORT));
    }

    /*
     * Simple main that lists the contents of a registry.  Useful for listing
     * remote objects bound to a given registry.
     */
    public static void main(String[] args) throws Exception {
	if (args.length != 2) {
	    System.err.println("Usage: java RegistryUtils host port");
	    System.exit(-1);
	}
	String hostname = args[0];
	int port = Integer.parseInt(args[1]);
	Registry reg = getRunningRegistry(hostname, port);
	String[] names = reg.list();
	System.err.println("Registered Names [" + hostname + ":" + port + "]:");
        if (names.length == 0) {
	    System.err.println("\tNO BOUND OBJECTS");	    
	}
	for (int i = 0; i < names.length; i++) {
	    System.err.println("\tname[" + i + "]: " + names[i]);
	}
    }

} // end class RegistryUtils
