/*
 * Decompiled with CFR 0.152.
 */
package org.glassfish.main.jdke.security;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

public class KeyTool {
    private static final System.Logger LOG = System.getLogger(KeyTool.class.getName());
    private static final long EXEC_TIMEOUT = Integer.getInteger("org.glassfish.main.keytool.timeout", 60).intValue();
    private static final String KEYTOOL;
    private final File keyStore;
    private final String keyStoreType;
    private char[] password;

    public KeyTool(File keyStore, char[] password) {
        this(keyStore, KeyTool.guessKeyStoreType(keyStore), password);
    }

    public KeyTool(File keyStore, String keyStoreType, char[] password) {
        this.keyStore = keyStore;
        this.password = password;
        this.keyStoreType = keyStoreType;
    }

    public KeyStore loadKeyStore() throws IOException {
        try {
            return KeyStore.getInstance(this.keyStore, this.password);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new IOException("Could not load keystore: " + String.valueOf(this.keyStore), e);
        }
    }

    public void generateKeyPair(String alias, String dn, String keyAlgorithm, int certValidity) throws IOException {
        List<String> command = List.of(KEYTOOL, "-J-Duser.language=en", "-noprompt", "-genkeypair", "-alias", alias, "-dname", dn, "-keyalg", keyAlgorithm, "-validity", Integer.toString(certValidity), "-keystore", this.keyStore.getAbsolutePath(), "-storetype", this.keyStoreType);
        if (this.keyStore.getParentFile().mkdirs()) {
            LOG.log(System.Logger.Level.DEBUG, "Created directory for keystore: {0}", this.keyStore.getParentFile());
        }
        this.execute(command, this.password, this.password, this.password, this.password);
    }

    public void copyCertificate(String alias, File destKeyStore) throws IOException {
        this.copyCertificate(alias, destKeyStore, this.password);
    }

    public void copyCertificate(String alias, File destKeyStoreFile, char[] destKeyStorePassword) throws IOException {
        Certificate certificate;
        KeyStore ks = this.loadKeyStore();
        try {
            certificate = ks.getCertificate(alias);
        }
        catch (KeyStoreException e) {
            throw new IOException("Failed to get the certificate under alias " + alias + " from the keystore " + String.valueOf(this.keyStore));
        }
        if (certificate == null) {
            throw new IOException("Alias " + alias + " not found in the key store: " + String.valueOf(this.keyStore));
        }
        try {
            KeyTool destKeyStoreTool = destKeyStoreFile.exists() ? new KeyTool(destKeyStoreFile, destKeyStorePassword) : KeyTool.createEmptyKeyStore(destKeyStoreFile, destKeyStorePassword);
            KeyStore destKeyStore = destKeyStoreTool.loadKeyStore();
            destKeyStore.setCertificateEntry(alias, certificate);
            try (FileOutputStream output = new FileOutputStream(destKeyStoreFile);){
                destKeyStore.store(output, destKeyStorePassword);
            }
        }
        catch (GeneralSecurityException e) {
            throw new IOException("Could not copy certificate with alias: " + alias + " to key store: " + String.valueOf(destKeyStoreFile), e);
        }
    }

    public void exportCertificate(String alias, File outputFile) throws IOException {
        List<String> exportCommand = List.of(KEYTOOL, "-J-Duser.language=en", "-noprompt", "-exportcert", "-alias", alias, "-keystore", this.keyStore.getAbsolutePath(), "-file", outputFile.getAbsolutePath());
        this.execute(exportCommand, new char[][]{this.password});
    }

    public void changeKeyStorePassword(char[] newPassword) throws IOException {
        ArrayList<String> aliases;
        KeyStore ks = this.loadKeyStore();
        char[] oldPassword = this.password;
        this.password = newPassword;
        try {
            aliases = Collections.list(ks.aliases());
        }
        catch (KeyStoreException e) {
            throw new IOException("Could not list aliases in keystore: " + String.valueOf(this.keyStore), e);
        }
        for (String alias : aliases) {
            try {
                if (!ks.isKeyEntry(alias)) continue;
                KeyTool.changeKeyPassword(ks, alias, oldPassword, newPassword);
            }
            catch (IOException | KeyStoreException e) {
                LOG.log(System.Logger.Level.WARNING, "Could not change key password for alias: {0}, it may use different password.", alias);
            }
        }
        try (FileOutputStream output = new FileOutputStream(this.keyStore);){
            ks.store(output, this.password);
        }
        catch (GeneralSecurityException e) {
            throw new IOException("Keystore password successfuly changed, however failed changing key passwords: " + String.valueOf(this.keyStore), e);
        }
    }

    public void changeKeyPassword(String alias, char[] oldPassword, char[] newPassword) throws IOException {
        try {
            KeyStore sourceStore = this.loadKeyStore();
            Certificate[] chain = sourceStore.getCertificateChain(alias);
            PrivateKey key = (PrivateKey)sourceStore.getKey(alias, oldPassword);
            sourceStore.setKeyEntry(alias, key, newPassword, chain);
            try (FileOutputStream output = new FileOutputStream(this.keyStore);){
                sourceStore.store(output, this.password);
            }
        }
        catch (GeneralSecurityException e) {
            throw new IOException("Could not change key password for alias: " + alias, e);
        }
    }

    private void execute(List<String> command, char[] ... stdinLines) throws IOException {
        KeyTool.execute(this.keyStore, command, stdinLines);
    }

    public static KeyTool createEmptyKeyStore(File file, char[] password) throws IOException {
        return KeyTool.createEmptyKeyStore(file, KeyTool.guessKeyStoreType(file), password);
    }

    public static KeyTool createEmptyKeyStore(File file, String keyStoreType, char[] password) throws IOException {
        if (file == null || password == null) {
            throw new IllegalArgumentException("Key store file and password must not be null (usually must have at least 6 characters, depends on keystore implementation).");
        }
        try {
            KeyStore cacerts = KeyStore.getInstance(keyStoreType);
            cacerts.load(null, password);
            try (FileOutputStream output = new FileOutputStream(file);){
                cacerts.store(output, password);
            }
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IOException("Could not create new keystore: " + String.valueOf(file), e);
        }
        return new KeyTool(file, password);
    }

    private static String guessKeyStoreType(File keyStore) {
        String suffix;
        String filename = keyStore.getName();
        int lastDot = filename.lastIndexOf(46);
        if (lastDot < 0) {
            throw new IllegalArgumentException("Key store file name must have an extension to guess the key store type: " + String.valueOf(keyStore));
        }
        switch (suffix = filename.substring(lastDot + 1).toUpperCase()) {
            case "JKS": {
                return "JKS";
            }
            case "P12": 
            case "PFX": {
                return "PKCS12";
            }
            case "JCEKS": {
                return "JCEKS";
            }
        }
        LOG.log(System.Logger.Level.WARNING, "Unknown key store type for file {0}, using its suffix as a keystore type.", keyStore);
        return suffix;
    }

    private static void changeKeyPassword(KeyStore keyStore, String alias, char[] oldPassword, char[] newPassword) throws IOException {
        try {
            Certificate[] chain = keyStore.getCertificateChain(alias);
            PrivateKey key = (PrivateKey)keyStore.getKey(alias, oldPassword);
            keyStore.setKeyEntry(alias, key, newPassword, chain);
        }
        catch (GeneralSecurityException e) {
            throw new IOException("Could not change key password for alias: " + alias, e);
        }
    }

    private static void execute(File keyStore, List<String> command, char[] ... stdinLines) throws IOException {
        Process process;
        LOG.log(System.Logger.Level.INFO, () -> "Executing command: " + command.stream().collect(Collectors.joining(" ")));
        ProcessBuilder builder = new ProcessBuilder(command).directory(keyStore.getParentFile());
        try {
            process = builder.start();
        }
        catch (IOException e) {
            throw new IOException("Could not execute command: " + String.valueOf(builder.command()), e);
        }
        try (OutputStreamWriter stdin = new OutputStreamWriter(process.getOutputStream(), Charset.defaultCharset());){
            if (stdinLines != null && stdinLines.length > 0) {
                KeyTool.writeStdIn(stdinLines, stdin);
            }
            if (!process.waitFor(EXEC_TIMEOUT, TimeUnit.SECONDS)) {
                throw new IOException("KeyTool command timed out after " + EXEC_TIMEOUT + " seconds. Output: " + KeyTool.getOutput(process));
            }
            int exitCode = process.exitValue();
            String output = KeyTool.getOutput(process);
            LOG.log(System.Logger.Level.DEBUG, () -> "Command output: " + output);
            if (exitCode != 0) {
                throw new IOException("KeyTool command failed with exit code: " + exitCode + " and output: " + output);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted", e);
        }
    }

    private static String getOutput(Process process) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        process.getInputStream().transferTo(output);
        process.getErrorStream().transferTo(output);
        return output.toString(Charset.defaultCharset());
    }

    private static void writeStdIn(char[][] stdinLines, Writer stdin) throws IOException {
        for (char[] line : stdinLines) {
            KeyTool.writeLine(line, stdin);
        }
    }

    private static void writeLine(char[] content, Writer stdin) throws IOException {
        stdin.write(content);
        stdin.write(System.lineSeparator());
        stdin.flush();
    }

    static {
        String javaHome = System.getProperty("java.home");
        if (javaHome == null || javaHome.isEmpty()) {
            throw new IllegalStateException("java.home system property is not set. Cannot locate keytool.");
        }
        String keyToolFileName = System.getProperty("os.name").toLowerCase().contains("windows") ? "keytool.exe" : "keytool";
        KEYTOOL = new File(new File(javaHome, "bin"), keyToolFileName).getAbsolutePath();
    }
}

