/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.linear;

import java.util.Arrays;
import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.BlockRealMatrix;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SingularMatrixException;
import org.apache.commons.math3.util.FastMath;

public class QRDecomposition {
    private double[][] qrt;
    private double[] rDiag;
    private RealMatrix cachedQ;
    private RealMatrix cachedQT;
    private RealMatrix cachedR;
    private RealMatrix cachedH;
    private final double threshold;

    public QRDecomposition(RealMatrix matrix) {
        this(matrix, 0.0);
    }

    public QRDecomposition(RealMatrix matrix, double threshold) {
        this.threshold = threshold;
        int m = matrix.getRowDimension();
        int n = matrix.getColumnDimension();
        this.qrt = matrix.transpose().getData();
        this.rDiag = new double[FastMath.min(m, n)];
        this.cachedQ = null;
        this.cachedQT = null;
        this.cachedR = null;
        this.cachedH = null;
        this.decompose(this.qrt);
    }

    protected void decompose(double[][] matrix) {
        int minor = 0;
        while (minor < FastMath.min(matrix.length, matrix[0].length)) {
            this.performHouseholderReflection(minor, matrix);
            ++minor;
        }
    }

    protected void performHouseholderReflection(int minor, double[][] matrix) {
        double a;
        double[] qrtMinor = matrix[minor];
        double xNormSqr = 0.0;
        int row = minor;
        while (row < qrtMinor.length) {
            double c = qrtMinor[row];
            xNormSqr += c * c;
            ++row;
        }
        this.rDiag[minor] = a = qrtMinor[minor] > 0.0 ? -FastMath.sqrt(xNormSqr) : FastMath.sqrt(xNormSqr);
        if (a != 0.0) {
            int n = minor;
            qrtMinor[n] = qrtMinor[n] - a;
            int col = minor + 1;
            while (col < matrix.length) {
                double[] qrtCol = matrix[col];
                double alpha = 0.0;
                int row2 = minor;
                while (row2 < qrtCol.length) {
                    alpha -= qrtCol[row2] * qrtMinor[row2];
                    ++row2;
                }
                alpha /= a * qrtMinor[minor];
                row2 = minor;
                while (row2 < qrtCol.length) {
                    int n2 = row2;
                    qrtCol[n2] = qrtCol[n2] - alpha * qrtMinor[row2];
                    ++row2;
                }
                ++col;
            }
        }
    }

    public RealMatrix getR() {
        if (this.cachedR == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            double[][] ra = new double[m][n];
            int row = FastMath.min(m, n) - 1;
            while (row >= 0) {
                ra[row][row] = this.rDiag[row];
                int col = row + 1;
                while (col < n) {
                    ra[row][col] = this.qrt[col][row];
                    ++col;
                }
                --row;
            }
            this.cachedR = MatrixUtils.createRealMatrix(ra);
        }
        return this.cachedR;
    }

    public RealMatrix getQ() {
        if (this.cachedQ == null) {
            this.cachedQ = this.getQT().transpose();
        }
        return this.cachedQ;
    }

    public RealMatrix getQT() {
        if (this.cachedQT == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            double[][] qta = new double[m][m];
            int minor = m - 1;
            while (minor >= FastMath.min(m, n)) {
                qta[minor][minor] = 1.0;
                --minor;
            }
            minor = FastMath.min(m, n) - 1;
            while (minor >= 0) {
                double[] qrtMinor = this.qrt[minor];
                qta[minor][minor] = 1.0;
                if (qrtMinor[minor] != 0.0) {
                    int col = minor;
                    while (col < m) {
                        double alpha = 0.0;
                        int row = minor;
                        while (row < m) {
                            alpha -= qta[col][row] * qrtMinor[row];
                            ++row;
                        }
                        alpha /= this.rDiag[minor] * qrtMinor[minor];
                        row = minor;
                        while (row < m) {
                            double[] dArray = qta[col];
                            int n2 = row;
                            dArray[n2] = dArray[n2] + -alpha * qrtMinor[row];
                            ++row;
                        }
                        ++col;
                    }
                }
                --minor;
            }
            this.cachedQT = MatrixUtils.createRealMatrix(qta);
        }
        return this.cachedQT;
    }

    public RealMatrix getH() {
        if (this.cachedH == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            double[][] ha = new double[m][n];
            int i = 0;
            while (i < m) {
                int j = 0;
                while (j < FastMath.min(i + 1, n)) {
                    ha[i][j] = this.qrt[j][i] / -this.rDiag[j];
                    ++j;
                }
                ++i;
            }
            this.cachedH = MatrixUtils.createRealMatrix(ha);
        }
        return this.cachedH;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.qrt, this.rDiag, this.threshold);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] qrt;
        private final double[] rDiag;
        private final double threshold;

        private Solver(double[][] qrt, double[] rDiag, double threshold) {
            this.qrt = qrt;
            this.rDiag = rDiag;
            this.threshold = threshold;
        }

        @Override
        public boolean isNonSingular() {
            double[] dArray = this.rDiag;
            int n = this.rDiag.length;
            int n2 = 0;
            while (n2 < n) {
                double diag = dArray[n2];
                if (FastMath.abs(diag) <= this.threshold) {
                    return false;
                }
                ++n2;
            }
            return true;
        }

        @Override
        public RealVector solve(RealVector b) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b.getDimension() != m) {
                throw new DimensionMismatchException(b.getDimension(), m);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            double[] x = new double[n];
            double[] y = b.toArray();
            int minor = 0;
            while (minor < FastMath.min(m, n)) {
                double[] qrtMinor = this.qrt[minor];
                double dotProduct = 0.0;
                int row = minor;
                while (row < m) {
                    dotProduct += y[row] * qrtMinor[row];
                    ++row;
                }
                dotProduct /= this.rDiag[minor] * qrtMinor[minor];
                row = minor;
                while (row < m) {
                    int n2 = row;
                    y[n2] = y[n2] + dotProduct * qrtMinor[row];
                    ++row;
                }
                ++minor;
            }
            int row = this.rDiag.length - 1;
            while (row >= 0) {
                int n3 = row;
                y[n3] = y[n3] / this.rDiag[row];
                double yRow = y[row];
                double[] qrtRow = this.qrt[row];
                x[row] = yRow;
                int i = 0;
                while (i < row) {
                    int n4 = i;
                    y[n4] = y[n4] - yRow * qrtRow[i];
                    ++i;
                }
                --row;
            }
            return new ArrayRealVector(x, false);
        }

        @Override
        public RealMatrix solve(RealMatrix b) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b.getRowDimension() != m) {
                throw new DimensionMismatchException(b.getRowDimension(), m);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int columns = b.getColumnDimension();
            int cBlocks = (columns + 52 - 1) / 52;
            double[][] xBlocks = BlockRealMatrix.createBlocksLayout(n, columns);
            double[][] y = new double[b.getRowDimension()][52];
            double[] alpha = new double[52];
            int kBlock = 0;
            while (kBlock < cBlocks) {
                int kStart = kBlock * 52;
                int kEnd = FastMath.min(kStart + 52, columns);
                int kWidth = kEnd - kStart;
                b.copySubMatrix(0, m - 1, kStart, kEnd - 1, y);
                int minor = 0;
                while (minor < FastMath.min(m, n)) {
                    int k;
                    double[] yRow;
                    double d;
                    double[] qrtMinor = this.qrt[minor];
                    double factor = 1.0 / (this.rDiag[minor] * qrtMinor[minor]);
                    Arrays.fill(alpha, 0, kWidth, 0.0);
                    int row = minor;
                    while (row < m) {
                        d = qrtMinor[row];
                        yRow = y[row];
                        k = 0;
                        while (k < kWidth) {
                            int n2 = k;
                            alpha[n2] = alpha[n2] + d * yRow[k];
                            ++k;
                        }
                        ++row;
                    }
                    int k2 = 0;
                    while (k2 < kWidth) {
                        int n3 = k2++;
                        alpha[n3] = alpha[n3] * factor;
                    }
                    row = minor;
                    while (row < m) {
                        d = qrtMinor[row];
                        yRow = y[row];
                        k = 0;
                        while (k < kWidth) {
                            int n4 = k;
                            yRow[n4] = yRow[n4] + alpha[k] * d;
                            ++k;
                        }
                        ++row;
                    }
                    ++minor;
                }
                int j = this.rDiag.length - 1;
                while (j >= 0) {
                    int jBlock = j / 52;
                    int jStart = jBlock * 52;
                    double factor = 1.0 / this.rDiag[j];
                    double[] yJ = y[j];
                    double[] xBlock = xBlocks[jBlock * cBlocks + kBlock];
                    int index = (j - jStart) * kWidth;
                    int k = 0;
                    while (k < kWidth) {
                        int n5 = k;
                        yJ[n5] = yJ[n5] * factor;
                        xBlock[index++] = yJ[k];
                        ++k;
                    }
                    double[] qrtJ = this.qrt[j];
                    int i = 0;
                    while (i < j) {
                        double rIJ = qrtJ[i];
                        double[] yI = y[i];
                        int k3 = 0;
                        while (k3 < kWidth) {
                            int n6 = k3;
                            yI[n6] = yI[n6] - yJ[k3] * rIJ;
                            ++k3;
                        }
                        ++i;
                    }
                    --j;
                }
                ++kBlock;
            }
            return new BlockRealMatrix(n, columns, xBlocks, false);
        }

        @Override
        public RealMatrix getInverse() {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.qrt[0].length));
        }
    }
}

