/***********************************************************************
 * Copyright (c) 2008 Actuate Corporation.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0/.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 *
 * Contributors:
 * Actuate Corporation - initial API and implementation
 ***********************************************************************/

package org.eclipse.birt.report.engine.layout.pdf.emitter;

import java.util.Iterator;
import java.util.Stack;

import org.eclipse.birt.core.exception.BirtException;
import org.eclipse.birt.core.format.NumberFormatter;
import org.eclipse.birt.report.engine.content.Dimension;
import org.eclipse.birt.report.engine.content.IAutoTextContent;
import org.eclipse.birt.report.engine.content.IBandContent;
import org.eclipse.birt.report.engine.content.ICellContent;
import org.eclipse.birt.report.engine.content.IContainerContent;
import org.eclipse.birt.report.engine.content.IContent;
import org.eclipse.birt.report.engine.content.IForeignContent;
import org.eclipse.birt.report.engine.content.IListBandContent;
import org.eclipse.birt.report.engine.content.IListGroupContent;
import org.eclipse.birt.report.engine.content.IPageContent;
import org.eclipse.birt.report.engine.content.IReportContent;
import org.eclipse.birt.report.engine.content.IRowContent;
import org.eclipse.birt.report.engine.content.ITableBandContent;
import org.eclipse.birt.report.engine.content.ITableGroupContent;
import org.eclipse.birt.report.engine.css.engine.value.DataFormatValue;
import org.eclipse.birt.report.engine.css.engine.value.css.CSSConstants;
import org.eclipse.birt.report.engine.emitter.ContentEmitterUtil;
import org.eclipse.birt.report.engine.emitter.IContentEmitter;
import org.eclipse.birt.report.engine.emitter.IEmitterServices;
import org.eclipse.birt.report.engine.executor.IReportExecutor;
import org.eclipse.birt.report.engine.ir.MasterPageDesign;
import org.eclipse.birt.report.engine.ir.SimpleMasterPageDesign;
import org.eclipse.birt.report.engine.layout.ILayoutPageHandler;
import org.eclipse.birt.report.engine.layout.PDFConstants;
import org.eclipse.birt.report.engine.layout.area.IContainerArea;
import org.eclipse.birt.report.engine.layout.area.impl.AbstractArea;
import org.eclipse.birt.report.engine.layout.area.impl.AreaFactory;
import org.eclipse.birt.report.engine.layout.area.impl.ContainerArea;
import org.eclipse.birt.report.engine.layout.area.impl.PageArea;
import org.eclipse.birt.report.engine.layout.pdf.text.Chunk;
import org.eclipse.birt.report.engine.layout.pdf.text.ChunkGenerator;
import org.eclipse.birt.report.engine.layout.pdf.util.HTML2Content;
import org.eclipse.birt.report.engine.layout.pdf.util.PropertyUtil;
import org.eclipse.birt.report.engine.util.BidiAlignmentResolver;

import com.ibm.icu.util.ULocale;

public class PDFLayoutEmitter extends LayoutEmitterAdapter implements IContentEmitter {
	IContentEmitter emitter;
	Stack layoutStack = new Stack();
	ContainerLayout current;
	LayoutContextFactory factory;
	LayoutEngineContext context;

	/**
	 * identify if current area is the first area generated by content
	 */
	protected boolean isFirst = true;

	protected ILayoutPageHandler pageHandler;

	public PDFLayoutEmitter(IReportExecutor executor, IContentEmitter emitter, LayoutEngineContext context) {
		this.emitter = emitter;
		this.context = context;
		factory = new LayoutContextFactory(executor, context);
	}

	public PDFLayoutEmitter(LayoutEngineContext context) {
		this.context = context;
		factory = new LayoutContextFactory(null, context);
	}

	@Override
	public void initialize(IEmitterServices service) throws BirtException {
		emitter.initialize(service);
	}

	@Override
	public String getOutputFormat() {
		return emitter.getOutputFormat();
	}

	@Override
	public void start(IReportContent report) throws BirtException {
		emitter.start(report);
		context.setReport(report);
	}

	@Override
	public void end(IReportContent report) throws BirtException {
		resolveTotalPage(emitter);
		if (pageHandler != null) {
			pageHandler.onPage(context.pageNumber, context);
		}
		emitter.end(report);
	}

	@Override
	public ILayoutPageHandler getPageHandler() {
		return pageHandler;
	}

	@Override
	public void setPageHandler(ILayoutPageHandler pageHandler) {
		this.pageHandler = pageHandler;
	}

	protected void resolveTotalPage(IContentEmitter emitter) throws BirtException {
		IContent con = context.getUnresolvedContent();
		if (!(con instanceof IAutoTextContent)) {
			return;
		}

		IAutoTextContent totalPageContent = (IAutoTextContent) con;
		if (null != totalPageContent) {
			DataFormatValue format = totalPageContent.getComputedStyle().getDataFormat();
			NumberFormatter nf = null;
			if (format == null) {
				nf = new NumberFormatter();
			} else {
				String pattern = format.getNumberPattern();
				String locale = format.getNumberLocale();
				if (locale == null) {
					nf = new NumberFormatter(pattern);
				} else {
					nf = new NumberFormatter(pattern, new ULocale(locale));
				}
			}

			long totalPageCount = 0;
			if (context.autoPageBreak) {
				totalPageCount = context.pageCount;
			} else {
				// FIXME: handle the TOTAL_PAGE and UNFILTERED_TOTAL_PAGE
				// differently.
				totalPageCount = context.totalPage > 0 ? context.totalPage : context.pageCount;
			}
			totalPageContent.setText(nf.format(totalPageCount));

			AbstractArea totalPageArea = null;
			ChunkGenerator cg = new ChunkGenerator(context.getFontManager(), totalPageContent, true, true);
			if (cg.hasMore()) {
				Chunk c = cg.getNext();
				Dimension d = new Dimension(
						(int) (c.getFontInfo().getWordWidth(c.getText()) * PDFConstants.LAYOUT_TO_PDF_RATIO),
						(int) (c.getFontInfo().getWordHeight() * PDFConstants.LAYOUT_TO_PDF_RATIO));
				totalPageArea = (AbstractArea) AreaFactory.createTextArea(totalPageContent, c.getFontInfo(), false);
				totalPageArea.setWidth(Math.min(context.getMaxWidth(), d.getWidth()));
				totalPageArea.setHeight(Math.min(context.getMaxHeight(), d.getHeight()));
			}

			String align = totalPageContent.getComputedStyle().getTextAlign();
			// bidi_hcg: handle empty or justify align in RTL direction as right
			// alignment
			boolean isRightAligned = BidiAlignmentResolver.isRightAligned(totalPageContent, align, false);
			if ((isRightAligned || CSSConstants.CSS_CENTER_VALUE.equalsIgnoreCase(align))) {
				int spacing = context.getTotalPageTemplateWidth() - totalPageArea.getWidth();
				if (spacing > 0) {
					if (isRightAligned) {
						totalPageArea.setAllocatedPosition(spacing + totalPageArea.getAllocatedX(),
								totalPageArea.getAllocatedY());
					} else if (CSSConstants.CSS_CENTER_VALUE.equalsIgnoreCase(align)) {
						totalPageArea.setAllocatedPosition(spacing / 2 + totalPageArea.getAllocatedX(),
								totalPageArea.getAllocatedY());
					}
				}
			}

			totalPageContent.setExtension(IContent.LAYOUT_EXTENSION, totalPageArea);
			emitter.startAutoText(totalPageContent);
		}
	}

	@Override
	public void startContainer(IContainerContent container) throws BirtException {
		_startContainer(container);
	}

	public void _startContainer(IContent container) throws BirtException {
		boolean isInline = PropertyUtil.isInlineElement(container);
		Layout layout;
		if (isInline) {
			if (current instanceof InlineStackingLayout) {
				// layout = factory.createLayoutManager( current, container );
			} else {
				Layout lineLayout = factory.createLayoutManager(current, null);
				lineLayout.initialize();
				current = (ContainerLayout) lineLayout;
			}
		} else if (current instanceof InlineStackingLayout) {
			while (current instanceof InlineStackingLayout) {
				current.closeLayout();
				current = current.getParent();
			}
		}
		layout = factory.createLayoutManager(current, container);
		if (layout != null) {
			current = (ContainerLayout) layout;
			layout.initialize();
		}
	}

	@Override
	public void endContainer(IContainerContent container) throws BirtException {
		_endContainer(container);
	}

	private void _endContainer(IContent container) throws BirtException {
		boolean isInline = PropertyUtil.isInlineElement(container);
		if (!isInline) {
			while (current instanceof InlineStackingLayout) {
				current.closeLayout();
				current = current.getParent();
			}
		}
		current.closeLayout();
		current = current.getParent();
	}

	@Override
	public void startContent(IContent content) throws BirtException {
		boolean isInline = PropertyUtil.isInlineElement(content);
		Layout layout;
		if (isInline) {
			if (current instanceof InlineStackingLayout) {
				// layout = factory.createLayoutManager( current, content );
			} else {
				ContainerLayout lineLayout = (ContainerLayout) factory.createLayoutManager(current, null);
				lineLayout.initialize();
				current = lineLayout;
			}
		} else {
			while (current instanceof InlineStackingLayout) {
				current.closeLayout();
				current = current.getParent();
			}
		}
		layout = factory.createLayoutManager(current, content);
		if (layout != null) {
			layout.initialize();
			layout.layout();
			layout.closeLayout();
		}
	}

	@Override
	public void endContent(IContent content) {
		// do nothing;
	}

	@Override
	public void startListBand(IListBandContent listBand) {
		if (current instanceof RepeatableLayout) {
			((RepeatableLayout) current).bandStatus = listBand.getBandType();
		}
	}

	@Override
	public void startListGroup(IListGroupContent listGroup) throws BirtException {
		if (current instanceof RepeatableLayout) {
			((RepeatableLayout) current).bandStatus = IBandContent.BAND_DETAIL;
		}
		super.startListGroup(listGroup);
	}

	@Override
	public void endListBand(IListBandContent listBand) {

	}

	@Override
	public void startPage(IPageContent page) throws BirtException {
		if (!context.autoPageBreak) {
			long number = page.getPageNumber();
			if (number > 0) {
				context.pageNumber = number;
			}
		}
		super.startPage(page);
	}

	boolean showPageFooter(SimpleMasterPageDesign masterPage, IPageContent page) {
		boolean showFooter = true;
		if (!masterPage.isShowFooterOnLast()) {
			if (page.getPageNumber() == context.totalPage) {
				showFooter = false;
			}
		}
		return showFooter;
	}

	@Override
	public void outputPage(IPageContent page) throws BirtException {
		MasterPageDesign mp = (MasterPageDesign) page.getGenerateBy();

		if (mp instanceof SimpleMasterPageDesign) {
			Object obj = page.getExtension(IContent.LAYOUT_EXTENSION);

			if (obj instanceof PageArea) {
				PageArea pageArea = (PageArea) obj;

				if (isFirst && !((SimpleMasterPageDesign) mp).isShowHeaderOnFirst()) {
					pageArea.removeHeader();
					isFirst = false;
				}
				if (!showPageFooter((SimpleMasterPageDesign) mp, page)) {
					pageArea.removeFooter();
				}

				if (((SimpleMasterPageDesign) mp).isFloatingFooter()) {
					ContainerArea footer = (ContainerArea) page.getFooter();
					IContainerArea body = pageArea.getBody();
					IContainerArea header = pageArea.getHeader();
					if (footer != null) {
						footer.setPosition(footer.getX(),
								(header == null ? 0 : header.getHeight()) + (body == null ? 0 : body.getHeight()));
					}
				}
			}
		}
		emitter.startPage(page);
		emitter.endPage(page);
		context.pageCount++;
	}

	protected void startTableContainer(IContainerContent container) throws BirtException {
		ContainerLayout layout = (ContainerLayout) factory.createLayoutManager(current, container);
		current = layout;
		current.initialize();
	}

	protected void endTableContainer(IContainerContent container) throws BirtException {
		current.closeLayout();
		current = current.getParent();
	}

	@Override
	public void startRow(IRowContent row) throws BirtException {
		startTableContainer(row);
	}

	@Override
	public void endRow(IRowContent row) throws BirtException {
		endTableContainer(row);
	}

	@Override
	public void startTableBand(ITableBandContent band) throws BirtException {
		if (current instanceof RepeatableLayout) {
			((RepeatableLayout) current).bandStatus = band.getBandType();
		}
		startTableContainer(band);
	}

	@Override
	public void startTableGroup(ITableGroupContent group) throws BirtException {
		if (current instanceof RepeatableLayout) {
			((RepeatableLayout) current).bandStatus = IBandContent.BAND_DETAIL;
		}
		startTableContainer(group);
	}

	@Override
	public void endTableBand(ITableBandContent band) throws BirtException {
		endTableContainer(band);
	}

	@Override
	public void endTableGroup(ITableGroupContent group) throws BirtException {
		endTableContainer(group);
	}

	@Override
	public void startCell(ICellContent cell) throws BirtException {
		startTableContainer(cell);
	}

	@Override
	public void endCell(ICellContent cell) throws BirtException {
		endContainer(cell);
	}

	protected void visitContent(IContent content, IContentEmitter emitter) throws BirtException {
		ContentEmitterUtil.startContent(content, emitter);
		java.util.Collection children = content.getChildren();
		if (children != null && !children.isEmpty()) {
			Iterator iter = children.iterator();
			while (iter.hasNext()) {
				IContent child = (IContent) iter.next();
				visitContent(child, emitter);
			}
		}
		ContentEmitterUtil.endContent(content, emitter);
	}

	@Override
	public void startForeign(IForeignContent foreign) throws BirtException {
		if (IForeignContent.HTML_TYPE.equals(foreign.getRawType())) {
			_startContainer(foreign);
			// build content DOM tree for HTML text
			HTML2Content.html2Content(foreign);
			java.util.Collection children = foreign.getChildren();
			if (children != null && !children.isEmpty()) {
				Iterator iter = children.iterator();
				IContent child = (IContent) iter.next();
				visitContent(child, this);
			}
			// FIXME
			foreign.getChildren().clear();
			_endContainer(foreign);
		} else {
			startContent(foreign);
		}
	}

}
