/*
** Yet Another Eval 1.0
** By Peter Gordon (pete@petergordon.org.uk)
**
** A bit like the standard eval command but with some extra features:
**
** - Operator precedence
** - A floating point mode with trig functions
** - 64bit integer size, or double float precision
** - Integers can be specified in decimal, binary or hex
**
** Its not a direct replacement for eval, as it doesn't (currently)
** have LFORMAT support or a directly corresponding argument format.
**
** If you improve it, please email me your changes so they can be
** merged with the official version.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>

#include <proto/exec.h>
#include <proto/dos.h>

static const TEXT __attribute((used)) __version[] = "$VER: yae 1.0 (02.04.2008)";

#if defined(__AROS__)
  typedef UWORD uint16;
  typedef LONG  int32;
  typedef ULONG uint32;
  typedef QUAD  int64;

// warning: this works only because no tags are specified in this application
# define AllocVecTags(size, tag) AllocVec(size, MEMF_ANY)
#else
  // integer type which is large enough to store a pointer
  typedef ULONG IPTR;
#endif

/*** Arguments ***/
#define ARG_EXP 0
#define ARG_FLOAT 1
#define ARG_HEX 2

TEXT *argfmt = "EXPRESSION/A,FLOAT/S,HEX/S";
APTR rargs[] = { NULL, NULL, NULL };
APTR argh = NULL;

// Environment variables are read into this
TEXT varbuf[1024];

// Structure to hold information about operator tokens
struct oper
{
  TEXT *token;
  uint32 toklen;
  uint16 action;
  uint16 precedence;
};

// Unary operators
enum
{
  UO_NOT = 1,
  UO_ASIN,
  UO_ACOS,
  UO_ATAN,
  UO_SIN,
  UO_COS,
  UO_TAN,
  UO_NEG,
  UO_ABS,
  UO_SQRT,
  UO_SINH,
  UO_COSH,
  UO_TANH,
  UO_LOG,
  UO_LOG10,
  UO_EXP,
  UO_CEIL,
  UO_FLOOR,
  UO_TRUNC,
  UO_ROUND
};

// Must be length-sorted
struct oper unops[] = { { "log10", 5, UO_LOG10, 2 },
                        { "floor", 5, UO_FLOOR, 2 },
                        { "trunc", 5, UO_TRUNC, 2 },
                        { "round", 5, UO_ROUND, 2 },
                        { "ceil",  4, UO_CEIL,  2 },
                        { "asin",  4, UO_ASIN,  2 },
                        { "acos",  4, UO_ACOS,  2 },
                        { "atan",  4, UO_ATAN,  2 },
                        { "sqrt",  4, UO_SQRT,  2 },
                        { "sinh",  4, UO_SINH,  2 },
                        { "cosh",  4, UO_COSH,  2 },
                        { "tanh",  4, UO_TANH,  2 },
                        { "not",   3, UO_NOT,   2 },
                        { "sin",   3, UO_SIN,   2 },
                        { "cos",   3, UO_COS,   2 },
                        { "tan",   3, UO_TAN,   2 },
                        { "abs",   3, UO_ABS,   2 },
                        { "log",   3, UO_LOG,   2 },
                        { "exp",   3, UO_EXP,   2 },
                        { "-",     1, UO_NEG,   2 },
                        { "~",     1, UO_NOT,   2 },
                        { NULL, } };

// Binary operators (as in operators that work on two values
// rather than ones that are necessarily bit-wise).
enum
{
  OP_MOD = 1,
  OP_XOR,
  OP_SHR,
  OP_SHL,
  OP_NE,
  OP_GE,
  OP_LE,
  OP_GT,
  OP_LT,
  OP_AND,
  OP_OR,
  OP_EQ,
  OP_MLT,
  OP_DIV,
  OP_ADD,
  OP_SUB,
  OP_POW
};

// Must be length sorted
struct oper biops[] = { { "mod", 3, OP_MOD, 3 },
                        { "xor", 3, OP_XOR, 6 },
                        { ">>",  2, OP_SHR, 3 },
                        { "<<",  2, OP_SHL, 3 },
                        { "!=",  2, OP_NE,  7 },
                        { ">=",  2, OP_GE,  7 },
                        { "<=",  2, OP_LE,  7 },
                        { ">",   1, OP_GT,  7 },
                        { "<",   1, OP_LT,  7 },
                        { "&",   1, OP_AND, 5 },
                        { "|",   1, OP_OR,  5 },
                        { "=",   1, OP_EQ,  7 },
                        { "*",   1, OP_MLT, 3 },
                        { "",   1, OP_MLT, 3 },
                        { "/",   1, OP_DIV, 3 },
                        { "",   1, OP_DIV, 3 },
                        { "+",   1, OP_ADD, 4 },
                        { "-",   1, OP_SUB, 4 },
                        { "^",   1, OP_POW, 3 },
                        { NULL, } };


/**** ASCII helper functions ****/
BOOL isws( TEXT c )
{
  if( ( c == 32 ) || ( c == 9 ) ) return TRUE;
  return FALSE;
}

BOOL isnum( TEXT c )
{
  if( ( c >= '0' ) && ( c <= '9' ) ) return TRUE;
  return FALSE;
}

BOOL isbin( TEXT c )
{
  if( ( c == '0' ) || ( c == '1' ) ) return TRUE;
  return FALSE;
}

int32 hexit( TEXT c )
{
  if( ( c >= '0' ) && ( c <= '9' ) ) return c-'0';
  if( ( c >= 'a' ) && ( c <= 'f' ) ) return c-('a'-10);
  if( ( c >= 'A' ) && ( c <= 'F' ) ) return c-('A'-10);
  return -1;
}


/*** General initialisation/shutdown ***/
BOOL init( void )
{
  argh = ReadArgs( argfmt, (IPTR *)rargs, NULL );
  if( !argh )
  {
    printf( "Usage: yae %s\n", argfmt );
    return FALSE;
  }
  
  return TRUE;
}

void shut( void )
{
  if( argh )     FreeArgs( argh );
}

/*
** This pushes a double value and operator identifier onto a
** dynamically (re-)allocated stack.
*/
BOOL dbl_push( double v, uint16 op, double **stack, uint16 **opstack, uint32 *stacksize, uint32 *stackptr )
{
  // Need to reallocate?
  if( *stackptr == *stacksize )
  {
    uint32  tmpsize;
    uint16 *tmpopstack;
    double *tmpstack;

    // Get new buffers    
    tmpsize = (*stacksize) + 32;
    tmpstack = AllocVecTags( sizeof( double ) * tmpsize, TAG_DONE );
    if( !tmpstack )
    {
      printf( "Out of memory\n" );
      return FALSE;
    }
    
    tmpopstack = AllocVecTags( sizeof( uint16 ) * tmpsize, TAG_DONE );
    if( !tmpopstack )
    {
      FreeVec( tmpstack );
      printf( "Out of memory\n" );
      return FALSE;
    }
    
    // Copy from old ones and free them
    if( *stack )
    {
      CopyMem( *stack,   tmpstack,   (*stacksize) * sizeof( double ) );
      CopyMem( *opstack, tmpopstack, (*stacksize) * sizeof( uint16 ) );
    
      FreeVec( *stack );
      FreeVec( *opstack );
    }
    
    // Change pointers
    *stack   = tmpstack;
    *opstack = tmpopstack;
  }
  
  // Place new values onto the top of the stack
  (*stack)[*stackptr]   = v;
  (*opstack)[*stackptr] = op;
  (*stackptr)++;
  return TRUE;
}

// Perform a binary operator on two double values
BOOL do_dbl_op( uint16 op, double *res, double lft, double rgt )
{
  switch( op )
  {
    case OP_MOD:
      printf( "%% operator is invalid in floating point mode\n" );
      return FALSE;
    
    case OP_XOR:
      printf( "XOR operator is invalid in floating point mode\n" );
      return FALSE;
    
    case OP_SHR:
    case OP_SHL:
      printf( "Shift operators are invalid in floating point mode\n" );
      return FALSE;
    
    case OP_NE:
      *res = lft != rgt;
      break;
    
    case OP_GE:
      *res = lft >= rgt;
      break;
    
    case OP_LE:
      *res = lft <= rgt;
      break;
    
    case OP_GT:
      *res = lft > rgt;
      break;
    
    case OP_LT:
      *res = lft < rgt;
      break;
    
    case OP_AND:
      printf( "AND operator is invalid in floating point mode\n" );
      return FALSE;
    
    case OP_OR:
      printf( "OR operator is invalid in floating point mode\n" );
      return FALSE;
    
    case OP_EQ:
      *res = lft == rgt;
      break;
    
    case OP_MLT:
      *res = lft * rgt;
      break;
    
    case OP_DIV:
      if( rgt == 0 )
      {
        printf( "Division by zero!\n" );
        return FALSE;
      }
      *res = lft / rgt;
      break;
    
    case OP_ADD:
      *res = lft + rgt;
      break;
    
    case OP_SUB:
      *res = lft - rgt;
      break;
    
    case OP_POW:
      *res = pow( lft, rgt );
      break;
    
    default:
      printf( "Internal error: Unimplimented operator!? Please report\n" );
      return FALSE;
  }

  return TRUE;
}

/*
** This parses an expression. It recurses upon finding an
** open-bracket.
*/
BOOL dbl_parse_part( TEXT *ex, double *resbuf, uint32 *ip )
{
  uint32 u, o, i;
  TEXT *tmpp;
  double v;
  
  double *stack = NULL;
  uint16 *opstack = NULL;
  uint32 stacksize, stackptr;

  // Start with an empty stack  
  stacksize = 0;
  stackptr = 0;

  i = *ip;
  while( 1 )
  {
    // skip shitespace
    while( isws( ex[i] ) ) i++;
    
    // Look for a unary operator
    u = 0;
    while( unops[u].token != NULL )
    {
      if( strnicmp( &ex[i], unops[u].token, unops[u].toklen ) == 0 )
        break;
      u++;
    }

    // Found one! Skip past it.    
    if( unops[u].token )
      i += unops[u].toklen;
    
    // skip more whitespace
    while( isws( ex[i] ) ) i++;
    
    // Look for a bracket
    if( ex[i] == '(' )
    {
      i++;

      // Recurse into the bracket and get
      // the resulting value
      if( !dbl_parse_part( ex, &v, &i ) )
      {
        if( stack ) FreeVec( stack );
        if( opstack ) FreeVec( opstack );
        return FALSE;
      }
      if( ex[i] != ')' )
      {
        printf( "Closing bracket expected\n" );
        if( stack ) FreeVec( stack );
        if( opstack ) FreeVec( opstack );
        return FALSE;
      }
      i++;
    } else {
      // Check for an environment variable
      if( ex[i] == '\'' )
      {
        uint32 vs;
        int32 vl;
        
        i++;
        vs = i;
        while( ex[i] != '\'' )
        {
          if( ex[i] == 0 )
          {
            printf( "Missing apostrophe\n" );
            if( stack ) FreeVec( stack );
            if( opstack ) FreeVec( opstack );
            return FALSE;
          }
          i++;
        }
        
        ex[i] = 0;
        vl = GetVar( &ex[vs], varbuf, 1024, 0 );
        
        if( vl == -1 )
        {
          printf( "Variable '%s' not found.\n", &ex[vs] );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }
        
        v = strtod( varbuf, &tmpp );

        if( tmpp == varbuf )
        {
          printf( "Floating point value expected in variable '%s'\n", &ex[vs] );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }
        
        if( errno == ERANGE )
        {
          printf( "Value in variable '%s' out of range\n", &ex[vs] );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }

        ex[i++] = '\'';        
      } else {
        v = strtod( &ex[i], &tmpp );
        
        if( tmpp == &ex[i] )
        {
          printf( "Floating point value expected\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }
        
        if( errno == ERANGE )
        {
          printf( "Value out of range\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }
        
        i = tmpp - ex;
      }
    }
    
    // OK, got a value either directly, from a recursion
    // or from an environment variable.
    // Need to perform a unary operation?
    if( unops[u].token )
    {
      switch( unops[u].action )
      {
        case UO_NEG:
          v = -v;
          break;
        
        case UO_NOT:
          printf( "NOT operator is invalid on floating point mode.\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_SIN:
          v = sin( v );
          break;

        case UO_COS:
          v = cos( v );
          break;

        case UO_TAN:
          v = tan( v );
          break;

        case UO_ASIN:
          v = asin( v );
          break;
        
        case UO_SINH:
          v = sinh( v );
          break;

        case UO_COSH:
          v = cosh( v );
          break;

        case UO_TANH:
          v = tanh( v );
          break;

        case UO_ACOS:
          v = acos( v );
          break;

        case UO_ATAN:
          v = atan( v );
          break;
        
        case UO_ABS:
          v = fabs( v );
          break;
        
        case UO_SQRT:
          v = sqrt( v );
          break;
        
        case UO_LOG:
          v = log( v );
          break;
        
        case UO_LOG10:
          v = log10( v );
          break;
        
        case UO_EXP:
          v = exp( v );
          break;
        
        case UO_CEIL:
          v = ceil( v );
          break;
        
        case UO_FLOOR:
          v = floor( v );
          break;
        
        case UO_TRUNC:
          v = trunc( v );
          break;
        
        case UO_ROUND:
          v = round( v );
          break;
      }
    }
    
    // Skip whitespace
    while( isws( ex[i] ) ) i++;
    
    // Look for next binary operation
    o = 0;
    while( biops[o].token != NULL )
    {
      if( strnicmp( &ex[i], biops[o].token, biops[o].toklen ) == 0 )
        break;
      o++;
    }
    
    // Nothing left to do?
    if( !biops[o].token ) break;

    // Skip the operator
    i += biops[o].toklen;
    
    // Should we flush the stack?
    if( stackptr > 0 )
    {
      // Well, lets see
      if( biops[opstack[stackptr-1]].precedence <= biops[o].precedence )
      {
        // Empty the stack!
        while( stackptr > 0 )
        {
          stackptr--;
          if( !do_dbl_op( biops[opstack[stackptr]].action,
                          &v, 
                          stack[stackptr],
                          v ) )
          {
            if( stack ) FreeVec( stack );
            if( opstack ) FreeVec( opstack );
            return FALSE;
          }
        }
      }
    }
    
    // Shove it on the stack!
    if( !dbl_push( v, o, &stack, &opstack, &stacksize, &stackptr ) )
    {
      if( stack ) FreeVec( stack );
      if( opstack ) FreeVec( opstack );
      return FALSE;
    }
  }
  
  // Empty the stack!
  while( stackptr > 0 )
  {
    stackptr--;
    if( !do_dbl_op( biops[opstack[stackptr]].action,
                    &v, 
                    stack[stackptr],
                    v ) )
    {
      if( stack ) FreeVec( stack );
      if( opstack ) FreeVec( opstack );
      return FALSE;
    }
  }
  
  *resbuf = v;
  *ip = i;
  
  if( stack ) FreeVec( stack );
  if( opstack ) FreeVec( opstack );
  return TRUE;  
}

int do_dbl_parse( void )
{
  TEXT *ex;
  double result;
  uint32 i;

  // Get the expression to parse  
  ex = rargs[ARG_EXP];
  if( !ex ) return 5;
  
  // Initial result is zero
  result = 0;
  i = 0;

  // Parse it!
  if( !dbl_parse_part( ex, &result, &i ) )
    return 5;

  // Parser stopped before the end?
  if( ex[i] != 0 )
  {
    printf( "Syntax error: '%s'\n", &ex[i] );
    return 5;
  }
  
  // Print result
  if( rargs[ARG_HEX] )
  {
    printf( "0x%llx\n", *((int64 *)&result) );
  } else {
    printf( "%lf\n", result );
  }
  return 0;
}

// Decode an integer number in decimal, binary or hex
BOOL get_int( TEXT *ex, int64 *resbuf, uint32 *ip )
{
  uint32 i;
  int32 j;
  int64 v;
  
  i = *ip;
  while( isws( ex[i] ) ) i++;

  v = 0;
  if( ex[i] == '%' )
  {
    i++;
    if( !isbin( ex[i] ) )
    {
      printf( "Invalid binary digit\n" );
      return FALSE;
    }
    
    while( isbin( ex[i] ) )
    {
      v = (v<<1LL) | (ex[i]-'0');
      i++;
    }

    *resbuf = v;
    *ip = i;
    return TRUE;    
  }
  
  if( ( ex[i] == '$' ) ||
      ( ( ex[i] == '0' ) && ( ex[i+1] == 'x' ) ) )
  {
    if( ex[i] == '0' ) i++;
    i++;
    
    j = hexit( ex[i] );
    if( j == -1 )
    {
      printf( "Invalid hexadecimal digit\n" );
      return FALSE;
    }
    
    do
    {
      if( v & 0xf000000000000000LL )
      {
        printf( "Integer overflow\n" );
        return FALSE;
      }
      v = (v<<4) | ((int64)j);
    } while( ( ( j = hexit( ex[++i] ) ) != -1 ) );

    *resbuf = v;
    *ip = i;
    return TRUE;
  }
  
  if( !isnum( ex[i] ) )
  {
    printf( "Number expected\n" );
    return FALSE;
  }
  
  while( isnum( ex[i] ) )
  {
    v = (v*10) + (ex[i]-'0');
    i++;
  }

  *resbuf = v;
  *ip = i;
  return TRUE;
}

BOOL int_push( int64 v, uint16 op, int64 **stack, uint16 **opstack, uint32 *stacksize, uint32 *stackptr )
{
  // Need to reallocate?
  if( *stackptr == *stacksize )
  {
    uint32  tmpsize;
    uint16 *tmpopstack;
    int64  *tmpstack;
    
    tmpsize = (*stacksize) + 32;
    tmpstack = AllocVecTags( sizeof( int64 ) * tmpsize, TAG_DONE );
    if( !tmpstack )
    {
      printf( "Out of memory\n" );
      return FALSE;
    }
    
    tmpopstack = AllocVecTags( sizeof( uint16 ) * tmpsize, TAG_DONE );
    if( !tmpopstack )
    {
      FreeVec( tmpstack );
      printf( "Out of memory\n" );
      return FALSE;
    }
    
    if( *stack )
    {
      CopyMem( *stack,   tmpstack,   (*stacksize) * sizeof( int64 ) );
      CopyMem( *opstack, tmpopstack, (*stacksize) * sizeof( uint16 ) );
    
      FreeVec( *stack );
      FreeVec( *opstack );
    }
    
    *stack   = tmpstack;
    *opstack = tmpopstack;
  }
  
  (*stack)[*stackptr]   = v;
  (*opstack)[*stackptr] = op;
  (*stackptr)++;
  return TRUE;
}

BOOL do_int_op( uint16 op, int64 *res, int64 lft, int64 rgt )
{
  switch( op )
  {
    case OP_MOD:
      *res = lft % rgt;
      break;
    
    case OP_XOR:
      *res = lft ^ rgt;
      break;
    
    case OP_SHR:
      *res = lft >> rgt;
      break;
    
    case OP_SHL:
      *res = lft << rgt;
      break;
    
    case OP_NE:
      *res = lft != rgt;
      break;
    
    case OP_GE:
      *res = lft >= rgt;
      break;
    
    case OP_LE:
      *res = lft <= rgt;
      break;
    
    case OP_GT:
      *res = lft > rgt;
      break;
    
    case OP_LT:
      *res = lft < rgt;
      break;
    
    case OP_AND:
      *res = lft & rgt;
      break;
    
    case OP_OR:
      *res = lft | rgt;
      break;
    
    case OP_EQ:
      *res = lft == rgt;
      break;
    
    case OP_MLT:
      *res = lft * rgt;
      break;
    
    case OP_DIV:
      if( rgt == 0 )
      {
        printf( "Division by zero!\n" );
        return FALSE;
      }
      *res = lft / rgt;
      break;
    
    case OP_ADD:
      *res = lft + rgt;
      break;
    
    case OP_SUB:
      *res = lft - rgt;
      break;
    
    case OP_POW:
      *res = (int64)pow( (double)lft, (double)rgt );
      break;
    
    default:
      printf( "Internal error: Unimplimented operator!? Please report\n" );
      return FALSE;
  }

  return TRUE;
}

BOOL int_parse_part( TEXT *ex, int64 *resbuf, uint32 *ip )
{
  uint32 u, o, i;
  int64 v;
  
  int64 *stack = NULL;
  uint16 *opstack = NULL;
  uint32 stacksize, stackptr;
  
  stacksize = 0;
  stackptr = 0;

  i = *ip;
  while( 1 )
  {
    // skip shitespace
    while( isws( ex[i] ) ) i++;
    
    // Look for a unary operator
    u = 0;
    while( unops[u].token != NULL )
    {
      if( strnicmp( &ex[i], unops[u].token, unops[u].toklen ) == 0 )
        break;
      u++;
    }
    
    if( unops[u].token )
      i += unops[u].toklen;
    
    // skip more whitespace
    while( isws( ex[i] ) ) i++;
    
    // Look for a bracket
    if( ex[i] == '(' )
    {
      i++;
      if( !int_parse_part( ex, &v, &i ) )
      {
        if( stack ) FreeVec( stack );
        if( opstack ) FreeVec( opstack );
        return FALSE;
      }
      if( ex[i] != ')' )
      {
        printf( "Closing bracket expected\n" );
        if( stack ) FreeVec( stack );
        if( opstack ) FreeVec( opstack );
        return FALSE;
      }
      i++;
    } else {
      // Check for an environment variable
      if( ex[i] == '\'' )
      {
        uint32 vs, t;
        int32 vl;
        
        i++;
        vs = i;
        while( ex[i] != '\'' )
        {
          if( ex[i] == 0 )
          {
            printf( "Missing apostrophe\n" );
            if( stack ) FreeVec( stack );
            if( opstack ) FreeVec( opstack );
            return FALSE;
          }
          i++;
        }
        
        ex[i] = 0;
        vl = GetVar( &ex[vs], varbuf, 1024, 0 );
        
        if( vl == -1 )
        {
          printf( "Variable '%s' not found.\n", &ex[vs] );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }

        t = 0;
        if( !get_int( varbuf, &v, &t ) )
        {
          printf( "No valid number found in variable '%s'\n", &ex[vs] );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }

        ex[i++] = '\'';        
      } else {
        if( !get_int( ex, &v, &i ) )
        {
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
        }
      }
    }
    
    // Need to perform a unary operation?
    if( unops[u].token )
    {
      switch( unops[u].action )
      {
        case UO_NEG:
          v = -v;
          break;
        
        case UO_NOT:
          v = ~v;
          break;
        
        case UO_ABS:
          v = abs( v );
          break;

        case UO_SIN:
          printf( "SIN requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_COS:
          printf( "COS requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_TAN:
          printf( "TAN requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_SINH:
          printf( "SINH requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_COSH:
          printf( "COSH requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_TANH:
          printf( "TANH requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_ASIN:
          printf( "ASIN requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_ACOS:
          printf( "ACOS requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_ATAN:
          printf( "ATAN requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;          
        
        case UO_SQRT:
          v = (int64)sqrt( (double)v );
          break;

        case UO_LOG:
          v = (int64)log( (double)v );
          break;
        
        case UO_LOG10:
          v = (int64)log10( (double)v );
          break;

        case UO_EXP:
          printf( "EXP requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_CEIL:
          printf( "CEIL requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_FLOOR:
          printf( "FLOOR requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_TRUNC:
          printf( "TRUNC requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;

        case UO_ROUND:
          printf( "ROUND requires float mode\n" );
          if( stack ) FreeVec( stack );
          if( opstack ) FreeVec( opstack );
          return FALSE;
      }
    }
    
    // Skip whitespace
    while( isws( ex[i] ) ) i++;
    
    // Look for next binary operation
    o = 0;
    while( biops[o].token != NULL )
    {
      if( strnicmp( &ex[i], biops[o].token, biops[o].toklen ) == 0 )
        break;
      o++;
    }
    
    // Nothing left to do?
    if( !biops[o].token ) break;

    // Skip the operator
    i += biops[o].toklen;
    
    // Should we flush the stack?
    if( stackptr > 0 )
    {
      // Well, lets see
      if( biops[opstack[stackptr-1]].precedence <= biops[o].precedence )
      {
        // Empty the stack!
        while( stackptr > 0 )
        {
          stackptr--;
          if( !do_int_op( biops[opstack[stackptr]].action,
                          &v, 
                          stack[stackptr],
                          v ) )
          {
            if( stack ) FreeVec( stack );
            if( opstack ) FreeVec( opstack );
            return FALSE;
          }
        }
      }
    }
    
    // Shove it on the stack!
    if( !int_push( v, o, &stack, &opstack, &stacksize, &stackptr ) )
    {
      if( stack ) FreeVec( stack );
      if( opstack ) FreeVec( opstack );
      return FALSE;
    }
  }
  
  // Empty the stack!
  while( stackptr > 0 )
  {
    stackptr--;
    if( !do_int_op( biops[opstack[stackptr]].action,
                    &v, 
                    stack[stackptr],
                    v ) )
    {
      if( stack ) FreeVec( stack );
      if( opstack ) FreeVec( opstack );
      return FALSE;
    }
  }
  
  *resbuf = v;
  *ip = i;
  
  if( stack ) FreeVec( stack );
  if( opstack ) FreeVec( opstack );
  return TRUE;  
}

int do_int_parse( void )
{
  TEXT *ex;
  int64 result;
  uint32 i;
  
  ex = rargs[ARG_EXP];
  if( !ex ) return 5;
  
  result = 0;
  i = 0;

  if( !int_parse_part( ex, &result, &i ) )
    return 5;

  if( ex[i] != 0 )
  {
    printf( "Syntax error: '%s'\n", &ex[i] );
    return 5;
  }
  
  if( rargs[ARG_HEX] )
  {
    printf( "0x%llx\n", result );
  } else {
    printf( "%lld\n", result );
  }
  return 0;
}

int main( void )
{
  int rval=0;

  if( init() )
  {
    if( rargs[ARG_FLOAT] )
      rval = do_dbl_parse();
    else
      rval = do_int_parse();
  }
  shut();

  return rval;
}
