/*
    Name:    Prefs.c
    Author:  Peter Hutchison 2009 (C)
    Project: PrinterPS Prefences
    Lang:    English
*/

/*********************************************************************************************/

#include "global.h"
#include <aros/macros.h>

#define DEBUG 1
#include <aros/debug.h>

#include <prefs/printertxt.h>
#include <prefs/printergfx.h>
#include <prefs/printerps.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

/*********************************************************************************************/

struct FilePrefHeader
{
    UBYTE ph_Version;
    UBYTE ph_Type;
    UBYTE ph_Flags[4];
};

/*********************************************************************************************/

static struct PrinterPSPrefs  restore_prefs;
struct PrinterPSPrefs         printerpsprefs;

static APTR mempool;

/*********************************************************************************************/

void InitPrefs(STRPTR filename, BOOL use, BOOL save)
{
    mempool = CreatePool(MEMF_PUBLIC | MEMF_CLEAR, 2048, 2048);
    if (!mempool) Cleanup("Out of memory!");

    if (!LoadPrefs(filename))
    {
        if (!DefaultPrefs())
    {
        Cleanup("Panic! Cannot setup default prefs!");
    }
    }
    
    restore_prefs = printerpsprefs;
   
    if (use || save)
    {
        SavePrefs(CONFIGNAME_ENV);
    }
    
    if (save)
    {
        SavePrefs(CONFIGNAME_ENVARC);
    }
    
    if (use || save) Cleanup(NULL);
}

/*********************************************************************************************/

void CleanupPrefs(void)
{
    if (mempool) DeletePool(mempool);
}

/*********************************************************************************************/

BOOL LoadPrefs(STRPTR filename)
{
    /* PrinterPS.prefs has two chunks as below */
    static struct PrinterPSPrefs        loadprefs;
    struct IFFHandle *iff;    
    BOOL   retval = FALSE;
    struct ContextNode *cn;
    LONG chunks[2] = { ID_PREF,ID_PSPD};
        
                          
    if ((iff = AllocIFF()))
    {
        if ((iff->iff_Stream = Open(filename, (LONG)MODE_OLDFILE)))
    {
    
        InitIFFasDOS(iff);
        
        if (!OpenIFF(iff, IFFF_READ))
        {   
            if (!StopChunks(iff, chunks, 1)) /* 1 pair of chunks to search */
                {
                    /* Scan until end of file reached */
            while (ParseIFF(iff, IFFPARSE_SCAN) != IFFERR_EOF)
            {       
            cn = CurrentChunk(iff);

            if (cn->cn_Size == sizeof(struct PrinterPSPrefs))
                {
                    if (ReadChunkBytes(iff, &loadprefs, sizeof(struct PrinterPSPrefs)) == sizeof(struct PrinterPSPrefs))
                {
                            printerpsprefs = loadprefs;
                retval = TRUE;
                } 
                        }
                                                
            } /* if (!ParseIFF(iff, IFFPARSE_SCAN)) */

        } /* if (!StopChunk(iff, ID_PREF, ID_PTXT)) */

            CloseIFF(iff);

        } /* if (!OpenIFF(iff, IFFF_READ)) */
                                
        Close((BPTR)(iff->iff_Stream));
        
    } /* if ((iff->iff_Stream = (IPTR)Open(filename, MODE_OLDFILE))) */
    
    FreeIFF(iff);
    
    } /* if ((iff = AllocIFF())) */
    
    return retval;
}

/*********************************************************************************************/

BOOL SavePrefs(STRPTR filename)
{
    static struct PrinterPSPrefs     saveprefs;
    struct IFFHandle            *iff;    
    BOOL                        retval = FALSE, delete_if_error = FALSE;
    struct FilePrefHeader head;
    
    saveprefs = printerpsprefs; 
       
    if ((iff = AllocIFF()))
    {
        if ((iff->iff_Stream = Open(filename, (LONG)MODE_NEWFILE)))
    {    
        delete_if_error = TRUE;
        
        InitIFFasDOS(iff);
        
        if (!OpenIFF(iff, IFFF_WRITE))
        {   
        if (!PushChunk(iff, ID_PREF, ID_FORM, IFFSIZE_UNKNOWN))
        {
        
            if (!PushChunk(iff, ID_PREF, ID_PRHD, sizeof(struct FilePrefHeader)))
            {
        
            head.ph_Version  = 0; /* FIXME: should be PHV_CURRENT, but see <prefs/prefhdr.h> */
            head.ph_Type     = 0;
            head.ph_Flags[0] = 0;
            head.ph_Flags[1] = 0;
            head.ph_Flags[2] = 0;
            head.ph_Flags[3] = 0;
            
            if (WriteChunkBytes(iff, &head, sizeof(head)) == sizeof(head))
            {
            
                PopChunk(iff);
                
                if (!PushChunk(iff, ID_PREF, ID_PSPD, sizeof(struct PrinterPSPrefs)))
                {
            
                    if (WriteChunkBytes(iff, &saveprefs, sizeof(saveprefs)) == sizeof(saveprefs))
                {
                
                    retval = TRUE;
                }
                
                        PopChunk(iff);

                } /* if (!PushChunk(iff, ID_PREF, ID_PTXT, sizeof(struct LocalePrefs))) */
                                                                    
            } /* if (WriteChunkBytes(iff, &head, sizeof(head)) == sizeof(head)) */
            else
                {
                PopChunk(iff);
            }
            
            } /* if (!PushChunk(iff, ID_PREF, ID_PRHD, sizeof(struct PrefHeader))) */
            
            PopChunk(iff);
                        
        } /* if (!PushChunk(iff, ID_PREF, ID_FORM, IFFSIZE_UNKNOWN)) */
        
            CloseIFF(iff);
                
        } /* if (!OpenIFF(iff, IFFFWRITE)) */
        
        Close((BPTR)iff->iff_Stream);
        
    } /* if ((iff->iff_Stream = (IPTR)Open(filename, MODE_NEWFILE))) */
    
    FreeIFF(iff);
    
    } /* if ((iff = AllocIFF())) */
    
    if (!retval && delete_if_error)
    {
        DeleteFile(filename);
    }
    
    return retval;    
}

/*********************************************************************************************/

BOOL DefaultPrefs(void)
{
    int r;
    
    /* Global */
    printerpsprefs.ps_DriverMode = DM_POSTSCRIPT;
    printerpsprefs.ps_PaperFormat = PF_USLETTER;
    printerpsprefs.ps_Reserved1[0]     = 0;
    printerpsprefs.ps_Reserved1[1]     = 0;
    printerpsprefs.ps_Copies = 1;
    /* Convert from inches to 1/72000 in */
    printerpsprefs.ps_PaperWidth = (LONG)(8.5 * 72000) ;
    printerpsprefs.ps_PaperHeight = (LONG)(11.0 * 72000);
    printerpsprefs.ps_HorizontalDPI = 300;
    printerpsprefs.ps_VerticalDPI = 300;
    
    /* Text Options */
    printerpsprefs.ps_Font = FONT_COURIER;
    printerpsprefs.ps_Pitch = PITCH_NORMAL;
    printerpsprefs.ps_Orientation = ORIENT_PORTRAIT;
    printerpsprefs.ps_Tab = TAB_INCH;
    for (r=0; r<8; r++) {
      printerpsprefs.ps_Reserved2[r] = 0;
    }
       
    /* Text Dimensions */
    printerpsprefs.ps_LeftMargin = 1 * 72000;
    printerpsprefs.ps_RightMargin = 1 * 72000;
    printerpsprefs.ps_TopMargin = 1 * 72000;
    /* Font and leading are in 1/1000 ths */
    printerpsprefs.ps_BottomMargin = 1 * 1000;
    printerpsprefs.ps_FontPointSize = 10 * 1000;
    printerpsprefs.ps_Leading = 2 * 10;
    for (r=0; r<8; r++) {
      printerpsprefs.ps_Reserved3[r] = 0;
    }
    
    /* Graphics Options */
    printerpsprefs.ps_LeftEdge = 1 * 72000;
    printerpsprefs.ps_TopEdge = 1 * 72000;
    printerpsprefs.ps_Width = 6.5 * 72000;
    printerpsprefs.ps_Height = 9 * 72000;
    printerpsprefs.ps_Image = IM_POSITIVE;
    printerpsprefs.ps_Shading = SHAD_GREYSCALE;
    printerpsprefs.ps_Dithering = DITH_DEFAULT;
    for (r=0; r<9; r++) {
      printerpsprefs.ps_Reserved4[r] = 0;
    }


    /* Graphics Scaling */
    printerpsprefs.ps_Aspect = PA_HORIZONTAL;
    printerpsprefs.ps_ScalingType = ST_ASPECT_BOTH;
    printerpsprefs.ps_Reserved5 = 0;
    printerpsprefs.ps_Centering = CENT_BOTH;
    for (r=0; r<8; r++) {
      printerpsprefs.ps_Reserved6[r] = 0;
    }
       
    return TRUE;
}

/*********************************************************************************************/

void RestorePrefs(void)
{
    printerpsprefs         = restore_prefs; 
}

void Cleanup(char *msg)
{
    printf("%s\n", msg);
    CleanupPrefs();
}

/*********************************************************************************************/
/*********************************************************************************************/
