/*
    Name:    Prefs.c
    Author:  Peter Hutchison 2007 (C)
    Project: Printer Prefences
    Lang:    English
*/

/*********************************************************************************************/

#include "global.h"
#include <aros/macros.h>

#define DEBUG 1
#include <aros/debug.h>

#include <prefs/printertxt.h>
#include <prefs/printergfx.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/*********************************************************************************************/

struct FilePrefHeader
{
    UBYTE ph_Version;
    UBYTE ph_Type;
    UBYTE ph_Flags[4];
};

/*********************************************************************************************/

static struct PrinterTxtPrefs        restore_prefs1;
static struct PrinterUnitPrefs       restore_prefs2;
static struct PrinterGfxPrefs        restore_prefs3;

struct PrinterTxtPrefs        printertxtprefs;
struct PrinterUnitPrefs       printerunitprefs;
struct PrinterGfxPrefs        printergfxprefs;

static APTR mempool;

/*********************************************************************************************/

void InitPrefs(STRPTR filename, BOOL use, BOOL save)
{
    mempool = CreatePool(MEMF_PUBLIC | MEMF_CLEAR, 2048, 2048);
    if (!mempool) Cleanup("Out of memory!");

    if (!LoadPrefs(filename))
    {
    	if (!DefaultPrefs())
	{
	    Cleanup("Panic! Cannot setup default prefs!");
	}
    }
    
    restore_prefs1 = printertxtprefs;
    restore_prefs2 = printerunitprefs;
    restore_prefs3 = printergfxprefs;
    
    if (use || save)
    {
    	SavePrefs(CONFIGNAME_ENV);
    }
    
    if (save)
    {
    	SavePrefs(CONFIGNAME_ENVARC);
    }
    
    if (use || save) Cleanup(NULL);
}

/*********************************************************************************************/

void CleanupPrefs(void)
{
    if (mempool) DeletePool(mempool);
}

/*********************************************************************************************/

BOOL LoadPrefs(STRPTR filename)
{
    /* Printer.prefs has four chunks as below */
    static struct PrinterTxtPrefs        loadprefs1;
    static struct PrinterUnitPrefs       loadprefs2;
    static struct PrinterGfxPrefs        loadprefs3;
    struct IFFHandle *iff;    
    BOOL   retval = FALSE;
    struct ContextNode *cn;
    LONG chunks[8] = { ID_PREF,ID_PTXT, ID_PREF,ID_PUNT, ID_PREF, ID_PGFX };
        
                          
    if ((iff = AllocIFF()))
    {
    	if ((iff->iff_Stream = Open(filename, MODE_OLDFILE)))
	{
    
	    InitIFFasDOS(iff);
	    
	    if (!OpenIFF(iff, IFFF_READ))
	    {	
	    	if (!StopChunks(iff, chunks, 4)) /* 4 pairs of chunks to search */
                {
                    /* Scan until end of file reached */
		    while (ParseIFF(iff, IFFPARSE_SCAN) != IFFERR_EOF)
		    {		
			cn = CurrentChunk(iff);

			if (cn->cn_Size == sizeof(struct PrinterTxtPrefs))
		        {
		    	    if (ReadChunkBytes(iff, &loadprefs1, sizeof(struct PrinterTxtPrefs)) == sizeof(struct PrinterTxtPrefs))
			    {
   	    	    	    	printertxtprefs = loadprefs1;
				retval = TRUE;
			    } 
                        }
                        
                        if (cn->cn_Size == sizeof(struct PrinterUnitPrefs))
			{   
		    	    if (ReadChunkBytes(iff, &loadprefs2, sizeof(struct PrinterUnitPrefs)) == sizeof(struct PrinterUnitPrefs))
			    {
   	    	    	    	printerunitprefs = loadprefs2;		
				retval = TRUE;
			    } 
			}

			if (cn->cn_Size == sizeof(struct PrinterGfxPrefs))
			{
		    	    if (ReadChunkBytes(iff, &loadprefs3, sizeof(struct PrinterGfxPrefs)) == sizeof(struct PrinterGfxPrefs))
			    {
   	    	    	    	printergfxprefs = loadprefs3;  			    			
				retval = TRUE;
			    } 
			}
                        
		    } /* if (!ParseIFF(iff, IFFPARSE_SCAN)) */

		} /* if (!StopChunk(iff, ID_PREF, ID_PTXT)) */

	    	CloseIFF(iff);

	    } /* if (!OpenIFF(iff, IFFF_READ)) */
            	           	    
	    Close(iff->iff_Stream);
	    
	} /* if ((iff->iff_Stream = (IPTR)Open(filename, MODE_OLDFILE))) */
	
	FreeIFF(iff);
	
    } /* if ((iff = AllocIFF())) */
    
    return retval;
}

/*********************************************************************************************/

BOOL SavePrefs(STRPTR filename)
{
    static struct PrinterTxtPrefs 	 saveprefs1;
    static struct PrinterUnitPrefs       saveprefs2;
    static struct PrinterGfxPrefs        saveprefs3;
    struct IFFHandle 	     	*iff;    
    BOOL    	    	    	retval = FALSE, delete_if_error = FALSE;
    struct FilePrefHeader head;
    
    saveprefs1 = printertxtprefs;
    saveprefs2 = printerunitprefs;
    saveprefs3 = printergfxprefs;
    
       
    if ((iff = AllocIFF()))
    {
    	if ((iff->iff_Stream = Open(filename, MODE_NEWFILE)))
	{    
	    delete_if_error = TRUE;
	    
	    InitIFFasDOS(iff);
	    
	    if (!OpenIFF(iff, IFFF_WRITE))
	    {	
		if (!PushChunk(iff, ID_PREF, ID_FORM, IFFSIZE_UNKNOWN))
		{
	    
		    if (!PushChunk(iff, ID_PREF, ID_PRHD, sizeof(struct FilePrefHeader)))
		    {
		
			head.ph_Version  = 0; /* FIXME: should be PHV_CURRENT, but see <prefs/prefhdr.h> */
			head.ph_Type     = 0;
			head.ph_Flags[0] = 0;
			head.ph_Flags[1] = 0;
			head.ph_Flags[2] = 0;
			head.ph_Flags[3] = 0;
			
			if (WriteChunkBytes(iff, &head, sizeof(head)) == sizeof(head))
			{
		    
			    PopChunk(iff);
			    
			    if (!PushChunk(iff, ID_PREF, ID_PTXT, sizeof(struct PrinterTxtPrefs)))
			    {
			
			    	if (WriteChunkBytes(iff, &saveprefs1, sizeof(saveprefs1)) == sizeof(saveprefs1))
				{
			    
				    retval = TRUE;
				}
				
    			    	PopChunk(iff);

			    } /* if (!PushChunk(iff, ID_PREF, ID_PTXT, sizeof(struct LocalePrefs))) */
			    			    
			    if (!PushChunk(iff, ID_PREF, ID_PUNT, sizeof(struct PrinterUnitPrefs)))
			    {
			
			    	if (WriteChunkBytes(iff, &saveprefs2, sizeof(saveprefs2)) == sizeof(saveprefs2))
				{
			    
				    retval = TRUE;
				}
				
    			    	PopChunk(iff);

			    } /* if (!PushChunk(iff, ID_PREF, ID_PUNT, sizeof(struct LocalePrefs))) */
	                                                                              
	                  
			    if (!PushChunk(iff, ID_PREF, ID_PGFX, sizeof(struct PrinterGfxPrefs)))
			    {
			
			    	if (WriteChunkBytes(iff, &saveprefs3, sizeof(saveprefs3)) == sizeof(saveprefs3))
				{
			    
				    retval = TRUE;
				}
				
    			    	PopChunk(iff);

			    } /* if (!PushChunk(iff, ID_PREF, ID_PGFX, sizeof(struct LocalePrefs))) */
	                                                                    
			} /* if (WriteChunkBytes(iff, &head, sizeof(head)) == sizeof(head)) */
			else
		    	{
			    PopChunk(iff);
			}
			
		    } /* if (!PushChunk(iff, ID_PREF, ID_PRHD, sizeof(struct PrefHeader))) */
		    
		    PopChunk(iff);
		    		    
		} /* if (!PushChunk(iff, ID_PREF, ID_FORM, IFFSIZE_UNKNOWN)) */
		
	    	CloseIFF(iff);
				
	    } /* if (!OpenIFF(iff, IFFFWRITE)) */
	    
	    Close((BPTR)iff->iff_Stream);
	    
	} /* if ((iff->iff_Stream = (IPTR)Open(filename, MODE_NEWFILE))) */
	
	FreeIFF(iff);
	
    } /* if ((iff = AllocIFF())) */
    
    if (!retval && delete_if_error)
    {
    	DeleteFile(filename);
    }
    
    return retval;    
}

/*********************************************************************************************/

BOOL DefaultPrefs(void)
{
   
    printertxtprefs.pt_Reserved[0]     = 0;
    printertxtprefs.pt_Reserved[1]     = 0;
    printertxtprefs.pt_Reserved[2]     = 0;
    printertxtprefs.pt_Reserved[3]     = 0;
    strcpy(printertxtprefs.pt_Driver, "Generic");
    printertxtprefs.pt_Port = PP_PARALLEL;
    printertxtprefs.pt_PaperType = PT_FANFOLD;
    printertxtprefs.pt_PaperSize = PS_US_LETTER;
    printertxtprefs.pt_PaperLength = 66;
    printertxtprefs.pt_Pitch = PP_PICA;
    printertxtprefs.pt_Spacing = PS_SIX_LPI;
    printertxtprefs.pt_LeftMargin = 5;
    printertxtprefs.pt_RightMargin = 75;
    printertxtprefs.pt_Quality = PQ_DRAFT;
    
    printerunitprefs.pu_Reserved[0] = 0;
    printerunitprefs.pu_Reserved[1] = 0;
    printerunitprefs.pu_Reserved[2] = 0;
    printerunitprefs.pu_Reserved[2] = 0;
    printerunitprefs.pu_UnitNum = 0;
    printerunitprefs.pu_OpenDeviceFlags = 0;
    strcpy(printerunitprefs.pu_DeviceName , "\0");
       
    printergfxprefs.pg_Reserved[0] = 0;
    printergfxprefs.pg_Reserved[1] = 0;
    printergfxprefs.pg_Reserved[2] = 0;
    printergfxprefs.pg_Reserved[3] = 0;
    printergfxprefs.pg_Aspect = PA_HORIZONTAL;
    printergfxprefs.pg_Shade = PS_BW;
    printergfxprefs.pg_Image = PI_POSITIVE;
    printergfxprefs.pg_Threshold = 7;
    printergfxprefs.pg_ColorCorrect = 0;
    printergfxprefs.pg_Dimensions = 0;
    printergfxprefs.pg_Dithering = PD_ORDERED;
    printergfxprefs.pg_GraphicFlags = 0;
    printergfxprefs.pg_PrintDensity = 1;
    printergfxprefs.pg_PrintMaxWidth = 0;
    printergfxprefs.pg_PrintMaxHeight = 0;
    printergfxprefs.pg_PrintXOffset = 0;
    printergfxprefs.pg_PrintYOffset = 0;
       
    return TRUE;
}

/*********************************************************************************************/

void RestorePrefs(void)
{
    printertxtprefs         = restore_prefs1;
    printerunitprefs        = restore_prefs2;
    printergfxprefs         = restore_prefs3;
 
}

void Cleanup(char *msg)
{
    printf("%s\n", msg);
    CleanupPrefs();
}

/*********************************************************************************************/
