/* Copyright 2010 Fredrik Wikstrom. All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
**
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
**
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS `AS IS'
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
** LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
** CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
** SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
** CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
** POSSIBILITY OF SUCH DAMAGE.
*/

#include <exec/exec.h>
#include <dos/dos.h>
#include <libraries/mui.h>
#include <proto/exec.h>
#include <proto/dos.h>
#include <proto/intuition.h>
#include <proto/graphics.h>
#include <proto/muimaster.h>
#include <proto/icon.h>
#include <clib/alib_protos.h>
#include <SDI_hook.h>
#include <stdio.h>
#include <string.h>
#include "ahi.h"
#include "scsicmd.h"
#include "diskchange.h"
#include "seekbar.h"
#include "PlayCDDA_rev.h"

#define PROGNAME "PlayCDDA"
const char USED verstag[] = VERSTAG;

#ifndef DtpicObject
#define DtpicObject MUI_NewObject(MUIC_Dtpic
#endif

enum {
	GID_POSITION = 0,
	GID_EJECT,
	GID_STOP,
	GID_PAUSE,
	GID_PREV,
	GID_PLAY,
	GID_NEXT,
	GID_VOLUME,
	GID_TRACK01,
	GID_TRACK32 = GID_TRACK01 + 31,
	NUM_GADGETS
};

void UpdateGUI (Object *wnd, Object **gad, struct CDTOC *cd_toc);
Object *CreateTable (LONG rows, LONG cols, Object **buttons);
Object *CreateColumn (LONG rows, LONG cols, LONG i, Object **buttons);

HOOKPROTO(SeekFunc, IPTR, APTR unused, IPTR *params);
MakeHook(SeekHook, SeekFunc);
HOOKPROTO(EjectFunc, IPTR, APTR unused, IPTR *params);
MakeHook(EjectHook, EjectFunc);
HOOKPROTO(StopFunc, IPTR, APTR unused, IPTR *params);
MakeHook(StopHook, StopFunc);
HOOKPROTO(PauseFunc, IPTR, APTR unused, IPTR *params);
MakeHook(PauseHook, PauseFunc);
HOOKPROTO(PrevFunc, IPTR, APTR unused, IPTR *params);
MakeHook(PrevHook, PrevFunc);
HOOKPROTO(PlayFunc, IPTR, APTR unused, IPTR *params);
MakeHook(PlayHook, PlayFunc);
HOOKPROTO(NextFunc, IPTR, APTR unused, IPTR *params);
MakeHook(NextHook, NextFunc);
HOOKPROTO(TrackFunc, IPTR, APTR unused, IPTR *params);
MakeHook(TrackHook, TrackFunc);

struct MUI_CustomClass *SeekBarClass = NULL;
struct IOStdReq *cd_io = NULL;
LONG load_eject = 1;

int main (void) {
	APTR cdda_buf = NULL;
	struct DiskObject *icon = NULL;
	const char *dosdev;
	APTR cd_change = NULL;
	LONG cd_signal = -1;
	ULONG ahi_sig, gui_sig, cd_sig;
	LONG cd_in_drive = FALSE;
	struct CDTOC cd_toc;
	Object *app = NULL, *wnd;
	Object *gad[NUM_GADGETS];
	ULONG signals = 0;
	volatile ULONG volume = 64;
	ULONG track;
	
	cdda_buf = AllocVec(2352*15, MEMF_ANY);
	if (!cdda_buf) {
		goto error;
	}

	ahi_sig = OpenAHIDevice(2352*15);
	if (!ahi_sig) {
		goto error;
	}
	
	icon = GetDiskObjectNew("PROGDIR:"PROGNAME);
	if (!icon) {
		goto error;
	}
	
	SeekBarClass = SeekBar_CreateClass();
	if (!SeekBarClass) {
		goto error;
	}
	
	app = ApplicationObject,
		MUIA_Application_Title,					PROGNAME,
		MUIA_Application_Version,				&verstag[1],
		MUIA_Application_DiskObject,			icon,
		SubWindow,								wnd = WindowObject,
			MUIA_Window_Title,					VERS,
			WindowContents,						HGroup,
				Child,							VGroup,
					Child,						RectangleObject,
					End,
					Child,						gad[GID_POSITION] = NewObject(SeekBarClass->mcc_Class, NULL,
						MUIA_InputMode,			MUIV_InputMode_RelVerify,
						MUIA_Disabled,			TRUE,
					TAG_END),
					Child,						HGroup,
						Child,					gad[GID_EJECT] = DtpicObject,
							ImageButtonFrame,
							MUIA_InputMode,		MUIV_InputMode_RelVerify,
							MUIA_Dtpic_Name,	"tbimages:tapeeject",
							MUIA_ShortHelp,		"Eject",
						End,
						Child,					gad[GID_STOP] = DtpicObject,
							ImageButtonFrame,
							MUIA_InputMode,		MUIV_InputMode_RelVerify,
							MUIA_Dtpic_Name,	"tbimages:tapestop",
							MUIA_ShortHelp,		"Stop",
							MUIA_Disabled,		TRUE,
						End,
						Child,					gad[GID_PAUSE] = DtpicObject,
							ImageButtonFrame,
							MUIA_InputMode,		MUIV_InputMode_RelVerify,
							MUIA_Dtpic_Name,	"tbimages:tapepause",
							MUIA_ShortHelp,		"Pause",
							MUIA_Disabled,		TRUE,
						End,
						Child,					gad[GID_PREV] = DtpicObject,
							ImageButtonFrame,
							MUIA_InputMode,		MUIV_InputMode_RelVerify,
							MUIA_Dtpic_Name,	"tbimages:tapelast",
							MUIA_ShortHelp,		"Previous",
							MUIA_Disabled,		TRUE,
						End,
						Child,					gad[GID_PLAY] = DtpicObject,
							ImageButtonFrame,
							MUIA_InputMode,		MUIV_InputMode_RelVerify,
							MUIA_Dtpic_Name,	"tbimages:tapeplay",
							MUIA_ShortHelp,		"Play",
							MUIA_Disabled,		TRUE,
						End,
						Child,					gad[GID_NEXT] = DtpicObject,
							ImageButtonFrame,
							MUIA_InputMode,		MUIV_InputMode_RelVerify,
							MUIA_Dtpic_Name,	"tbimages:tapenext",
							MUIA_ShortHelp,		"Next",
							MUIA_Disabled,		TRUE,
						End,
					End,
				End,
				Child,							CreateTable(4, 8, &gad[GID_TRACK01]),
				Child,							VGroup,
					Child,						gad[GID_VOLUME] = SliderObject,
						MUIA_Slider_Horiz,		FALSE,
						MUIA_Slider_Min,		0,
						MUIA_Slider_Max,		64,
						MUIA_Slider_Level,		64,
					End,
					Child,						TextObject,
						MUIA_Text_Contents,		"Volume",
					End,
				End,
			End,
		End,
	End;
	if (!app) {
		goto error;
	}
	
	DoMethod(wnd, MUIM_Notify, MUIA_Window_CloseRequest, TRUE,
		app, 2, MUIM_Application_ReturnID, MUIV_Application_ReturnID_Quit);
	DoMethod(gad[GID_POSITION], MUIM_Notify, MUIA_Prop_First, MUIV_EveryTime,
		app, 5, MUIM_CallHook, &SeekHook, wnd, gad, &cd_toc);
	DoMethod(gad[GID_EJECT], MUIM_Notify, MUIA_Pressed, FALSE,
		app, 5, MUIM_CallHook, &EjectHook, wnd, gad, &cd_toc);
	DoMethod(gad[GID_STOP], MUIM_Notify, MUIA_Pressed, FALSE,
		app, 5, MUIM_CallHook, &StopHook, wnd, gad, &cd_toc);
	DoMethod(gad[GID_PAUSE], MUIM_Notify, MUIA_Pressed, FALSE,
		app, 5, MUIM_CallHook, &PauseHook, wnd, gad, &cd_toc);
	DoMethod(gad[GID_PREV], MUIM_Notify, MUIA_Pressed, FALSE,
		app, 5, MUIM_CallHook, &PrevHook, wnd, gad, &cd_toc);
	DoMethod(gad[GID_PLAY], MUIM_Notify, MUIA_Pressed, FALSE,
		app, 5, MUIM_CallHook, &PlayHook, wnd, gad, &cd_toc);
	DoMethod(gad[GID_NEXT], MUIM_Notify, MUIA_Pressed, FALSE,
		app, 5, MUIM_CallHook, &NextHook, wnd, gad, &cd_toc);
	for (track = 0; track < 32; track++) {
		DoMethod(gad[GID_TRACK01 + track], MUIM_Notify, MUIA_Pressed, FALSE,
			app, 6, MUIM_CallHook, &TrackHook, wnd, gad, &cd_toc, track);
	}
	DoMethod(gad[GID_VOLUME], MUIM_Notify, MUIA_Slider_Level, MUIV_EveryTime,
		app, 3, MUIM_WriteLong, MUIV_TriggerValue, &volume);
	
	set(wnd, MUIA_Window_Open, TRUE);
	if (!XGET(wnd, MUIA_Window_Open)) {
		goto error;
	}
	
	dosdev = FindToolType(icon->do_ToolTypes, "DOSDEV");
	if (!dosdev) {
		dosdev = "CD0";
	}
	cd_io = GetCDDevice(dosdev);
	if (!cd_io) {
		goto error;
	}
	
	cd_signal = AllocSignal(-1);
	cd_sig = 1UL << cd_signal;
	cd_change = AddDiskChangeHandler(cd_io, cd_sig);
	if (!cd_change) {
		goto error;
	}
	
	cd_io->io_Command = TD_CHANGESTATE;
	DoIO((struct IORequest *)cd_io);
	cd_in_drive = (cd_io->io_Error || cd_io->io_Actual) ? FALSE : TRUE;
	memset(&cd_toc, 0, sizeof(cd_toc));
	if (cd_in_drive) {
		load_eject = 1;
		ReadCDTOC(cd_io, &cd_toc);
	}
	UpdateGUI(wnd, gad, &cd_toc);
	
	while ((LONG)DoMethod(app, MUIM_Application_NewInput, (IPTR)&gui_sig) !=
		MUIV_Application_ReturnID_Quit)
	{
		if (!cd_toc.IsPlaying || cd_toc.IsPaused) {
			signals = Wait(gui_sig|cd_sig|SIGBREAKF_CTRL_C);
		} else {
			signals = CheckSignal(gui_sig|cd_sig|SIGBREAKF_CTRL_C);
		}
		if (signals & SIGBREAKF_CTRL_C) {
			break;
		}
		if (signals & cd_sig) {
			cd_io->io_Command = TD_CHANGESTATE;
			DoIO((struct IORequest *)cd_io);
			cd_in_drive = (cd_io->io_Error || cd_io->io_Actual) ? FALSE : TRUE;
			memset(&cd_toc, 0, sizeof(cd_toc));
			if (cd_in_drive) {
				load_eject = 1;
				ReadCDTOC(cd_io, &cd_toc);
			}
			UpdateGUI(wnd, gad, &cd_toc);
		}
		if (cd_toc.IsPlaying && !cd_toc.IsPaused) {
			uint32 start_addr = cd_toc.CurrentAddr;
			uint32 end_addr = start_addr + 15;
			int32 cdda_size;
			if (end_addr > cd_toc.Tracks[cd_toc.CurrentTrack-1].EndAddr) {
				end_addr = cd_toc.Tracks[cd_toc.CurrentTrack-1].EndAddr;
			}
			if (!XGET(gad[GID_POSITION], MUIA_Pressed)) {
				struct Screen *screen;
				screen = (struct Screen *)XGET(wnd, MUIA_Window_Screen);
				if (screen && AttemptLockLayerRom(screen->BarLayer)) {
					UnlockLayerRom(screen->BarLayer);
					set(gad[GID_POSITION], MUIA_Slider_Level, start_addr);
				}
			}
			cdda_size = ReadMSF(cd_io, cdda_buf, start_addr, end_addr);
			if (cdda_size > 0) {
				PlayCDDA(cdda_buf, cdda_size, volume << 10);
				cd_toc.CurrentAddr = end_addr;
			}
			if (end_addr == cd_toc.Tracks[cd_toc.CurrentTrack-1].EndAddr) {
				if (cd_toc.CurrentTrack < cd_toc.LastTrack) {
					cd_toc.CurrentTrack++;
				} else {
					cd_toc.IsPlaying = FALSE;
					cd_toc.IsPaused = FALSE;
					cd_toc.CurrentTrack = cd_toc.FirstTrack;
				}
				cd_toc.CurrentAddr =
					cd_toc.Tracks[cd_toc.CurrentTrack-1].StartAddr;
				UpdateGUI(wnd, gad, &cd_toc);
			}
		}
	}
	
error:
	RemDiskChangeHandler(cd_change);
	FreeSignal(cd_signal);
	FreeCDDevice(cd_io);
	MUI_DisposeObject(app);
	SeekBar_DeleteClass(SeekBarClass);
	FreeDiskObject(icon);
	CloseAHIDevice();
	FreeVec(cdda_buf);

	return 0;
}

void UpdateGUI (Object *wnd, Object **gad, struct CDTOC *cd_toc) {
	ULONG track;
	
	track = cd_toc->CurrentTrack;
	if (cd_toc->IsPlaying && track && cd_toc->Tracks[track-1].IsAudio) {
		SetAttrs(gad[GID_POSITION],
			MUIA_Disabled,		FALSE,
			MUIA_Slider_Min,	cd_toc->Tracks[track-1].StartAddr,
			MUIA_Slider_Max,	cd_toc->Tracks[track-1].EndAddr,
			MUIA_Slider_Level,	cd_toc->CurrentAddr,
			TAG_END);
	} else {
		SetAttrs(gad[GID_POSITION],
			MUIA_Disabled,		TRUE,
			MUIA_Slider_Level,	0,
			TAG_END);
	}
	
	for (track = 0; track < MAX_TRACKS; track++) {
		SetAttrs(gad[GID_TRACK01 + track],
			MUIA_Disabled, 		cd_toc->Tracks[track].IsAudio ? FALSE : TRUE,
			MUIA_Text_PreParse,	cd_toc->CurrentTrack == (track + 1) ? "\33c\33b" : "\33c",
			TAG_END);
	}

	set(gad[GID_EJECT], MUIA_Disabled, FALSE);
	set(gad[GID_STOP], MUIA_Disabled, !cd_toc->IsPlaying && !cd_toc->IsPaused);
	set(gad[GID_PAUSE], MUIA_Disabled, !cd_toc->IsPlaying || cd_toc->IsPaused);
	set(gad[GID_PREV], MUIA_Disabled, !cd_toc->CurrentTrack || (cd_toc->CurrentTrack <= cd_toc->FirstTrack));
	set(gad[GID_PLAY], MUIA_Disabled, !cd_toc->CurrentTrack || (cd_toc->IsPlaying && !cd_toc->IsPaused));
	set(gad[GID_NEXT], MUIA_Disabled, !cd_toc->CurrentTrack || (cd_toc->CurrentTrack >= cd_toc->LastTrack));
}

Object *CreateTable (LONG rows, LONG cols, Object **buttons) {
	struct TagItem *tags, *tag;
	Object *table, *column;
	LONG i;
	
	tags = AllocVec(sizeof(struct TagItem) * (cols + 2), MEMF_CLEAR);
	if (!tags) {
		return NULL;
	}
	
	tag = tags;
	tag->ti_Tag = MUIA_Group_Horiz;
	tag->ti_Data = TRUE;
	tag++;
	
	for (i = 0; i < cols; i++) {
		column = CreateColumn(rows, cols, i, buttons);
		tag->ti_Tag = MUIA_Group_Child;
		tag->ti_Data = (IPTR)column;
		tag++;
	}
	
	tag->ti_Tag = TAG_END;
	table = MUI_NewObjectA(MUIC_Group, tags);
	FreeVec(tags);
	
	return table;
}

Object *CreateColumn (LONG rows, LONG cols, LONG i, Object **buttons) {
	static char text[32][4];
	struct TagItem *tags, *tag;
	Object *column, *button;
	LONG j;
	
	tags = AllocVec(sizeof(struct TagItem) * (cols + 2), MEMF_CLEAR);
	if (!tags) {
		return NULL;
	}
	
	tag = tags;
	tag->ti_Tag = MUIA_Group_Horiz;
	tag->ti_Data = FALSE;
	tag++;
	
	for (j = 0; j < rows; j++, i += cols) {
		snprintf(text[i], 4, "%d", i+1);
		button = TextObject,
			ButtonFrame,
			MUIA_InputMode,		MUIV_InputMode_RelVerify,
			MUIA_Text_Contents,	text[i],
			MUIA_Text_PreParse,	"\33c",
		End;
		if (buttons) buttons[i] = button;
		if (button) set(button, MUIA_Disabled, TRUE);
		tag->ti_Tag = MUIA_Group_Child;
		tag->ti_Data = (IPTR)button;
		tag++;
	}
	
	tag->ti_Tag = TAG_END;
	column = MUI_NewObjectA(MUIC_Group, tags);
	FreeVec(tags);
	
	return column;
}

HOOKPROTO(SeekFunc, IPTR, APTR unused, IPTR *params) {
	Object **gad = (Object **)params[1];
	struct CDTOC *cd_toc = (struct CDTOC *)params[2];
	if (cd_toc->IsPlaying) {
		cd_toc->CurrentAddr = (ULONG)XGET(gad[GID_POSITION], MUIA_Slider_Level);
	}
	return 0;
}

HOOKPROTO(EjectFunc, IPTR, APTR unused, IPTR *params) {
	cd_io->io_Command = TD_EJECT;
	cd_io->io_Length = load_eject;
	DoIO((struct IORequest *)cd_io);
	load_eject ^= 1;
	return 0;
}

HOOKPROTO(StopFunc, IPTR, APTR unused, IPTR *params) {
	Object *wnd = (Object *)params[0];
	Object **gad = (Object **)params[1];
	struct CDTOC *cd_toc = (struct CDTOC *)params[2];
	if (cd_toc->IsPlaying) {
		cd_toc->IsPlaying = FALSE;
		cd_toc->IsPaused = FALSE;
		cd_toc->CurrentAddr = cd_toc->Tracks[cd_toc->CurrentTrack-1].StartAddr;
		UpdateGUI(wnd, gad, cd_toc);
	}
	return 0;
}

HOOKPROTO(PauseFunc, IPTR, APTR unused, IPTR *params) {
	Object *wnd = (Object *)params[0];
	Object **gad = (Object **)params[1];
	struct CDTOC *cd_toc = (struct CDTOC *)params[2];
	if (cd_toc->IsPlaying && !cd_toc->IsPaused) {
		cd_toc->IsPaused = TRUE;
		UpdateGUI(wnd, gad, cd_toc);
	}
	return 0;
}

HOOKPROTO(PrevFunc, IPTR, APTR unused, IPTR *params) {
	Object *wnd = (Object *)params[0];
	Object **gad = (Object **)params[1];
	struct CDTOC *cd_toc = (struct CDTOC *)params[2];
	if (cd_toc->CurrentTrack > cd_toc->FirstTrack) {
		cd_toc->CurrentTrack--;
		cd_toc->CurrentAddr = cd_toc->Tracks[cd_toc->CurrentTrack-1].StartAddr;
		cd_toc->IsPaused = FALSE;
		UpdateGUI(wnd, gad, cd_toc);
	}
	return 0;
}

HOOKPROTO(PlayFunc, IPTR, APTR unused, IPTR *params) {
	Object *wnd = (Object *)params[0];
	Object **gad = (Object **)params[1];
	struct CDTOC *cd_toc = (struct CDTOC *)params[2];
	if (!cd_toc->IsPlaying || cd_toc->IsPaused) {
		cd_toc->IsPlaying = TRUE;
		cd_toc->IsPaused = FALSE;
		UpdateGUI(wnd, gad, cd_toc);
	}
	return 0;
}

HOOKPROTO(NextFunc, IPTR, APTR unused, IPTR *params) {
	Object *wnd = (Object *)params[0];
	Object **gad = (Object **)params[1];
	struct CDTOC *cd_toc = (struct CDTOC *)params[2];
	if (cd_toc->CurrentTrack < cd_toc->LastTrack) {
		cd_toc->CurrentTrack++;
		cd_toc->CurrentAddr = cd_toc->Tracks[cd_toc->CurrentTrack-1].StartAddr;
		cd_toc->IsPaused = FALSE;
		UpdateGUI(wnd, gad, cd_toc);
	}
	return 0;
}

HOOKPROTO(TrackFunc, IPTR, APTR unused, IPTR *params) {
	Object *wnd = (Object *)params[0];
	Object **gad = (Object **)params[1];
	struct CDTOC *cd_toc = (struct CDTOC *)params[2];
	ULONG track = (ULONG)params[3];
	if (cd_toc->Tracks[track].IsAudio) {
		cd_toc->CurrentTrack = track + 1;
		cd_toc->CurrentAddr = cd_toc->Tracks[cd_toc->CurrentTrack-1].StartAddr;
		cd_toc->IsPlaying = TRUE;
		cd_toc->IsPaused = FALSE;
		UpdateGUI(wnd, gad, cd_toc);
	}
	return 0;
}
