#include <string.h>

bool AddPart(char* dirname, const char* filename, unsigned long size)
/*  FUNCTION
	AddPart() will add a file, directory or other path name to a
	directory path. It will take into account any pre-existing
	separator characters (':','/').

	If filename is a fully qualified path, then it will replace
	the current value of dirname.

	INPUTS
	dirname     -   the path to add the new path to
	filename    -   the path you wish added
	size        -   The size of the dirname buffer (must NOT be 0)

	RESULT
	Non-zero if everything succeeded, FALSE if the buffer would have
	overflowed.

	If the buffer would have overflowed, then dirname will not have
	been changed.

	NOTES

	EXAMPLE
	UBYTE buffer[80];
	buffer[0]='\0';
	AddPart(buffer, "Work:", 80);
	AddPart(buffer, "Programming/Include/exec", 80);

	FPuts(Output(), buffer);
	--> Work:Programming/Include/exec

	AddPart(buffer, "/graphics", 80);

	FPuts(Output(), buffer);
	--> Work:Programming/Include/graphics

	AddPart(buffer, "gfxmacros.h", 80);
	FPuts(Output(), buffer);
	--> Work:Programming/Include/graphics/gfxmacros.h

	BUGS

	SEE ALSO
	FilePart(), PathPart()

	INTERNALS

*****************************************************************************/
{
	char *stringpos;
	long stringlen;
	int mustaddslash = 0;
	
	stringpos = strchr(filename, ':');
	if (stringpos)
	{
		if (stringpos == (char *)filename)
		{
			/* The first char of filename is a ':' */
			
			/* Find dirname position to where we copy filename later */
			
			stringpos = strchr(dirname, ':');
			if (!stringpos) stringpos = dirname;
		}	
		else
		{
			/* Filename is a fully qualified path including
		volume/device name -> replace dirname with filename */
			
			stringpos = dirname;
		}
	}
	else
	{
		/* filename did not contain ':' */
		
		stringlen = strlen(dirname);
		stringpos = dirname + stringlen;
		
		/* Check if we must add a slash */
		
		if (stringlen > 0)
		{
			if ((stringpos[-1] != ':') && (stringpos[-1] != '/'))
			mustaddslash = 1;
		}
	}
	
	/* Check if dirname buffer is big enough */
	
	stringlen = ((long)(stringpos - (char *)dirname)) + strlen(filename) + 1 + mustaddslash;
	if (stringlen <= size)
	{
		if (mustaddslash) *stringpos++ = '/';
		strcpy(stringpos, filename);
		return true;
	}
	else
	{
		return false;
	}
} /* AddPart */
