/*
------------------------------------------------------------
	MicroCon - A tiny, simple SDL text console
------------------------------------------------------------
 * (C) David Olofson <david@olofson.net>, 2004
 *
 * This software is released under the terms of the LGPL.
 */

#include <stdlib.h>
#include "ucon.h"

UCON_console *ucon_open(int w, int h, UCON_drawc_cb drawc)
{
	int x, y;
	UCON_console *uc = (UCON_console *)calloc(1, sizeof(UCON_console));
	if(!uc)
		return NULL;

	uc->w = w;
	uc->h = h;
	uc->drawc = drawc;
	uc->buf = (UCON_char *)malloc(uc->w * uc->h * sizeof(UCON_char));
	if(!uc->buf)
	{
		ucon_close(uc);
		return NULL;
	}
	for(y = 0; y < uc->h; ++y)
		for(x = 0; x < uc->w; ++x)
		{
			UCON_char *c = &uc->buf[y * uc->w + x];
			c->c = ' ';
			c->marks = UCON_MARK_CHANGED;
		}
	return uc;
}


void ucon_close(UCON_console *uc)
{
	free(uc->buf);
	free(uc);
}


void ucon_render(UCON_console *uc, int refresh)
{
	int x, y;
	if(refresh)
		for(y = 0; y < uc->h; ++y)
			for(x = 0; x < uc->w; ++x)
				uc->buf[y * uc->w + x].marks |=
						UCON_MARK_CHANGED;
	for(y = 0; y < uc->h; ++y)
		for(x = 0; x < uc->w; ++x)
		{
			int marks;
			UCON_char *c = &uc->buf[y * uc->w + x];
			if(!(c->marks & UCON_MARK_CHANGED))
				continue;
			c->marks &= ~UCON_MARK_CHANGED;
			if((x == uc->cx) && (y == uc->cy))
			{
				marks = c->marks;
				c->marks |= UCON_MARK_CURSOR;
				uc->drawc(uc, x, y, c);
				c->marks = marks;
			}
			else
				uc->drawc(uc, x, y, c);
		}
	uc->changed = 0;
}


static __inline__ int inside(UCON_console *uc, int x, int y)
{
	if(x < 0)
		return 0;
	if(y < 0)
		return 0;
	if(x >= uc->w)
		return 0;
	if(y >= uc->h)
		return 0;
	return 1;
}


void ucon_setc(UCON_console *uc, int x, int y, int c, int marks)
{
	UCON_char *ucc = &uc->buf[y * uc->w + x];
	marks &= UCON_MARK__PUBLIC;
	if(!(ucc->marks & UCON_MARK_CHANGED))
		if((c == ucc->c) && (marks == (ucc->marks & UCON_MARK__PUBLIC)))
			return;
	ucc->c = c;
	ucc->marks &= UCON_MARK__PRIVATE;
	ucc->marks |= marks | UCON_MARK_CHANGED;
	uc->changed = 1;
}


void ucon_scroll(UCON_console *uc)
{
	int x, y;
	for(y = 0; y < uc->h - 1; ++y)
		for(x = 0; x < uc->w; ++x)
		{
			UCON_char *c = &uc->buf[(y + 1) * uc->w + x];
			ucon_setc(uc, x, y, c->c, c->marks);
		}
	for(x = 0; x < uc->w; ++x)
		ucon_setc(uc, x, uc->h - 1, ' ', uc->mark);
}


void ucon_locate(UCON_console *uc, int x, int y)
{
	if((uc->cx == x) && (uc->cy == y))
		return;

	if(inside(uc, uc->cx, uc->cy))
	{
		uc->buf[uc->cy * uc->w + uc->cx].marks |= UCON_MARK_CHANGED;
		uc->changed = 1;
	}
	uc->cx = x;
	uc->cy = y;
	if(inside(uc, x, y))
	{
		uc->buf[y * uc->w + x].marks |= UCON_MARK_CHANGED;
		uc->changed = 1;
	}
}


void ucon_putc(UCON_console *uc, int c)
{
	int cx = uc->cx;
	int cy = uc->cy;
	switch(c)
	{
	  case 0:
		return;
	  case 10:
		while(cx < uc->w)
			ucon_setc(uc, cx++, cy, ' ', uc->mark);
		cx = 0;
		++cy;
		if(cy >= uc->h)
		{
			--cy;
			ucon_scroll(uc);
		}
		ucon_locate(uc, cx, cy);
		break;
	  case 13:
		break;
	  default:
		ucon_setc(uc, uc->cx, uc->cy, c, uc->mark);
		++cx;
		if(cx >= uc->w)
		{
			cx = 0;
			++cy;
			if(cy >= uc->h)
			{
				--cy;
				ucon_scroll(uc);
			}
		}
		ucon_locate(uc, cx, cy);
		break;
	}
}


void ucon_puts(UCON_console *uc, const char *s)
{
	int c;
	while((c = *(unsigned char *)s++))
		ucon_putc(uc, c);
}
