/*
------------------------------------------------------------
    SDLCalc - SDL Infix Calculator with Scrollback
------------------------------------------------------------
 * (C) David Olofson <david@olofson.net> 2004
 *
 * This software is released under the terms of the GPL.
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "SDL.h"
#include "SDL_image.h"
#ifdef	HAVE_OPENGL
#include "SDL_opengl.h"
#endif

#include "ucon.h"
#include "calc.h"

/* Text and console stuff */
#define	ED_LENGTH	256
#define	ED_HISTORY	32


/*------------------------------------------------
	Common stuff
------------------------------------------------*/
UCON_drawc_cb video_drawc;
Uint32 (*video_map_rgb)(Uint8 r, Uint8 g, Uint8 b);
void (*video_fillrect)(SDL_Rect *r, Uint32 color);
void (*video_update)(void);
void (*video_close)(void);

int		scale = -1;		/* Screen scale factor */

int		console_w = 55;		/* User selected console size */
int		console_h = 25;

int		font_w, font_h;		/* Calculated from skin size */

const char	*skin = "amber12x16.png";

UCON_console *console;

int do_exit = 0;

char edbuf[ED_HISTORY][ED_LENGTH];
int edpos, edhistory;

Uint32 cursor_color, edit_color, msg_color, result_color, bg_color;


/*------------------------------------------------
	Dirtyrect management
------------------------------------------------*/
SDL_Rect	*dirtytab = NULL;

static inline void dirty(int x, int y)
{
	if(dirtytab[y].w)
	{
		if(x < dirtytab[y].x)
			dirtytab[y].x = x;
		else if(x >= dirtytab[y].x + dirtytab[y].w)
			dirtytab[y].w = x - dirtytab[y].x + 1;
	}
	else
	{
		dirtytab[y].x = x;
		dirtytab[y].w = 1;
	}
}


/*------------------------------------------------
	SDL rendering backend
------------------------------------------------*/

int		flags = 0;		/* SDL video flags */
int		bpp = 0;		/* Preferred screen bpp */

SDL_Surface *screen = NULL;
SDL_Surface *font = NULL;

void drawc_sdl(UCON_console *uc, int x, int y, UCON_char *c)
{
	Uint32 color;
	SDL_Rect source_rect, dest_rect;
	if(!font)
		return;
	source_rect.x = c->c % 16 * font_w;
	source_rect.y = c->c / 16 * font_h;
	source_rect.w = font_w;
	source_rect.h = font_h;
	dest_rect.x = x * font_w;
	dest_rect.y = y * font_h;
	dest_rect.w = font_w;
	dest_rect.h = font_h;

	if(c->marks & UCON_MARK_CURSOR)
		color = cursor_color;
	else if(y >= uc->h - 1)
		color = edit_color;
	else if(c->marks & MARK_MESSAGE)
		color = msg_color;
	else if(c->marks & MARK_RESULT)
		color = result_color;
	else
		color = bg_color;
	SDL_FillRect(screen, &dest_rect, color);

	SDL_BlitSurface(font, &source_rect, screen, &dest_rect);
	dirty(x, y);
}

Uint32 map_rgb_sdl(Uint8 r, Uint8 g, Uint8 b)
{
	return SDL_MapRGB(screen->format, r, g, b);
}

void fillrect_sdl(SDL_Rect *r, Uint32 color)
{
	SDL_FillRect(screen, r, color);
}

void update_sdl(void)
{
	int y;
	int rects = 0;
	for(y = 0; y < console->h; ++y)
		if(dirtytab[y].w)
		{
			dirtytab[rects].x = dirtytab[y].x * font_w;
			dirtytab[rects].y = y * font_h;
			dirtytab[rects].w = dirtytab[y].w * font_w;
			dirtytab[rects].h = font_h;
			++rects;
		}
	if(rects)
		SDL_UpdateRects(screen, rects, dirtytab);
	for(y = 0; y < console->h; ++y)
		dirtytab[y].w = 0;
}

void close_sdl(void)
{
	if(font)
		SDL_FreeSurface(font);
	SDL_Quit();
}

int init_sdl(void)
{
	SDL_Surface *tmp = NULL;
	video_drawc = drawc_sdl;
	video_map_rgb = map_rgb_sdl;
	video_fillrect = fillrect_sdl;
	video_update = update_sdl;
	video_close = close_sdl;

	tmp = IMG_Load(skin);
	if(!tmp)
	{
		fprintf(stderr, "Could not load font!\n");
		return -1;
	}
	font_w = tmp->w / 16;
	font_h = tmp->h / 16;

	screen = SDL_SetVideoMode(console_w * font_w, console_h * font_h,
			bpp, flags);
	if(!screen)
	{
		fprintf(stderr, "Failed to open screen!\n");
		SDL_FreeSurface(tmp);
		exit(-1);
	}

	if(tmp->format->Amask)
		font = SDL_DisplayFormatAlpha(tmp);
	else
		font = SDL_DisplayFormat(tmp);
	if(!font)
	{
		fprintf(stderr, "Could not convert font!\n");
		return -1;
	}
	SDL_FreeSurface(tmp);
	return 0;
}


#ifdef	HAVE_OPENGL
/*----------------------------------------------------------
	OpenGL interface
----------------------------------------------------------*/

static struct
{
	void	(*Begin)(GLenum);
	void	(*BindTexture)(GLenum, GLuint);
	void	(*BlendFunc)(GLenum, GLenum);
	void	(*Color4ub)(GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha);
	void	(*Color3ub)(GLubyte red, GLubyte green, GLubyte blue);
	void	(*DeleteTextures)(GLsizei n, const GLuint *textures);
	void	(*Disable)(GLenum cap);
	void	(*Enable)(GLenum cap);
	void	(*End)(void);
	void	(*Flush)(void);
	void	(*GenTextures)(GLsizei n, GLuint *textures);
	GLenum	(*GetError)(void);
	void	(*GetIntegerv)(GLenum pname, GLint *params);
	void	(*LoadIdentity)(void);
	void	(*MatrixMode)(GLenum mode);
	void	(*Ortho)(GLdouble left, GLdouble right, GLdouble bottom,
			GLdouble top, GLdouble zNear, GLdouble zFar);
	void	(*PixelStorei)(GLenum pname, GLint param);
	void	(*ReadPixels)(GLint x, GLint y, GLsizei width, GLsizei height,
			GLenum format, GLenum type, GLvoid *pixels);
	void	(*TexCoord2f)(GLfloat s, GLfloat t);
	void	(*TexImage2D)(GLenum target, GLint level, GLint internalformat,
			GLsizei width, GLsizei height, GLint border,
			GLenum format, GLenum type, const GLvoid *pixels);
	void	(*TexParameteri)(GLenum target, GLenum pname, GLint param);
	void	(*TexSubImage2D)(GLenum target, GLint level,
			GLint xoffset, GLint yoffset, GLsizei width, GLsizei height,
			GLenum format, GLenum type, const GLvoid *pixels);
	void	(*Translatef)(GLfloat x, GLfloat y, GLfloat z);
	void	(*Vertex2f)(GLfloat x, GLfloat y);
	void	(*Vertex2i)(GLint x, GLint y);
	void	(*Viewport)(GLint x, GLint y, GLsizei width, GLsizei height);
} gl;


static int GetGL(void)
{
	int i;
	struct
	{
		const char	*name;
		void		**fn;
	} glfuncs[] = {
		{"glBegin", (void *)&gl.Begin },
		{"glBindTexture", (void *)&gl.BindTexture },
		{"glBlendFunc", (void *)&gl.BlendFunc },
		{"glColor4ub", (void *)&gl.Color4ub },
		{"glColor3ub", (void *)&gl.Color3ub },
		{"glDeleteTextures", (void *)&gl.DeleteTextures },
		{"glDisable", (void *)&gl.Disable },
		{"glEnable", (void *)&gl.Enable },
		{"glEnd", (void *)&gl.End },
		{"glFlush", (void *)&gl.Flush },
		{"glGenTextures", (void *)&gl.GenTextures },
		{"glGetError", (void *)&gl.GetError },
		{"glGetIntegerv", (void *)&gl.GetIntegerv },
		{"glLoadIdentity", (void *)&gl.LoadIdentity },
		{"glMatrixMode", (void *)&gl.MatrixMode },
		{"glOrtho", (void *)&gl.Ortho },
		{"glPixelStorei", (void *)&gl.PixelStorei },
		{"glReadPixels", (void *)&gl.ReadPixels },
		{"glTexCoord2f", (void *)&gl.TexCoord2f },
		{"glTexImage2D", (void *)&gl.TexImage2D },
		{"glTexParameteri", (void *)&gl.TexParameteri },
		{"glTexSubImage2D", (void *)&gl.TexSubImage2D },
		{"glTranslatef", (void *)&gl.Translatef },
		{"glVertex2f", (void *)&gl.Vertex2f },
		{"glVertex2i", (void *)&gl.Vertex2i },
		{"glViewport", (void *)&gl.Viewport },
		{NULL, NULL }
	};
	for(i = 0; glfuncs[i].name; ++i)
	{
		*glfuncs[i].fn = SDL_GL_GetProcAddress(glfuncs[i].name);
		if(!*glfuncs[i].fn)
			return -1;
	}
	return 0;
}


static int LoadGL(void)
{
	if(GetGL() < 0)
	{
		SDL_GL_LoadLibrary(NULL);
		if(GetGL() < 0)
			return -1;
	}
	return 0;
}


/*------------------------------------------------
	OpenGL rendering backend
------------------------------------------------*/
GLint gl_font;
int gl_texsize;

/* OpenGL "state optimizer" hack from glSDL */
static struct
{
	int	do_blend;
	int	do_texture;
	GLint	texture;
	GLenum	sfactor, dfactor;
} glstate;

static void gl_reset(void)
{
	glstate.do_blend = -1;
	glstate.do_blend = -1;
	glstate.texture = -1;
	glstate.sfactor = 0xffffffff;
	glstate.dfactor = 0xffffffff;
}

static __inline__ void gl_do_blend(int on)
{
	if(glstate.do_blend == on)
		return;

	if(on)
		gl.Enable(GL_BLEND);
	else
		gl.Disable(GL_BLEND);
	glstate.do_blend = on;
}

static __inline__ void gl_do_texture(int on)
{
	if(glstate.do_texture == on)
		return;

	if(on)
		gl.Enable(GL_TEXTURE_2D);
	else
		gl.Disable(GL_TEXTURE_2D);
	glstate.do_texture = on;
}

static __inline__ void gl_blendfunc(GLenum sfactor, GLenum dfactor)
{
	if((sfactor == glstate.sfactor) && (dfactor == glstate.dfactor))
		return;

	gl.BlendFunc(sfactor, dfactor);

	glstate.sfactor = sfactor;
	glstate.dfactor = dfactor;
}

static __inline__ void gl_texture(GLuint tx)
{
	if(tx == glstate.texture)
		return;

	gl.BindTexture(GL_TEXTURE_2D, tx);
	glstate.texture = tx;
}


void drawc_gl(UCON_console *uc, int x, int y, UCON_char *c)
{
	Uint32 color;
	float tx1 = (float)(c->c % 16) * font_w / gl_texsize;
	float ty1 = (float)(c->c / 16) * font_h / gl_texsize;
	float tx2 = tx1 + ((float)font_w / gl_texsize);
	float ty2 = ty1 + ((float)font_h / gl_texsize);
	float x1 = x * font_w;
	float y1 = y * font_h;
	float x2 = x1 + font_w;
	float y2 = y1 + font_h;

	if(c->marks & UCON_MARK_CURSOR)
		color = cursor_color;
	else if(y >= uc->h - 1)
		color = edit_color;
	else if(c->marks & MARK_MESSAGE)
		color = msg_color;
	else if(c->marks & MARK_RESULT)
		color = result_color;
	else
		color = bg_color;

	gl_do_texture(0);
	gl_do_blend(0);
	gl.Begin(GL_QUADS);
	gl.Color3ub((color>>16) & 255, (color>>8) & 255, color & 255);
	gl.Vertex2i(x1, y1);
	gl.Vertex2i(x2, y1);
	gl.Vertex2i(x2, y2);
	gl.Vertex2i(x1, y2);
	gl.End();

	gl_do_texture(1);
	gl_blendfunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	gl_do_blend(1);
	gl_texture(gl_font);
	gl.Begin(GL_QUADS);
	gl.Color4ub(255, 255, 255, 255);
	gl.TexCoord2f(tx1, ty1); gl.Vertex2f(x1, y1);
	gl.TexCoord2f(tx2, ty1); gl.Vertex2f(x2, y1);
	gl.TexCoord2f(tx2, ty2); gl.Vertex2f(x2, y2);
	gl.TexCoord2f(tx1, ty2); gl.Vertex2f(x1, y2);
	gl.End();
	dirty(x, y);
}


Uint32 map_rgb_gl(Uint8 r, Uint8 g, Uint8 b)
{
	return r<<16 | g<<8 | b;
}


void fillrect_gl(SDL_Rect *r, Uint32 color)
{
	SDL_Rect zr;
	int dx1, dy1, dx2, dy2;

	gl_do_texture(0);
	gl_do_blend(0);

	if(!r)
	{
		zr.x = 0;
		zr.y = 0;
		zr.w = screen->w;
		zr.h = screen->h;
		r = &zr;
	}

	dx1 = r->x;
	dy1 = r->y;
	dx2 = dx1 + r->w;
	dy2 = dy1 + r->h;

	gl.Begin(GL_QUADS);
	gl.Color3ub((color>>16) & 255, (color>>8) & 255, color & 255);
	gl.Vertex2i(dx1, dy1);
	gl.Vertex2i(dx2, dy1);
	gl.Vertex2i(dx2, dy2);
	gl.Vertex2i(dx1, dy2);
	gl.End();
}


void update_gl(void)
{
	SDL_GL_SwapBuffers();
}


void close_gl(void)
{
	gl.DeleteTextures(1, &gl_font);
	SDL_Quit();
}


int init_gl(void)
{
	SDL_Surface *tmp, *tmp2;

	GLint gl_doublebuf;
	GLint maxtexsize;

	video_drawc = drawc_gl;
	video_map_rgb = map_rgb_gl;
	video_fillrect = fillrect_gl;
	video_update = update_gl;
	video_close = close_gl;

	flags |= SDL_OPENGL;

	LoadGL();

	tmp = IMG_Load(skin);
	if(!tmp)
	{
		fprintf(stderr, "Could not load font!\n");
		SDL_Quit();
		return -1;
	}
	font_w = tmp->w / 16;
	font_h = tmp->h / 16;

	gl_doublebuf = flags & SDL_DOUBLEBUF;
	if(bpp == 15)
	{
		SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
		SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 5);
		SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);
	}
	if(bpp == 16)
	{
		SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
		SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 6);
		SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);
	}
	else if(bpp >= 24)
	{
		SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
		SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
		SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
	}
	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, gl_doublebuf);

	screen = SDL_SetVideoMode(
			console_w * font_w * scale,
			console_h * font_h * scale,
			bpp, flags);
	if(!screen)
	{
		fprintf(stderr, "Failed to open screen!\n");
		SDL_FreeSurface(tmp);
		return -1;
	}

	{
		int ts = 16;
		if(tmp->w > tmp->h)
			gl_texsize = tmp->w;
		else
			gl_texsize = tmp->h;
		while(ts < gl_texsize)
			ts <<= 1;
		gl_texsize = ts;
	}

	gl.GetIntegerv(GL_MAX_TEXTURE_SIZE, &maxtexsize);
	if(maxtexsize < gl_texsize)
	{
		fprintf(stderr, "Need at least %dx%d textures!\n",
				gl_texsize, gl_texsize);
		SDL_FreeSurface(tmp);
		SDL_Quit();
		return -1;
	}

	if(tmp->format->Amask)
		tmp2 = SDL_CreateRGBSurface(SDL_SWSURFACE,
				gl_texsize, gl_texsize, 32,
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
				0xff000000, 0xff0000, 0xff00, 0xff);
#else
				0xff, 0xff00, 0xff0000, 0xff000000);
#endif
	else
		tmp2 = SDL_CreateRGBSurface(SDL_SWSURFACE,
				gl_texsize, gl_texsize, 24,
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
				0x00ff0000, 0x0000ff00, 0x000000ff, 0);
#else
				0x000000ff, 0x0000ff00, 0x00ff0000, 0);
#endif
	if(!tmp2)
	{
		fprintf(stderr, "Failed to make texture from graphics!\n");
		SDL_FreeSurface(tmp);
		SDL_Quit();
		return -1;
	}

	SDL_SetAlpha(tmp, 0, SDL_ALPHA_OPAQUE);
	SDL_SetAlpha(tmp2, 0, SDL_ALPHA_OPAQUE);
	SDL_BlitSurface(tmp, NULL, tmp2, NULL);

	gl.GenTextures(1, &gl_font);
	gl.BindTexture(GL_TEXTURE_2D, gl_font);
	gl.PixelStorei(GL_UNPACK_ROW_LENGTH, tmp2->pitch /
			tmp2->format->BytesPerPixel);
	gl.TexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	gl.TexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	if(tmp->format->Amask)
		gl.TexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8,
				tmp2->w, tmp2->h, 0, GL_RGBA,
				GL_UNSIGNED_BYTE, tmp2->pixels);
	else
		gl.TexImage2D(GL_TEXTURE_2D, 0, GL_RGB8,
				tmp2->w, tmp2->h, 0, GL_RGB,
				GL_UNSIGNED_BYTE, tmp2->pixels);
	gl.Flush();
	SDL_FreeSurface(tmp2);
	SDL_FreeSurface(tmp);

	/*
	 * Set up OpenGL for 2D rendering.
	 */
	gl.Disable(GL_DEPTH_TEST);
	gl.Disable(GL_CULL_FACE);

	gl.Viewport(0, 0, screen->w, screen->h);
	gl.MatrixMode(GL_PROJECTION);
	gl.LoadIdentity();
	gl.Ortho(0, console_w * font_w, console_h * font_h, 0, -1.0, 1.0);

	gl.MatrixMode(GL_MODELVIEW);
	gl.LoadIdentity();
	gl.Translatef(0.0f, 0.0f, 0.0f);

	gl_reset();

	return 0;
}
#endif /* HAVE_OPENGL */


/*------------------------------------------------
	Editor
------------------------------------------------*/

static void feep(void)
{
}

static void ed_init(void)
{
	memset(edbuf, 0, sizeof(edbuf));
	edpos = 0;
	edhistory = 0;
}

static void ed_display(void)
{
	int x;
	for(x = 0; x < console->w && edbuf[0][x]; ++x)
		ucon_setc(console, x, console->h - 1,
				edbuf[0][x], console->mark);
	for( ; x < console->w; ++x)
		ucon_setc(console, x, console->h - 1, ' ', console->mark);
	ucon_locate(console, edpos, console->h - 1);
}

/* Push work buffer into history */
static void ed_push()
{
	int i;
	if(!edbuf[0][0])
		return;	/* No empty buffers, please! */

	for(i = ED_HISTORY-1; i > 0; --i)
		memcpy(edbuf[i], edbuf[i-1], sizeof(edbuf[0]));
}

/* Clear work buffer */
static void ed_clear()
{
	edbuf[0][0] = 0;
	edpos = 0;
}

/* Copy from history */
static void ed_history(int from)
{
	if(from)
	{
		memcpy(edbuf[0], edbuf[from], sizeof(edbuf[0]));
		edpos = strlen(edbuf[0]);
	}
	else
		ed_clear();
}


static void edkey(SDL_Event * ev)
{
	char *s;
	int i;
	switch(ev->key.keysym.sym)
	{
	  case SDLK_UP:
		if((edhistory < ED_HISTORY) && edbuf[edhistory+1][0])
			ed_history(++edhistory);
		else
			feep();
		break;
	  case SDLK_DOWN:
		if(edhistory)
			ed_history(--edhistory);
		else
			feep();
		break;
	  case SDLK_LEFT:
		if(edpos)
			--edpos;
		else
			feep();
		break;
	  case SDLK_RIGHT:
		if((edpos < 254) && edbuf[0][edpos])
			++edpos;
		else
			feep();
		break;
	  case SDLK_HOME:
		edpos = 0;
		break;
	  case SDLK_END:
		edpos = strlen(edbuf[0]);
		break;
	  case SDLK_BACKSPACE:
		if(edpos < 1)
		{
			feep();
			break;
		}
		--edpos;
		s = edbuf[0] + edpos;
		while(*s++)
			s[-1] = *s;
		break;
	  case SDLK_DELETE:
		s = edbuf[0] + edpos;
		while(*s++)
			s[-1] = *s;
		break;
	  case SDLK_RETURN:
	  case SDLK_KP_ENTER:
		if(edbuf[0][0])
		{
			edpos = 0;
			ed_display();
			ucon_scroll(console);
			ed_push();
			ed_clear();
			if(calc_eval(edbuf[1]) == 1)
				do_exit = 1;
			edhistory = 0;
		}
		else
			ucon_puts(console, " - - - - - - - - - - - -\n");
		break;
	  default:
		if(ev->key.keysym.unicode & 0xff80 ||
				ev->key.keysym.unicode < 32 ||
				ev->key.keysym.unicode > 126 ||
				edpos >= 255 ||
				strlen(edbuf[0]) > 254)
		{
			feep();
			break;
		}
		for(i = 254; i > edpos; --i)
			edbuf[0][i] = edbuf[0][i-1];
		edbuf[0][edpos++] = ev->key.keysym.unicode;
		break;
	}
}


void calc_run(void)
{
	int update = 1;
	edhistory = 0;
	edpos = 0;
	ucon_locate(console, edpos, console->h - 1);
	ed_clear();
	ed_display();
	while(!do_exit)
	{
		SDL_Event event;
		while(!update)
		{
			if(!SDL_WaitEvent(&event))
			{
				fprintf(stderr, "Error in SDL_WaitEvent()!\n");
				return;
			}
			switch(event.type)
			{
			  case SDL_KEYDOWN:
				edkey(&event);
				ed_display();
				update = console->changed;
				break;
			  case SDL_QUIT:
			  	do_exit = 1;
				update = 1;
				break;
			  case SDL_VIDEOEXPOSE:
				update = 2;
				break;
			  default:
				break;
			}
		}
		ucon_render(console, update - 1);
		video_update();
		update = 0;
	}
}


/*------------------------------------------------
	main
------------------------------------------------*/

int main(int argc, char* argv[])
{
	int res, i;
#ifdef	HAVE_OPENGL
	int use_gl = 0;
#endif
	SDL_Init(SDL_INIT_VIDEO);
	atexit(SDL_Quit);

	for(i = 1; i < argc; ++i)
	{
		if(strncmp(argv[i], "-skin", 5) == 0)
		{
			if(i + 1 < argc)
				skin = argv[++i];
			else
			{
				fprintf(stderr, "-skin option needs"
						" a file name!\n");
				exit(1);
			}
		}
#ifdef	HAVE_OPENGL
		else if(strncmp(argv[i], "-gl", 3) == 0)
			use_gl = 1;
		else if(strncmp(argv[i], "-s", 2) == 0)
			scale = atoi(&argv[i][2]);
#endif
		else if(strncmp(argv[i], "-d", 2) == 0)
			flags |= SDL_DOUBLEBUF;
		else if(strncmp(argv[i], "-f", 2) == 0)
			flags |= SDL_FULLSCREEN;
		else if(strncmp(argv[i], "-w", 2) == 0)
			console_w = atoi(&argv[i][2]);
		else if(strncmp(argv[i], "-h", 2) == 0)
			console_h = atoi(&argv[i][2]);
		else if(strncmp(argv[i], "-b", 2) == 0)
			bpp = atoi(&argv[i][2]);
		else
		{
			fprintf(stderr, "Unknown switch '%s'!\n", argv[i]);
			exit(1);
		}
	}
	if(scale < 0)
		scale = 1;

#ifdef	HAVE_OPENGL
	if(use_gl)
		res = init_gl();
	else
#endif
		res = init_sdl();
	if(res < 0)
		exit(-1);

	cursor_color = video_map_rgb(64, 128, 64);
	edit_color = video_map_rgb(0, 48, 48);
	msg_color = video_map_rgb(32, 32, 64);
	result_color = video_map_rgb(64, 32, 32);
	bg_color = video_map_rgb(0, 0, 0);

	dirtytab = calloc(console_h, sizeof(SDL_Rect));
	if(!dirtytab)
	{
		fprintf(stderr, "Could not allocate dirtyrect table!\n");
		exit(1);
	}

	console = ucon_open(console_w, console_h, video_drawc);
	if(!console)
	{
		fprintf(stderr, "Could not open console!\n");
		exit(1);
	}
	calc_open(console);

	ed_init();

	SDL_WM_SetCaption("SDLCalc - SDL Infix Calculator", "SDLCalc");
	SDL_EnableUNICODE(1);
	SDL_EnableKeyRepeat(200, 25);

	calc_run();

	calc_close();
	ucon_close(console);
	free(dirtytab);
	video_close();
	exit(0);
}
