%{
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "calc.h"
#include "ucon.h"

#define	CALC_PI	3.1415926535897932384626433832795029L
#define	CALC_E	2.7182818284590452353602874713526625L

#ifndef HAVE_SNPRINTF
#define	snprintf _snprintf
#endif

int yylex();
int yyerror(const char *s);
static UCON_console *uc;
static int result;
static void checkdef(symrec *sr);
%}
%union {
	double	val;	/* For returning numbers */
	symrec	*tptr;	/* For returning symbol-table pointers */
}

%expect 1	/* '-' operator vs unary minus... */

%token	<val>	NUM	/* Simple double precision number */
%token	<tptr>	VAR	/* Variable */
%token	<tptr>	CONST	/* Constant */
%token	<tptr>	FNCT	/* Function */
%token	<tptr>	CMD	/* Command */
%type	<val>	exp

%token	POW ROOT LEXERR
%token	INC DEC DIV MUL MOD

%right '=' INC DEC DIV MUL MOD
%left '-' '+'
%left '*' '/' '%'
%left NEG	/* Negation--unary minus */
%left POW	/* Exponentiation (left, like my Casio calculator...) */
%left ROOT	/* Root */
%left FNCT	/* Functions */

/* Grammar follows */
%% 
input:	/* empty */
	| input line
	;

line:	
	  CMD 		{ (*($1->value.fnctptr))(0); }
	| exp
	{
		char buf[128];
		snprintf(buf, sizeof(buf), "    %.10g\n", $1);
		uc->mark = MARK_RESULT;
		ucon_puts(uc, buf);
		uc->mark = 0;
	}
	| error	'\n'	{ yyerrok; }
	;

exp:	  NUM			{ $$ = $1; }
	| CONST			{ $$ = $1->value.var; }
	| VAR '=' exp
	{
		$$ = $3;
		$1->value.var = $3;
		$1->defined = 1;
	}
	| VAR			{ checkdef($1); $$ = $1->value.var; }
	| VAR INC exp
	{
		checkdef($1);
		$1->value.var += $3;
		$1->defined = 1;
		$$ = $1->value.var;
	}
	| VAR DEC exp
	{
		checkdef($1);
		$1->value.var -= $3;
		$1->defined = 1;
		$$ = $1->value.var;
	}
	| VAR MUL exp
	{
		checkdef($1);
		$1->value.var *= $3;
		$1->defined = 1;
		$$ = $1->value.var;
	}
	| VAR DIV exp
	{
		checkdef($1);
		$1->value.var /= $3;
		$1->defined = 1;
		$$ = $1->value.var;
	}
	| VAR MOD exp
	{
		checkdef($1);
		$1->value.var = fmod($1->value.var, $3);
		$1->defined = 1;
		$$ = $1->value.var;
	}
	| FNCT exp 		{ $$ = (*($1->value.fnctptr))($2); }
	| exp '+' exp		{ $$ = $1 + $3; }
	| exp '-' exp		{ $$ = $1 - $3; }
	| exp '*' exp		{ $$ = $1 * $3; }
	| exp '/' exp		{ $$ = $1 / $3; }
	| exp '%' exp		{ $$ = fmod($1, $3); }
	| '-' exp  %prec NEG	{ $$ = -$2; }
	| exp POW exp		{ $$ = pow($1, $3); }
	| exp ROOT exp		{ $$ = pow($3, 1/($1)); }
	| '(' exp ')'		{ $$ = $2; }
	;
%%                       
/* End of grammar */

int yyerror(const char *s)	/* Called by yyparse on error */
{
	ucon_puts(uc, s);
	ucon_putc(uc, 10);
	ucon_putc(uc, 10);
	return 0;
}

struct init
{
	const char *fname;
	double (*fnct)(double a);
};

static double cf = 1.0;

double w_sin(double a)
{
	return sin(a * cf);
}

double w_cos(double a)
{
	return cos(a * cf);
}

double w_tan(double a)
{
	return tan(a * cf);
}

double w_asin(double a)
{
	return asin(a) / cf;
}

double w_acos(double a)
{
	return acos(a) / cf;
}

double w_atan(double a)
{
	return atan(a) / cf;
}

#ifndef cbrt
double cbrt(double a)
{
	return pow(a, 1.0L / 3.0L);
}
#endif

double square(double a)
{
	return a * a;
}

double cube(double a)
{
	return a * a * a;
}

struct init arith_fncts[] = {
	{"sin", w_sin},
	{"cos", w_cos},
	{"tan", w_tan},
	{"asin", w_sin},
	{"acos", w_cos},
	{"atan", w_atan},
	{"ln", log},
	{"exp", exp},
	{"sqrt", sqrt},
	{"cbrt", cbrt},
	{"sq", square},
	{"square", square},
	{"cb", cube},
	{"cube", cube},
	{0, 0}
};

/* The symbol table: a chain of `struct symrec'.  */
symrec *sym_table = (symrec *)0;

double use_deg(double a)
{
	uc->mark = MARK_MESSAGE;
	ucon_puts(uc, "    Trig mode: DEG\n");
	uc->mark = 0;
	cf = CALC_PI / 180.0;
	return cf;
}

double use_rad(double a)
{
	uc->mark = MARK_MESSAGE;
	ucon_puts(uc, "    Trig mode: RAD\n");
	uc->mark = 0;
	cf = 1.0;
	return cf;
}


double quit(double a)
{
	uc->mark = MARK_MESSAGE;
	ucon_puts(uc, "    Bye!\n");
	uc->mark = 0;
	result = 1;
	return 0;
}


double help(double a)
{
	uc->mark = MARK_MESSAGE;
	ucon_puts(uc, "\n");
	ucon_puts(uc, " --- SDLCalc - a scientific style infix calculator ---\n");
	ucon_puts(uc, "\n");
	ucon_puts(uc, "    Math operators:             Math Functions:\n");
	ucon_puts(uc, "        = (assignment)              sin (sine)\n");
	ucon_puts(uc, "   +   += (addition)                cos (cosine)\n");
	ucon_puts(uc, "   -   -= (subtraction)             tan (tangent)\n");
	ucon_puts(uc, "   *   *= (multiplication)         asin (arc sine)\n");
	ucon_puts(uc, "   /   /= (division)               acos (arc cosine)\n");
	ucon_puts(uc, "   %   %= (modulus)                atan (arc tangent)\n");
	ucon_puts(uc, "      pow (power)                    ln (n logarithm)\n");
	ucon_puts(uc, "  rt root (x'th root)               exp (e raised)\n");
	ucon_puts(uc, "                                   sqrt (square root)\n");
	ucon_puts(uc, "    Commands:                      cbrt (cube root)\n");
	ucon_puts(uc, "      deg (Switch to DEG)     sq square (square)\n");
	ucon_puts(uc, "      rad (Switch to RAD)     cb   cube (cube)\n");
	ucon_puts(uc, "     quit (Exit SDLCalc)\n");
	ucon_puts(uc, "     help (This page)        Constants:  pi  e\n");
	ucon_puts(uc, "\n");
	uc->mark = 0;
	return 0;
}


symrec *putsym(const char *sym_name, int sym_type)
{
	symrec *ptr;
	ptr = (symrec *)malloc(sizeof(symrec));
	ptr->name = (char *) malloc(strlen(sym_name) + 1);
	strcpy(ptr->name, sym_name);
	ptr->type = sym_type;
	ptr->defined = 0;
	ptr->value.var = 0;
	ptr->next = (struct symrec *)sym_table;
	sym_table = ptr;
	return ptr;
}


symrec *putcmd(const char *name, double (*cb)(double a))
{
	symrec *ptr = putsym(name, CMD);
	ptr->value.fnctptr = cb;
	return ptr;
}


symrec *putconst(const char *name, double v)
{
	symrec *ptr = putsym(name, CONST);
	ptr->value.var = v;
	return ptr;
}


void init_table ()  /* puts arithmetic functions in table. */
{
	int i;
	symrec *ptr;
	for ( i = 0; arith_fncts[i].fname != 0; i++ )
	{
		ptr = putsym(arith_fncts[i].fname, FNCT);
		ptr->value.fnctptr = arith_fncts[i].fnct;
	}
	putcmd("deg", use_deg);
	putcmd("rad", use_rad);
	putcmd("quit", quit);
	putcmd("help", help);

	putsym("pow", POW);
	putsym("rt", ROOT);
	putsym("root", ROOT);

	putconst("pi", CALC_PI);
	putconst("e", CALC_E);
}


symrec *getsym (const char *sym_name)
{
	symrec *ptr;
	for (ptr = sym_table; ptr != (symrec *)0; ptr = (symrec *)ptr->next)
		if (strcmp(ptr->name,sym_name) == 0)
			return ptr;
	return 0;
}


static void checkdef(symrec *sr)
{
	if(!sr->defined)
	{
		char buf[128];
		snprintf(buf, sizeof(buf), "    Variable '%s' is not"
				" defined! (Evaluates to 0.)\n", sr->name);
		ucon_puts(uc, buf);
	}
}


static const char *buf;

int yylex()
{
	int c;

	/* Ignore whitespace, get first nonwhite character. */
	while((c = *buf++) == ' ' || c == '\t')
		;

	if(EOF == c)
		return 0;

	/* Char starts a number => parse the number. */
	if('.' == c || isdigit(c))
	{
		int count;
		--buf;
		if(sscanf(buf, "%lf%n", &yylval.val, &count) > 0)
			buf += count;
		else
		{
			++buf;
			return LEXERR;
		}
		return NUM;
	}

	/* Char starts an identifier => read the name. */
	if(isalpha(c))
	{
		symrec *s;
		static char *symbuf = 0;
		static int length = 0;
		int i;

		/* Initially make the buffer long enough
		   for a 40-character symbol name. */
		if( 0 == length )     
			length = 40, symbuf = (char *)malloc (length + 1);     

		i = 0;
		do
		{
			/* If buffer is full, make it bigger. */
			if( i == length )
			{
				length *= 2;
				symbuf = (char *)realloc (symbuf, length + 1);
			}
			/* Add this character to the buffer. */
			symbuf[i++] = c;
			/* Get another character. */
			c = *buf++;
		}
		while(c != EOF && (isalpha(c) || c == '.' || c == '_'));

		--buf;
		symbuf[i] = '\0';

		s = getsym(symbuf);
		if(!s)
			s = putsym(symbuf, VAR);
		yylval.tptr = s;
		return s->type;
	}

	if('?' == c)
	{
		yylval.tptr = getsym("help");
		return yylval.tptr->type;
	}

	/* Check for compound operators */
	if(*buf && ('=' == *buf))
	{
		++buf;
		switch(c)
		{
		  case '+':
			return INC;
		  case '-':
			return DEC;
		  case '/':
			return DIV;
		  case '*':
			return MUL;
		  case '%':
			return MOD;
		  default:
			--buf;
		}
	}

	/* Any other character is a token by itself. */
	return c;
}



int calc_open(UCON_console *console)
{
	uc = console;
	uc->cx = 0;
	uc->cy = uc->h - 1;
	uc->mark = MARK_MESSAGE;
	ucon_puts(uc,	"\n"
			"Dave's Sensible Infix Calculator\n");
	ucon_puts(uc,	"--------------------------------\n");
	ucon_puts(uc,	"          Version 0.0.1\n\n");
	ucon_puts(uc,	"Initializing...\n");
	init_table();
	ucon_puts(uc,	"    Ready.\n\n");
	uc->mark = 0;
	return 0;
}


int calc_eval(const char *b)
{
	result = 0;
	buf = b;
	yyparse();
	return result;
}


void calc_close(void)
{
}
