/*
 * Kuklomenos
 * Copyright (C) 2008 Martin Bays <mbays@sdf.lonestar.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 */

#include <vector>
#include <algorithm>
#include <cstdio>
#include <SDL/SDL.h>
#include <SDL/SDL_gfxPrimitives.h>

#include "state.h"
#include "shot.h"
#include "player.h"
#include "geom.h"
#include "gfx.h"
#include "random.h"
#include "settings.h"
#include "node.h"

GameState::GameState() :
    targettedNode(NULL), timeSinceLastInvader(0), mutilationWave(-1),
    extracted(0), you(), zoomdist(0), invaderRate(0), rating(0),
    extractMax(500), end(END_NOT)
{
    for (int i = 0; i < 6; i++)
    {
	NodeColour colour = (NodeColour)(i);
	if (i < 3)
	    nodes.push_back(
		    Node(RelPolarCoord(2+i*4.0/3, ARENA_RAD * 5/9),
			1, colour));
	else
	    nodes.push_back(
		    Node(RelPolarCoord(i*4.0/3, ARENA_RAD * 2/3),
			-1, colour));
    }
}

bool isNullInvp(Invader* p)
{
    return (p == NULL);
}

bool GameState::youHaveNode(NodeColour colour)
{
    for (std::vector<Node>::iterator it = nodes.begin();
	    it != nodes.end();
	    it++)
	if (it->status == NODEST_YOU && it->nodeColour == colour)
	    return true;
    return false;
}
bool GameState::youHaveShotNode(int type)
{
    switch (type)
    {
	case 0: return youHaveNode(NODEC_GREEN); break;
	case 1: return youHaveNode(NODEC_YELLOW); break;
	case 2: return youHaveNode(NODEC_RED); break;
	case 3: return youHaveNode(NODEC_BLUE); break;
	default: return false;
    }
}
bool GameState::evilHasNode(NodeColour colour)
{
    for (std::vector<Node>::iterator it = nodes.begin();
	    it != nodes.end();
	    it++)
	if (it->status == NODEST_EVIL && it->nodeColour == colour)
	    return true;
    return false;
}

int GameState::shotHeat(int type)
{
    int baseHeat;
    const bool super = youHaveShotNode(type);
    switch (type)
    {
	case 0: baseHeat = 4000; break;
	case 1: baseHeat = 7000; break;
	case 2: baseHeat = 10000; break;
	case 3: baseHeat = 30000; break;
	default: baseHeat = 0;
    }
    return super ? 7*baseHeat/10 : baseHeat;
}
int GameState::shotDelay(int type)
{
    int delay;
    const bool super = youHaveShotNode(type);
    switch (type)
    {
	case 0: delay = super ? 100 : 200; break;
	case 1: delay = super ? 250 : 300; break;
	case 2: delay = super ? 300 : 400; break;
	case 3: delay = super ? 7500 : 10000; break;
	default: delay = 0;
    }
    return delay;
}

void GameState::updateMutilation(int time)
{
    mutilationWave = std::max(0.0f,
	    mutilationWave - time*0.001f*ARENA_RAD/3);

    for (std::vector<Node>::iterator it = nodes.begin();
	    it != nodes.end();
	    it++)
	if (it->pos.dist >= mutilationWave-it->radius)
	    if (it->status != NODEST_DESTROYED)
		it->status = NODEST_NONE;

    for (std::vector<Invader*>::iterator it = invaders.begin();
	    it != invaders.end();
	    it++)
	if (((*it)->cpos() - ARENA_CENTRE).lengthsq() >=
		mutilationWave*mutilationWave)
	    (*it)->die();

    for (std::vector<Shot>::iterator it = shots.begin();
	    it != shots.end();
	    it++)
	if ((it->pos - ARENA_CENTRE).lengthsq() >=
		mutilationWave*mutilationWave)
	{
	    it->die();
	    deadShots = true;
	}

    while (!you.dead && mutilationWave <= you.radius())
    {
	you.shield -= 1;
	if (you.shield < 0)
	    you.dead = true;
	end = END_EXTRACTED;
    }
}

void GameState::update(int time)
{
    if (time <= 0)
	return;

    deadShots = false;

    updateObjects(time);
    if (!you.dead)
	handleInput(time);
    updateZoom(time);
    evilAI(time);
    cleanup();
}

void GameState::updateObjects(int time)
{
    you.update(time, youHaveNode(NODEC_CYAN));

    if (!end && you.dead)
	end = END_DEAD;

    int destroyedCount = 0;
    for (std::vector<Node>::iterator it = nodes.begin();
	    it != nodes.end();
	    it++)
    {
	it->update(time);

	if (it->status == NODEST_EVIL)
	{
	    it->extractionProgress += (evilHasNode(NODEC_CYAN) ? 2 :
		    1)*0.001*time;
	    while (it->extractionProgress > 1)
	    {
		it->extractionProgress--;
		extracted++;
		if (extracted > extractMax && mutilationWave == -1)
		// initiate wave of mutilation
		    mutilationWave = ARENA_RAD;
	    }
	}
	else if (it->status == NODEST_DESTROYED)
	    destroyedCount++;
    }
    if (destroyedCount >= 4 && end == END_NOT)
    {
	// you win - set invaders fleeing away
	end = END_WIN;
	for (std::vector<Invader*>::iterator it = invaders.begin();
		it != invaders.end();
		it++)
	    (*it)->fleeOnWin();
    }

    if (mutilationWave > -1)
    {
	updateMutilation(time);
    }

    std::vector<Invader*> topush;

    for (std::vector<Invader*>::iterator it = invaders.begin();
	    it != invaders.end();
	    it++)
    {
	Invader* inv = *it;
	inv->update(time);

	while (!inv->spawns.empty())
	{
	    Invader* spawned = inv->spawns.back();
	    inv->spawns.pop_back();
	    topush.push_back(spawned);
	}

	if (inv->hitsYou() &&
		inv->collObj().circleIntersects(ARENA_CENTRE, you.radius()))
	{
	    inv->die();
	    if (!you.dead)
	    {
		you.shield -= 1;
		if (you.shield < 0 && !settings.invuln)
		    you.dead = true;
	    }
	}

	// check for leaving arena
	if (((inv)->cpos() - ARENA_CENTRE).lengthsq() >=
		ARENA_RAD*ARENA_RAD)
	    inv->die();
    }

    // check for collisions between invaders
    // (note that this must come after updating all invaders, since we need
    // the collision objects to be set)
    for (std::vector<Invader*>::iterator it = invaders.begin();
	    it != invaders.end();
	    it++)
    {
	Invader* inv = *it;
	if (inv->hitsInvaders() > 0)
	{
	    Invader* hitInvader = NULL;
	    const float r = inv->hitsInvaders();
	    for (std::vector<Invader*>::iterator it2 = invaders.begin();
		    it2 != invaders.end();
		    it2++)
	    {
		if (*it2 == inv)
		    continue;
		if ((*it2)->collObj().circleIntersects(inv->cpos(), r))
		{
		    hitInvader = *it2;
		    break;
		}
	    }

	    if (hitInvader != NULL)
	    {
		int damage = hitInvader->die();
		if (hitInvader->dead())
		{
		    you.score += hitInvader->killScore();
		}
		inv->hit(damage);
	    }
	}
    }

    for (std::vector<Shot>::iterator it = shots.begin();
	    it != shots.end();
	    it++)
    {
	float hitTime = -1;
	Invader* hitInvader = NULL;

	for (std::vector<Invader*>::iterator invit = invaders.begin();
		invit != invaders.end();
		invit++)
	{
	    if (!(*invit)->hitsShots())
		continue;

	    RelCartCoord v = it->vel;
	    float t = (*invit)->collObj().pointHits(it->pos, v, time);
	    if (t >= 0 && (hitTime == -1 || t < hitTime))
	    {
		hitTime = t;
		hitInvader = *invit;
	    }
	}
	for (std::vector<Node>::iterator nodeit = nodes.begin();
		nodeit != nodes.end();
		nodeit++)
	{
	    if (nodeit->primed < 1)
		continue;

	    RelCartCoord v = it->vel;
	    float t = nodeit->collObj().pointHits(it->pos, v, time);
	    if (t >= 0 && (hitTime == -1 || t < hitTime))
	    {
		hitTime = t;
		hitInvader = &*nodeit;
	    }
	}

	if (hitInvader != NULL)
	{
	    int damage = hitInvader->hit(it->weight);
	    if (hitInvader->dead())
		you.score += hitInvader->killScore();
	    it->hit(damage);
	    if (Shot::is_dead(*it))
		deadShots = true;
	}

	it->update(time);
	RelCartCoord d = it->pos - ARENA_CENTRE;
	if (d.lengthsq() >= ARENA_RAD*ARENA_RAD)
	{
	    it->dead = 1;
	    deadShots = true;
	}
    }

    // add newly spawned invaders
    for (std::vector<Invader*>::iterator it = topush.begin();
	    it != topush.end();
	    it++)
    {
	invaders.push_back(*it);
    }
}

void GameState::handleInput(int time)
{
    Uint8* keystate = SDL_GetKeyState(NULL);

    // three keysets: dvorak vikeys, qwerty vikeys, cursor keys and numbers.
    // TODO: proper keybinding facilities
    bool keyRotLeft = (keystate[SDLK_h] || keystate[SDLK_h] || keystate[SDLK_LEFT]);
    bool keyRotRight = (keystate[SDLK_s] || keystate[SDLK_l] || keystate[SDLK_RIGHT]);
    bool keyDeAim = (keystate[SDLK_t] || keystate[SDLK_j] || keystate[SDLK_DOWN]);
    bool keyDeZoom = (keystate[SDLK_n] || keystate[SDLK_k] || keystate[SDLK_UP]);
    bool keyShoot1 = keystate[SDLK_1];
    bool keyShoot2 = keystate[SDLK_2];
    bool keyShoot3 = keystate[SDLK_3];
    bool keyShootPod = keystate[SDLK_4];

    if (keyRotLeft || keyRotRight)
    {
	if (keyRotLeft)
	    you.aim.angle += time*.015*you.aimAccuracy();
	else
	    you.aim.angle += time*-.015*you.aimAccuracy();
	you.aim.dist = std::max(AIM_MIN, (float)(you.aim.dist-time*.04));
    }
    if (keyDeAim)
	you.aim.dist = std::max(AIM_MIN, (float)(you.aim.dist-time*.1));
    if (keyDeZoom)
    {
	you.aim.dist = std::max(AIM_MIN, (float)(you.aim.dist-time*.04));
	zoomdist = std::max(ZOOM_MIN, (float)(zoomdist-time*.2));
    }
    if (!(keyDeAim || keyDeZoom || keyRotLeft || keyRotRight))
    {
	// exponential decay from AIM_MIN to AIM_MAX:
	// dist = max - (max-min)*2^{-time/rate}
	// so d(dist)/d(time) is proportional to (max-dist).
	const float aimRate = youHaveNode(NODEC_PURPLE) ? 0.0006 : 0.0004;
	you.aim.dist = std::min(AIM_MAX,
		(float)(you.aim.dist + (AIM_MAX-you.aim.dist)*time*aimRate));
    }

    float mintheta = 4;
    for (std::vector<Node>::iterator it = nodes.begin();
	    it != nodes.end();
	    it++)
    {
	Angle relAngle = you.aim.angle - it->pos.angle;
	const float dtheta = std::min(float(relAngle), 4.0f-relAngle);
	if (it->status != NODEST_DESTROYED &&
		dtheta < mintheta)
	{
	    mintheta = dtheta;
	    targettedNode = &*it;
	}
    }
    if (mintheta > 0.5)
	targettedNode = NULL;

    you.shootHeat = std::max(0, you.shootHeat - you.shootCoolrate*time);
    you.shootTimer -= time;
    you.podTimer -= time;

    if ((keyShoot1 || keyShoot2 || keyShoot3)
	    && you.shootTimer <= 0)
    {
	int weight=0;
	bool super;
	if (keyShoot3 &&
		you.shootHeat < you.shootMaxHeat - shotHeat(2))
	{
	    weight = 3;
	    super = youHaveNode(NODEC_RED);
	}
	else if (keyShoot2 &&
		you.shootHeat < you.shootMaxHeat - shotHeat(1))
	{
	    weight = 2;
	    super = youHaveNode(NODEC_YELLOW);
	}
	else if (keyShoot1 &&
		you.shootHeat < you.shootMaxHeat - shotHeat(0))
	{
	    weight = 1;
	    super = youHaveNode(NODEC_GREEN);
	}
	if (weight > 0)
	{
	    float noise = gaussian()*you.aimAccuracy();

	    Shot shot( ARENA_CENTRE,
		    RelPolarCoord(you.aim.angle+noise,
			0.1+0.05*(3-weight) + super*0.05),
		    weight, super);
	    // we're generous, and assume that the command to shoot was given
	    // just after the last update, and the shot was fired as soon as
	    // the heat became low enough:
	    shot.update(std::min(time,
			std::min(
			    (you.shootMaxHeat - you.shootHeat -
			     shotHeat(weight-1))/you.shootCoolrate,
			    -you.shootTimer)));

	    shots.push_back(shot);

	    you.shootHeat += shotHeat(weight-1);
	    you.shootTimer = shotDelay(weight-1);

	    //you.aim.angle += weight*noise/4;

	    you.aim.dist = std::max(AIM_MIN,
		    (float)(you.aim.dist-weight*6));
	}
    }
    if (keyShootPod && 
	    targettedNode != NULL &&
	    you.podTimer <= 0 &&
	    you.shootHeat < you.shootMaxHeat - shotHeat(3))
    {
	invaders.push_back(new CapturePod(targettedNode,
		    RelPolarCoord(you.aim.angle, 5),
		    youHaveNode(NODEC_BLUE)));
	you.podTimer = shotDelay(3);
	you.shootHeat += shotHeat(3);
	you.doneLaunchedPod = true;
    }
    
    you.shootTimer = std::max(0, you.shootTimer);
    you.podTimer = std::max(0, you.podTimer);
}

void GameState::updateZoom(int time)
{
    float targetZoomdist = you.aim.dist;
    if (targetZoomdist < 0)
	targetZoomdist = 0;
    if (targetZoomdist > ZOOMDIST_MAX)
	targetZoomdist = ZOOMDIST_MAX;
    if (end)
	targetZoomdist = 0;
    if (time >= 200)
	zoomdist = targetZoomdist;
    else
	zoomdist += (targetZoomdist-zoomdist)*time/200;
}

void GameState::evilAI(int time)
{
    timeSinceLastInvader += time;
    if (end != END_WIN && mutilationWave == -1 &&
	    timeSinceLastInvader >= invaderRate)
    {
	Invader* p_inv = NULL;
	while (p_inv == NULL)
	{
	    //int type = rani(5) == 0 ? 3 : rani(3);
	    int type = rani(4);
	    RelPolarCoord pos;
	    int ds;
	    switch (type)
	    {
		case 0: case 1: case 2:
		    {
			bool super = evilHasNode(
				type == 0 ? NODEC_RED :
				type == 1 ? NODEC_YELLOW :
				NODEC_GREEN);
			pos = RelPolarCoord(ranf()*4, ARENA_RAD-20);
			ds = super ? rani(9)-4 : rani(5)-2;
			switch (type)
			{
			    case 0: p_inv = new EggInvader(pos, ds, super);
				    break;
			    case 1: p_inv = new KamikazeInvader(pos, ds,
					    super); break;
			    case 2: p_inv = new SplittingInvader(pos, ds,
					    super); break;
			}
			break;
		    }
		case 3:
		    {
			std::vector<Node*> possibleTargets;
			for (std::vector<Node>::iterator it = nodes.begin();
				it != nodes.end();
				it++)
			{
			    if ((it->status == NODEST_NONE ||
					it->status == NODEST_YOU) &&
				    it->targettingInfester == NULL)
				possibleTargets.push_back(&*it);
			}
			if (possibleTargets.size() > 0)
			{
			    const int i = rani(possibleTargets.size());
			    p_inv = new InfestingInvader(possibleTargets[i],
				    evilHasNode(NODEC_BLUE));
			}
		    }
		    break;
		case 4:
		    pos = RelPolarCoord(ranf()*4, ARENA_RAD-(20+rani(5)*10));
		    ds = rani(5)-2;
		    p_inv = new FoulEggLayingInvader(pos, ds);
		    break;
	    }
	}
	invaders.push_back(p_inv);
	timeSinceLastInvader = 0;
	//invaderRate = std::max(250, invaderRate - 10);
    }

    if (evilHasNode(NODEC_PURPLE) &&
	    rani(10000) < time)
    {
	for (std::vector<Invader*>::iterator it = invaders.begin();
		it != invaders.end();
		it++)
	    (*it)->dodge();
    }
}

void GameState::cleanup()
{
    if (deadShots)
    {
	shots.erase(remove_if(shots.begin(),
		    shots.end(), Shot::is_dead),
		shots.end());
    }

    for (std::vector<Invader*>::iterator it = invaders.begin();
	    it != invaders.end();
	    it++)
    {
	if ((*it)->dead())
	{
	    (*it)->onDeath();
	    delete *it;
	    (*it) = NULL;
	}
    }
    invaders.erase(remove_if(invaders.begin(),
		invaders.end(), isNullInvp),
	    invaders.end());
}

void GameState::draw(SDL_Surface* surface, bool blank)
{
    const RelPolarCoord d(you.aim.angle, zoomdist);

    View zoomView(ARENA_CENTRE + d,
	    (float)screenGeom.rad/((float)ARENA_RAD-zoomdist),
	    settings.rotatingView ? -d.angle : 0);

    View outerView(ARENA_CENTRE,
	    (float)screenGeom.rad/(float)ARENA_RAD,
	    settings.rotatingView ? -d.angle : 0);

    View view = settings.zoomEnabled ? zoomView : outerView;

    if (settings.zoomEnabled)
    {
	circleColor(surface, screenGeom.centre.x, screenGeom.centre.y,
		screenGeom.rad, blank ? 0xff : 0x505050ff);
    }
    else
    {
	Circle(zoomView.centre, ARENA_RAD-zoomdist,
		0x505050ff).draw(surface, view, NULL, blank, true);
    }

    drawIndicators(surface, view, blank);
    drawGrid(surface, view, blank);
    drawTargettingLines(surface, view, blank);
    drawObjects(surface, view, &zoomView, blank);
    drawNodeTargetting(surface, view, blank);
}

void GameState::drawGrid(SDL_Surface* surface, const View& view,
	bool blank)
{
    Circle(ARENA_CENTRE, ARENA_RAD,
	    0x808080ff).draw(surface, view, NULL, blank, true);

    if (settings.showGrid)
    {
	for (int i=1; i<6; i++)
	{
	    Circle(ARENA_CENTRE, i*ARENA_RAD/6,
		    0x30303000 + (i%2==0)*0x08080800 + 0xff
		  ).draw(surface, view, NULL, blank, true);
	}
	for (int i=0; i<12; i++)
	{
	    Line(ARENA_CENTRE, ARENA_CENTRE +
		    RelPolarCoord(i*4.0/12, ARENA_RAD),
		    0x30303000 + (i%2==0)*0x08080800 + 0xff
		).draw(surface, view, NULL, blank, true);
	}
    }
}

void GameState::drawTargettingLines(SDL_Surface* surface, const View& view,
	bool blank)
{
    if (!you.dead)
    {
	const Uint32 aimColour = 
	    youHaveNode(NODEC_PURPLE) ? 0x01000100 : 0x01000000;

	for (int dir = -1; dir < 3; dir+=2)
	    Line(ARENA_CENTRE, ARENA_CENTRE +
		    RelPolarCoord(you.aim.angle + dir*you.aimAccuracy(),
			ARENA_RAD),
		    aimColour * 0x80 + 0xff).draw(surface, view, NULL, blank, true);

	if (fabsf(you.aimAccuracy()) <= .45)
	    for (int dir = -1; dir < 3; dir+=2)
		Line(ARENA_CENTRE, ARENA_CENTRE +
			RelPolarCoord(you.aim.angle + dir*2*you.aimAccuracy(),
			    ARENA_RAD),
			aimColour * 0x50 + 0xff).draw(surface, view, NULL, blank, true);
    }
}

void GameState::drawNodeTargetting(SDL_Surface* surface, const View& view,
	bool blank)
{
    if (!you.dead && targettedNode != NULL)
    {
	const Uint32 c = (you.podTimer <= 0) ? 
	    0xff0000e0 :
	    0xff000060;
	const CartCoord p = targettedNode->cpos();
	const float r = targettedNode->radius;
	Line(p + RelCartCoord(-9*r/5, 0), p + RelCartCoord(-7*r/5, 0),
		c).draw(surface, view, NULL, blank);
	Line(p + RelCartCoord(9*r/5, 0), p + RelCartCoord(7*r/5, 0),
		c).draw(surface, view, NULL, blank);
	Line(p + RelCartCoord(0, -9*r/5), p + RelCartCoord(0, -7*r/5),
		c).draw(surface, view, NULL, blank);
	Line(p + RelCartCoord(0, 9*r/5), p + RelCartCoord(0, 7*r/5),
		c).draw(surface, view, NULL, blank);
    }
}

void GameState::drawObjects(SDL_Surface* surface, const View& view,
	View* boundView, bool blank)
{
    you.draw(surface, view, NULL, blank);

    for (std::vector<Shot>::iterator it = shots.begin();
	    it != shots.end();
	    it++)
	it->draw(surface, view, boundView, blank);

    for (std::vector<Invader*>::iterator it = invaders.begin();
	    it != invaders.end();
	    it++)
	(*it)->draw(surface, view, boundView, blank);

    for (std::vector<Node>::iterator it = nodes.begin();
	    it != nodes.end();
	    it++)
	it->draw(surface, view, boundView, blank);

    if (mutilationWave > 0)
	Circle(ARENA_CENTRE, mutilationWave, 0x00ffffff).draw(surface, view, NULL, blank);
}

// approxAtan2Frac: approximates atan2(y,x)*(6/PI)-1
//  (being the linear function of atan2(y,x) which is 0 at PI/6 and 1 at PI/3)
float approxAtan2Frac(int y, int x)
{
    float z = float(y)/x;

    // linear approximations, centred at closest of z=0.5, z=1, z=1.5, or z=2
    if (z < 0.75)
	return -0.11 + 1.53*(z-0.5);
    else if (z < 1.25)
	return 0.5 + 0.95*(z-1);
    else if (z < 1.75)
	return 0.88 + 0.59*(z-1.5);
    else
	return 1.11 + 0.38*(z-2);

    // cubic approximation to atan(z) around z=1:
    // return PI/4 + (z-1)/2 - (z-1)*(z-1)/4 + (z-1)*(z-1)*(z-1)/12;
}

void GameState::drawIndicators(SDL_Surface* surface, const View& view,
	bool blank)
{
    // heat and shield indicators:
    if (!blank)
    {
	Uint32 colour;
	for (int x = screenGeom.rad/2;
		x <= 866*screenGeom.rad/1000 + 15;
		x++)
	{
	    const int xsq = x*x;
	    int rsq;
	    for (int y = int(sqrt(screenGeom.indicatorRsqLim1 - xsq));
		    (rsq = xsq + y*y) <= screenGeom.indicatorRsqLim4;
		    y++)
	    {
		if (rsq < screenGeom.indicatorRsqLim1)
		    continue;
		const float frac = approxAtan2Frac(y,x);
		if (frac >= 0 && frac <= 1)
		{
		    if (rsq <= screenGeom.indicatorRsqLim2)
		    {
			// decay towards the edges, for prettiness
			const int decay = std::min(255,
				std::min(rsq - screenGeom.indicatorRsqLim1,
				    screenGeom.indicatorRsqLim2 - rsq)/2);
			int intensity;

			// heat
			intensity = 
			    you.shootHeat > you.shootMaxHeat*frac ? 55+int(200*frac) :
			    35;
			colour = 0x01000000 * (decay * intensity >> 8) + 0xff;

			pixelColor(surface, screenGeom.centre.x - x,
				screenGeom.centre.y - y, colour);

			// extraction
			intensity = 
			    extracted > extractMax*frac ? 55+int(200*frac) :
			    evilHasNode(NODEC_CYAN) ? 55 : 35;
			if (extracted == extractMax)
			    intensity = 0xff - (0xff - intensity)/2;
			colour = 0x00010100 * (decay * intensity >> 8) + 0xff;

			pixelColor(surface, screenGeom.centre.x + x,
				screenGeom.centre.y - y, colour);
		    }
		    else if (rsq < screenGeom.indicatorRsqLim3)
		    {
			// shade between heat and shield indicators
			pixelColor(surface, screenGeom.centre.x - x,
				screenGeom.centre.y - y, 0xa0);
		    }
		    else
		    {
			const int decay = std::min(255,
				std::min(rsq - screenGeom.indicatorRsqLim3,
				    screenGeom.indicatorRsqLim4 - rsq)/2);
			int intensity;

			// shield
			const int i = int(frac*4);
			const Uint32 baseColour =
			    (i == 0) ? 0x01000000 :
			    (i == 1) ? 0x01010000 :
			    (i == 2) ? 0x00010000 :
			    0x00010100;

			intensity =
			    (you.shield > frac*4) ? 55+(int(4*200*frac))%200 :
			    youHaveNode(NODEC_CYAN) ? 55 :
			    35;

			colour =
			    baseColour * (decay * intensity >> 8) + 0xff;

			pixelColor(surface, screenGeom.centre.x - x,
				screenGeom.centre.y - y, colour);
		    }
		}
	    }
	}
    }

    // shot indicators
    static const double shotIndicatorCos = cos(PI/6-PI/120);
    static const double shotIndicatorSin = sin(PI/6-PI/120);
    for (int i = 0; i < 4; i++)
    {
	const float d = screenGeom.rad + 3 + 5*i;
	const float x = shotIndicatorCos * d;
	const float y = shotIndicatorSin * d;
	const View shotView(CartCoord(x,-y), 1, 0);
	const bool super = youHaveShotNode(i);


	if (you.shootHeat >= you.shootMaxHeat - shotHeat(i))
	    continue;

	if (i == 3)
	{
	    // capture pod
	    if (you.podTimer <= 0)
		CapturePod(NULL, RelPolarCoord(0,0), super).draw(surface, shotView, NULL, blank);
	}
	else
	{
	    const int weight = i+1;
		Shot( CartCoord(0,0),
			RelPolarCoord(-0.3,
			    0.1+0.05*(3-weight) + super*0.04),
			weight, super).draw(surface, shotView, NULL, blank);
	}
    }

    // node possession indicators
    static const double nodeIndicatorCos[6] = {
	cos(PI/6), cos(PI/6 + PI/36), cos(PI/6 + 2*PI/36),
	cos(PI/6 + 3*PI/36), cos(PI/6 + 4*PI/36), cos(PI/6 + 5*PI/36) };
    static const double nodeIndicatorSin[6] = {
	sin(PI/6), sin(PI/6 + PI/36), sin(PI/6 + 2*PI/36),
	sin(PI/6 + 3*PI/36), sin(PI/6 + 4*PI/36), sin(PI/6 + 5*PI/36) };
    const float dist = screenGeom.rad + 3 + 20 + 10;

    int youNodeCount = 0;
    int evilNodeCount = 0;
    for (std::vector<Node>::iterator it = nodes.begin();
	    it != nodes.end();
	    it++)
    {
	if (it->status == NODEST_YOU)
	{
	    const View nodeView(CartCoord(
			nodeIndicatorCos[youNodeCount] * dist,
			- nodeIndicatorSin[youNodeCount] * dist),
		    1, 0);
	    Node(RelPolarCoord(0,0), 0, it->nodeColour).draw(surface, nodeView, NULL, blank);
	    youNodeCount++;
	}
	else if (it->status == NODEST_EVIL)
	{
	    const View nodeView(CartCoord(
			- nodeIndicatorCos[evilNodeCount] * dist,
			- nodeIndicatorSin[evilNodeCount] * dist),
		    1, 0);
	    Node(RelPolarCoord(0,0), 0, it->nodeColour).draw(surface, nodeView, NULL, blank);
	    evilNodeCount++;
	}
    }
}

int GameState::rateOfRating(int rating)
{
    static int rates[] = {
	5000, 4000, 3200, 2500, 2250, 2000, 1750, 1500, 1300};
    if (rating < 1)
	return -1;
    if (rating > 9)
	return (int)(1000.0*pow(0.9, (double)(rating-9)));
    return rates[rating-1];
}

void GameState::setRating(double newRating)
{
    if (newRating < 1.0)
	newRating = 1.0;

    rating = newRating;

    int i = (int) rating;
    int r1 = rateOfRating(i);
    int r2 = rateOfRating(i+1);
    
    invaderRate = (r1 + (int)((rating-i)*(r2-r1)));
}

const char* ratingString(int rating)
{
    // Recognise these?
    switch (rating)
    {
	case 1: return "Harmless";
	case 2: return "Mostly Harmless";
	case 3: return "Poor";
	case 4: return "Average";
	case 5: return "Above Average";
	case 6: return "Competent";
	case 7: return "Dangerous";
	case 8: return "Deadly";
	case 9: return "Elite";
	case 10: return "Elite+";
	case 11: return "Elite++";
	case 12: return "Elite+++";
	default: return "Unrated";
    };
}

const char* GameState::getHint()
{
    if (rating > 5.0 || settings.invuln)
	return "";

    if (end != END_WIN)
    {
	// Let's see if we can work out what they need to know...
	if (you.score == 0)
	    return "Hint: use keys '1', '2', '3' to shoot";

	if (you.score >= 10)
	{
	    if (!you.doneLaunchedPod)
		return "Hint: use key '4' to capture Nodes";

	    bool destroyed = false;
	    for (std::vector<Node>::iterator it = nodes.begin();
		    it != nodes.end();
		    it++)
		if (it->status == NODEST_DESTROYED)
		{
		    destroyed = true;
		    break;
		}

	    if (!destroyed && (END_EXTRACTED || you.score >= 30))
		return "Hint: hit filled nodes with '3'; destroy 4 to win!";
	}
    }

    // Either they won, or they lost for no obvious reason. Maybe we should
    // tell them about some things they might not have thought of...
    switch (int(rating*10)%6)
    {
	case 0:
	    return "Hint: Nodes help whoever has them";
	case 1:
	    return "Hint: Your shield charges if you stop shooting";
	case 2:
	    return "Hint: Use Down to turn fast, Up to glance back";
	case 3:
	    return "Hint: Pods ('4') can kill even shielded blue Blobs";
	case 4:
	    return "Hint: Green shots won't hurt blue Blobs";
	case 5:
	    return "Hint: Don't let Evil fill the cyan bar";

	default:
	    return "Hint: THIS IS A BUG";
    }
}
