/*
 * Kuklomenos
 * Copyright (C) 2008 Martin Bays <mbays@sdf.lonestar.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 */

#include "node.h"
#include "invaders.h"

Node::Node(RelPolarCoord ipos, float ids, NodeColour inodeColour, float iradius) :
    HPInvader(1,2),
    SpirallingPolygonalInvader(3, ipos, ids, 0),
    radius(iradius),
    spinRate(ids/3000),
    nodeColour(inodeColour),
    status(NODEST_NONE), primed(0), primeRate(0), glowPhase(0),
    targettingInfester(NULL), extractionProgress(0)
{
    for (int i = 0; i < 3; i++)
	points[i] = RelPolarCoord(i*4.0/3, radius);
}

void Node::doUpdate(int time)
{
    SpirallingInvader::doUpdate(time);

    if (status == NODEST_YOU && primed < 1)
	primed += primeRate*0.001*time;

    if (primed >= 1)
    {
	glowPhase += time*0.005;
	if (glowPhase > 7)
	    glowPhase -= 2*PI;
    }

    for (int i = 0; i < 3; i++)
	points[i] = points[i].rotated(time*spinRate);
}

Uint32 Node::colour() const
{
    Uint32 col;
    switch (nodeColour)
    {
	case NODEC_RED:
	    col = 0x01000000;
	    break;
	case NODEC_YELLOW:
	    col = 0x01010000;
	    break;
	case NODEC_GREEN:
	    col = 0x00010000;
	    break;
	case NODEC_BLUE:
	    col = 0x00000100;
	    break;
	case NODEC_PURPLE:
	    col = 0x01000100;
	    break;
	case NODEC_CYAN:
	    col = 0x00010100;
	    break;
	default:
	    col = 0x01010100;
    }

    const int intensity = (status == NODEST_NONE ? 0x90 :
	    status == NODEST_DESTROYED ? 0x20 :
	    primed >= 1 ? (0xff - (int)(0x30 * (1 - sinf(glowPhase)))) :
	    NODEST_YOU ? 0xff :
	    NODEST_EVIL ? 0xff :
	    0xb0);
    return col*intensity + 0xff;
}

Uint32 Node::innerColour() const
{
    return 0;
}

void Node::draw(SDL_Surface* surface, const View& view, View*
	boundView, bool blank, bool noAA) const
{

    if (status == NODEST_YOU)
    {
	CartCoord tpoints[3];
	for (int i = 0; i < 3; i++)
	    tpoints[i] = ARENA_CENTRE + pos +
		RelPolarCoord(pos.angle +  (RelPolarCoord(points[i])).angle, radius*primed);
	Polygon(tpoints, 3, 0x00ffff00 + 
		((primed >= 1) ? (0xff - (int)(0x30 * (1 + sinf(glowPhase)))) :
		 (int)(0x40 + 0x60*primed)),
		true).draw(surface, view, boundView, blank, noAA);
    }

    SpirallingPolygonalInvader::draw(surface, view, boundView, blank, noAA);

    if (status == NODEST_EVIL)
    {
	static const float glintSep = ARENA_RAD/15.0;
	float glintDist = pos.dist + glintSep*extractionProgress;
	while (glintDist < ARENA_RAD)
	{
	    const CartCoord start = ARENA_CENTRE + RelPolarCoord(pos.angle,
		    glintDist);
	    const CartCoord end = ARENA_CENTRE + RelPolarCoord(pos.angle,
		    glintDist + glintSep*.7);

	    Pixel(end, 0x00ffffff
		 ).draw(surface, view, boundView, blank, noAA);

	    glintDist += glintSep;
	}
    }
}

bool Node::infest(InfestingInvader* inv)
{
    if (status == NODEST_DESTROYED || status == NODEST_EVIL)
	return false;
    if (status == NODEST_YOU)
    {
	uncapture();
	return false;
    }
    status = NODEST_EVIL;
    return true;
}
void Node::uninfest()
{
    if (status == NODEST_EVIL)
	status = NODEST_NONE;
}
bool Node::capture(CapturePod* pod)
{
    if (status != NODEST_NONE)
	return false;

    status = NODEST_YOU;
    primeRate = pod->primeRate;
    return true;
}
void Node::uncapture()
{
    primed = 0;
    status = NODEST_NONE;
}

int Node::hit(int weight)
{
    if (weight >= 3 && status == NODEST_YOU && primed >= 1)
    {
	uncapture();
	status = NODEST_DESTROYED;
	primed = 0;
	return 3;
    }

    return weight;
}
