/*
 * Kuklomenos
 * Copyright (C) 2008 Martin Bays <mbays@sdf.lonestar.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 */

#include <cstdlib>
#include <cstdio>
#include <ctime>
#include <getopt.h>
#include <SDL/SDL.h>
#include <SDL/SDL_gfxPrimitives.h>
#include <SDL/SDL_mixer.h>
#include "state.h"
#include "geom.h"
#include "clock.h"
#include "data.h"
#include "settings.h"
#include "conffile.h"
#include "random.h"

Mix_Music *music = NULL;
void musicDone();

#ifndef __APPLE__
# include "config.h"
#endif

SDL_Surface* screen = NULL;

enum EventsReturn
{
    ER_NONE,
    ER_QUIT,
    ER_RESTART,
    ER_SCREENSHOT
};

EventsReturn process_events(GameState* gameState, GameClock& gameClock)
{
    SDL_Event event;
    while (SDL_PollEvent(&event))
    {
	switch (event.type)
	{
	    case SDL_QUIT:
		return ER_QUIT;
	    case SDL_KEYDOWN:
		switch (event.key.keysym.sym)
		{
		    case SDLK_q:
			return ER_QUIT;
		    case SDLK_p:
			gameClock.paused = !gameClock.paused;
			break;
		    case SDLK_z:
			settings.zoomEnabled = !settings.zoomEnabled;
			break;
		    case SDLK_r:
			settings.rotatingView = !settings.rotatingView;
			break;
		    case SDLK_a:
			settings.useAA = !settings.useAA;
			break;
		    case SDLK_g:
			settings.showGrid = !settings.showGrid;
			break;
		    case SDLK_LEFTBRACKET:
			settings.fps = std::max(1, settings.fps-1);
			break;
		    case SDLK_RIGHTBRACKET:
			settings.fps = std::min(100, settings.fps+1);
			break;
		    case SDLK_MINUS:
		    case SDLK_KP_MINUS:
			gameClock.rate = 4*gameClock.rate/5;
			if (950 <= gameClock.rate && gameClock.rate <= 1050)
			    gameClock.rate = 1000;
			break;
		    case SDLK_EQUALS:
			if (!event.key.keysym.mod & KMOD_SHIFT)
			{
			    gameClock.rate = 1000;
			    break;
			}
			// fallthrough
		    case SDLK_PLUS:
		    case SDLK_KP_PLUS:
			gameClock.rate = std::max(5*gameClock.rate/4,
				gameClock.rate+1);
			if (950 <= gameClock.rate && gameClock.rate <= 1050)
			    gameClock.rate = 1000;
			break;
		    case SDLK_SPACE:
		    case SDLK_RETURN:
			if (gameState->end)
			    return ER_RESTART;
			break;
		    case SDLK_F12:
			return ER_SCREENSHOT;
			break;
		    default: ;
		}
		break;
	    case SDL_VIDEORESIZE:
		screenGeom = ScreenGeom(event.resize.w, event.resize.h);
		screen = SDL_SetVideoMode(event.resize.w, event.resize.h,
			screen->format->BitsPerPixel, screen->flags);
		break;
	    default: ;
	}
    }
    return ER_NONE;
}

void drawInfo(SDL_Surface* surface, GameState* gameState, GameClock& gameClock,
	float observedFPS, bool blank=false);

void drawInfo(SDL_Surface* surface, GameState* gameState, GameClock& gameClock,
	float observedFPS, bool blank)
{
    char fpsStr[5+10];
    snprintf(fpsStr, 5+10, "fps: %.1f/%d", observedFPS, settings.fps);

    char ratingStr[8+20+7];
    snprintf(ratingStr, 8+20+7, "rating: %s (%.1f)",
	    ratingString((int)(gameState->rating)), gameState->rating);

    char rateStr[6+5];
    snprintf(rateStr, 6+5, "rate: %d.%d", gameClock.rate/1000,
	    (gameClock.rate%1000)/100);

    if (!blank)
    {
	gfxPrimitivesSetFont(fontSmall,7,13);
	stringColor(surface, screenGeom.info.x, screenGeom.info.y, fpsStr,
		0xffffffff);
	stringColor(surface, screenGeom.info.x, screenGeom.info.y+15*2,
		ratingStr, 0xffffffff);
	if (gameClock.rate != 1000)
	    stringColor(surface, screenGeom.info.x, screenGeom.info.y+15*4,
		    rateStr, 0xffffffff);
    }
    else
	blankRect(surface, screenGeom.info.x, screenGeom.info.y,
		(8+20+7)*(7+1), (gameClock.rate == 1000) ? 15*3 : 15*5);
}

class OverlayText
{
    public:
	static const int text1Len = 60;
	static const int text2Len = 80;
	char text1[text1Len];
	char text2[text2Len];
	int colour;

	OverlayText();
	void blank();
	void draw(SDL_Surface* surface, bool blank=false);
};

OverlayText::OverlayText() :
    colour(0)
{
    blank();
}
void OverlayText::blank()
{
    text1[0] = '\0';
    text2[0] = '\0';
}

void OverlayText::draw(SDL_Surface* surface, bool blank)
{
    const int x1 = screenGeom.centre.x - strlen(text1)*10/2;
    const int y1 = screenGeom.centre.y - 30;
    const int x2 = screenGeom.centre.x - strlen(text2)*10/2;
    const int y2 = screenGeom.centre.y + 30;

    if (!blank)
    {
	gfxPrimitivesSetFont(fontBig, 10, 20);
	stringColor(surface, x1, y1,
		text1, colour);
	stringColor(surface, x2, y2,
		text2, colour);
    }
    else
    {
	blankRect(surface, x1, y1, strlen(text1)*10, 20);
	blankRect(surface, x2, y2, strlen(text2)*10, 20);
    }
}

const int MIN_STEP = 30;

void initialize_system()
{
    /* Initialize SDL */
      if ( SDL_Init(SDL_INIT_VIDEO|SDL_INIT_TIMER|SDL_INIT_AUDIO|SDL_INIT_JOYSTICK ) == -1 ) {
//    if ( SDL_Init(SDL_INIT_EVERYTHING) < 0 ) {
	fprintf(stderr,
		"Couldn't initialize SDL: %s\n", SDL_GetError());
	exit(1);
    }
    atexit(SDL_Quit);			/* Clean up on exit */
}

#ifndef __APPLE__
void load_settings(int argc, char** argv)
{
	while (1) {
	int c;
	static const struct option long_options[] =
	{
	    {"width", 1, 0, 'W'},
	    {"height", 1, 0, 'H'},
	    {"bpp", 1, 0, 'b'},
	    {"fps", 1, 0, 'f'},
	    {"rating", 1, 0, 'r'},
	    {"noantialias", 0, 0, 'A'},
	    {"nogrid", 0, 0, 'G'},
	    {"nozoom", 0, 0, 'Z'},
	    {"norotate", 0, 0, 'R'},
	    {"antialias", 0, 0, 'a'},
	    {"grid", 0, 0, 'g'},
	    {"zoom", 0, 0, 'z'},
	    {"rotate", 0, 0, '@'},
	    {"fullscreen", 0, 0, 'F'},
	    {"hwsurface", 0, 0, 's'},
	    {"hwpalette", 0, 0, 'P'},
	    {"noresizable", 0, 0, 'S'},
	    {"invulnerable", 0, 0, 'i'},
	    {"version", 0, 0, 'V'},
	    {"help", 0, 0, 'h'},
	    {0,0,0,0}
	};
	c = getopt_long(argc, argv, "W:H:b:f:r:I:AGZRagzFsPSiVh", long_options, NULL);
	if (c == -1)
	    break;
	switch (c)
	{
	    case 'W':
		settings.width = atoi(optarg);
		break;
	    case 'H':
		settings.height = atoi(optarg);
		break;
	    case 'b':
		settings.bpp = atoi(optarg);
		break;
	    case 'f':
		settings.fps = atoi(optarg);
		break;
	    case 'r':
		if (1.0 <= atof(optarg))
		    settings.requestedRating = atof(optarg);
		else
		{
		    printf("bad rating\n");
		    exit(1);
		}
		break;
	    case 'F':
		settings.fullscreen = true;
		break;
	    case 'S':
		settings.videoFlags &= ~SDL_RESIZABLE;
		break;
	    case 'P':
		settings.videoFlags |= SDL_HWPALETTE;
		break;
	    case 's':
		settings.videoFlags |= SDL_HWSURFACE;
		break;
	    case 'Z':
		settings.zoomEnabled = false;
		break;
	    case 'R':
		settings.rotatingView = false;
		break;
	    case 'A':
		settings.useAA = false;
		break;
	    case 'G':
		settings.showGrid = false;
		break;
	    case 'z':
		settings.zoomEnabled = true;
		break;
	    case '@':
		settings.rotatingView = true;
		break;
	    case 'a':
		settings.useAA = true;
		break;
	    case 'g':
		settings.showGrid = true;
		break;
	    case 'i':
		settings.invuln = true;
		break;
	    case 'V':
		printf("%s\n", PACKAGE_STRING);
		exit(0);
	    case 'h':
		printf("Options:\n\t"
			"-W --width WIDTH\n\t-H --height HEIGHT\n\t-b --bpp BITS\n\t-f --fps FPS\n\t"
			"-F --fullscreen\n\t-S --noresizable\n\t-P --hwpalette\n\t-s --hwsurface\n\t"
			"-Z,-z --[no]zoom\n\t-R --[no]rotate\n\t-G,-g --[no]grid\n\t-A,-a --[no]antialias\n\t"
			"-z --zoom\n\t--rotate\n\t-g --grid\n\t-a --antialias\n\t"
			"-r --rating RATING\t\t1: harmless... 4: average... 9: elite\n\t"
			"-i --invulnerable\n\t-V --version\n\t-h --help\n");
		exit(0);
		break;

	    case '?':
		exit(1);
		break;

	    default:
		printf("getopt returned unexpected character %c\n", c);
	}
    }

}
#endif

void initialize_video()
{
    if (settings.fullscreen)
	settings.videoFlags |= SDL_FULLSCREEN;

    if (settings.width == 0 || settings.height == 0)
    {
	SDL_Rect** modes = SDL_ListModes(NULL, settings.videoFlags |
		(settings.fullscreen ? SDL_FULLSCREEN : 0));
	if (modes == NULL)
	{
	    fprintf(stderr,
		    "SDL reports no modes available\n");
	    exit(1);
	}
	if (modes == (SDL_Rect**)(-1))
	{
	    // "All modes available"
	    settings.width = 1024;
	    settings.height = 768;
	}
	else
	{
	    // use the first (i.e. biggest) mode which is no bigger than
	    // 1024x768
	    SDL_Rect* bestMode;
	    do
	    {
		bestMode = *modes;
		if (bestMode->w <= 1024 && bestMode->h <= 768)
		    break;
	    } while (++modes);
	    settings.width = bestMode->w;
	    settings.height = bestMode->h;
	}
    }

    /* Initialize the display */
    screen = SDL_SetVideoMode(settings.width, settings.height, settings.bpp, settings.videoFlags);
    if ( screen == NULL ) {
	fprintf(stderr, "Couldn't set %dx%dx%d video mode: %s\n",
		settings.width, settings.height, settings.bpp, SDL_GetError());
	exit(1);
    }

#ifndef __APPLE__
    if (settings.bpp == 0 && screen->format->BitsPerPixel == 32)
    {
	/* There appears to be a bug in current (2.0.17) SDL_gfx relating to
	 * alpha blending in 32bpp mode. To work around this, for now we only
	 * use 32bpp if it was explicitly asked for.
	 */
	printf("Changing colour depth from 32 to 24; run with -b32 to force\n");

	screen = SDL_SetVideoMode(settings.width, settings.height, 24, settings.videoFlags);
	if ( screen == NULL ) {
	    fprintf(stderr, "Couldn't set %dx%dx%d video mode: %s\n",
		    settings.width, settings.height, 24, SDL_GetError());
	    exit(1);
	}
    }
#endif

    /* Show some info */
    printf("Set %dx%dx%d mode\n",
	    screen->w, screen->h, screen->format->BitsPerPixel);
    printf("Video surface located in %s memory.\n",
	    (screen->flags&SDL_HWSURFACE) ? "video" : "system");

    /* Check for double buffering */
    if ( screen->flags & SDL_DOUBLEBUF ) {
	printf("Double-buffering enabled - good!\n");
    }

    SDL_WM_SetCaption( "Kuklomenos", NULL );
    SDL_ShowCursor(SDL_DISABLE);
}

void run_game()
{
    GameState* gameState = new GameState;
    GameClock gameClock;
    OverlayText overlayText;
	
    if (settings.requestedRating > 0)
	gameState->setRating(settings.requestedRating);
    else
    {
	gameState->setRating(config.rating);
	config.shouldUpdateRating = true;
    }
	
    srand(time(NULL));

    if ( !initFont() )
    {
	fprintf(stderr, "Failed to load font data file\n");
	exit(1);
    }

    screenGeom = ScreenGeom(settings.width, settings.height);

    // main loop
    Uint32 lastStateUpdate = 0;
    Uint32 ticksBefore, ticksAfter;
    Uint32 loopTicks = SDL_GetTicks();
    int timeTillNextFrame = 0;
    int delayTime, updateTime;
    float avFrameTime = 1000/settings.fps;
    int fpsRegulatorTenthTicks = 0;
    const int avFrames = 10; // number of frames to average over
    EventsReturn eventsReturn = ER_NONE;
    bool wantScreenshot = false;
    bool quit = false;
    bool ended = false;
    while ( !quit ) {
	while (timeTillNextFrame > 0)
	{
	    delayTime = std::min(timeTillNextFrame, MIN_STEP);
	    SDL_Delay(delayTime);
	    timeTillNextFrame -= delayTime;

	    eventsReturn = process_events(gameState, gameClock);

	    switch (eventsReturn)
	    {
		case ER_RESTART:
		    {
			GameState* newGameState = new GameState;
			newGameState->setRating(gameState->rating);
			delete gameState;
			gameState = newGameState;
		    }
		    ended = false;
		    overlayText.blank();
		    break;
		case ER_QUIT:
		    quit = true;
		    break;
		case ER_SCREENSHOT:
		    wantScreenshot = true;
		    break;
		default: ;
	    }

	    updateTime = SDL_GetTicks() - lastStateUpdate;
	    lastStateUpdate += updateTime;
	    if (!gameClock.paused)
	    {
		gameState->update(gameClock.scale(updateTime));
		gameClock.update(updateTime);
	    }
	}

	ticksBefore = SDL_GetTicks();
	if (!gameClock.paused)
	{
	    gameState->draw(screen);
	    drawInfo(screen, gameState, gameClock, 1000.0/avFrameTime);
	    overlayText.draw(screen);
	    SDL_Flip(screen);

	    if (wantScreenshot)
	    {
		SDL_SaveBMP(screen, "screenshot.bmp");
		wantScreenshot = false;
	    }

	    // blank over what we've drawn:
	    gameState->draw(screen, true);
	    drawInfo(screen, gameState, gameClock, 1000.0/avFrameTime, true);
	    overlayText.draw(screen, true);
	}
	ticksAfter = SDL_GetTicks();

	const int renderingTicks = ticksAfter - ticksBefore;

	// Add in a manual tweak to the delay, to ensure we approximate the
	// requested fps (where possible):
	if (10000/avFrameTime < settings.fps * 10 &&
		1000/settings.fps - renderingTicks + fpsRegulatorTenthTicks/10 > 1)
	    fpsRegulatorTenthTicks -= 1;
	else if (10000/avFrameTime > settings.fps * 10)
	    fpsRegulatorTenthTicks += 1;

	timeTillNextFrame =
	    std::max(1, (1000/settings.fps) - renderingTicks +
		    fpsRegulatorTenthTicks/10 + (rani(10) < fpsRegulatorTenthTicks%10));

	if (!ended && gameState->end)
	{
	    ended = true;
	    switch (gameState->end)
	    {
		case END_DEAD:
		case END_EXTRACTED:
		    snprintf(overlayText.text1, overlayText.text1Len, "Failure.");
		    overlayText.colour = 0xff4444ff;
		    break;
		case END_WIN:
		    snprintf(overlayText.text1, overlayText.text1Len, "Victory!");
		    overlayText.colour = 0x88ff00ff;
		    break;
		default: ;
	    }

	    // give a hint if appropriate
	    strncpy(overlayText.text2, gameState->getHint(),
		    overlayText.text2Len);

	    if (config.shouldUpdateRating && !settings.invuln)
	    {
		int old = (int)(gameState->rating);

		if (gameState->end == END_DEAD ||
			gameState->end == END_EXTRACTED)
		{
		    gameState->rating -= 0.1;
		    if (gameState->rating < 1)
			gameState->rating = 1.0;

		}
		else if (gameState->end == END_WIN)
		{
		    gameState->rating += 0.4;
		    if (gameState->rating > 9)
			gameState->rating = 9.0;

		}

		if ((int)(gameState->rating) != old)
		{
		    if (overlayText.text2[0] == '\0')
		    {
			snprintf(overlayText.text2, overlayText.text2Len, "New rating: \"%s\".",
				ratingString( (int)(gameState->rating) ));
		    }
		    else
		    {
			char buf[40];
			snprintf(buf, 40, "  New rating: \"%s\".",
				ratingString( (int)(gameState->rating) ));
			strncat(overlayText.text1, buf,
				overlayText.text1Len - strlen(overlayText.text1) - 1);
		    }
		}
	    }
	}

	const int loopTime = SDL_GetTicks() - loopTicks;
	avFrameTime += (loopTime-avFrameTime)/avFrames;
	loopTicks = SDL_GetTicks();
    }
    
    if (gameState->extracted > 0)
    {
	// game is forfeit - reduce rating
	gameState->rating -= 0.1;
	if (gameState->rating < 1)
	    gameState->rating = 1.0;
    }
    if (config.shouldUpdateRating)
    {
	config.rating = gameState->rating;
	config.write();
    }

    delete gameState;

    Mix_HaltMusic();
    Mix_FreeMusic(music);
    music = NULL;

    SDL_Quit();
}

#ifndef __APPLE__
int main(int argc, char** argv)
{

if(Mix_OpenAudio(44100, AUDIO_S16, 1, 4096)) { printf("Unable to open audio!\n"); exit(1); }

    music = Mix_LoadMUS("data/Kuklomenos/amiga_machines.mod");
    Mix_PlayMusic(music, -1);
    Mix_HookMusicFinished(musicDone);


    load_settings(argc, argv);
    initialize_system();
    initialize_video();
    run_game();

    return 0;
}
#endif /* __APPLE__ */

 void musicDone()
 {
   Mix_HaltMusic();
   Mix_FreeMusic(music);
   music = NULL;
 }
