/*
 * Kuklomenos
 * Copyright (C) 2008 Martin Bays <mbays@sdf.lonestar.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 */

#include <algorithm>
#include <SDL/SDL.h>
#include <SDL/SDL_gfxPrimitives.h>

#include "coords.h"
#include "gfx.h"
#include "geom.h"
#include "settings.h"

View::View(CartCoord icentre, float izoom, float iangle) :
    centre(icentre), zoom(izoom), angle(iangle) {}

ScreenCoord View::coord(const CartCoord &c) const
{
    RelCartCoord d( (c-centre).rotated(angle) );
    return ScreenCoord(
	    screenGeom.centre.x+(int)(d.dx*zoom),
	    screenGeom.centre.y-(int)(d.dy*zoom));
}

bool View::inView(const CartCoord &c, float in) const
{
    RelCartCoord d = c-centre;
    return ( (zoom*zoom)*(d.dx*d.dx+d.dy*d.dy) <= (screenGeom.rad-in)*(screenGeom.rad-in) );
}

int Line::draw(SDL_Surface* surface, const View& view, View* boundView, bool blank, bool noAA)
{
    ScreenCoord s = view.coord(start);
    ScreenCoord e = view.coord(end);
    const bool useAA = !noAA && settings.useAA;

    if (blank && useAA)
    {
	const int x = std::min(s.x, e.x) - 1;
	const int y = std::min(s.y, e.y) - 1;
	const int w = (std::max(s.x, e.x) + 1) - x;
	const int h = (std::max(s.y, e.y) + 1) - y;
	blankRect(surface,x,y,w,h);
	return true;
    }

    // XXX: HACK! SDL_gfx buggily draws a slightly different line if alpha is
    // 0xff, so we give it 0xfe instead. The incredibly sharp-eyed might be
    // able to see trails as a result...
    Uint32 drawColour = colour;
    if (blank)
	drawColour = (colour & 0x000000ff == 0x000000ff) ?
	    0x000000ff :
	    0x000000fe;

    if (!boundView || (boundView->inView(start) && boundView->inView(end)))
	return ( useAA ? aalineColor : lineColor )
	    (surface, s.x, s.y, e.x, e.y, drawColour);
    else
	return 0;
}

int Circle::draw(SDL_Surface* surface, const View& view, View* boundView, bool blank, bool noAA)
{
    const bool useAA = !noAA && settings.useAA;
    ScreenCoord c = view.coord(centre);
    float screenRad = r*view.zoom;

    if (blank && useAA)
    {
	// yes, we really do need this big a margin, at least for very small
	// circles
	const int x = c.x - int(screenRad) - 2;
	const int y = c.y - int(screenRad) - 2;
	const int w = 2*int(screenRad) + 5;
	const int h = 2*int(screenRad) + 5;
	blankRect(surface,x,y,w,h);
	return true;
    }

    if (!boundView || boundView->inView(centre, screenRad))
	return (filled ? filledCircleColor : 
		useAA ? aacircleColor : circleColor)
	    (surface, c.x, c.y, int(screenRad), blank ? 0x000000ff : colour);
    else
	return 0;
}

int Polygon::draw(SDL_Surface* surface, const View& view, View* boundView, bool blank, bool noAA)
{
    const bool useAA = !noAA && settings.useAA;

    for (int i=0; i<n; i++)
    {
	if (boundView && !boundView->inView(points[i]))
	    return 0;
    }

    Sint16 *sx = new Sint16[n];
    Sint16 *sy = new Sint16[n];
    int ret;

    for (int i=0; i<n; i++)
    {
	ScreenCoord s = view.coord(points[i]);
	sx[i] = s.x;
	sy[i] = s.y;
    }

    if (blank && useAA)
    {
	Sint16* xp = NULL;
	Sint16* xep = NULL;
	Sint16* yp = NULL;
	Sint16* yep = NULL;
	for (int i=0; i<n; i++)
	{
	    if (xp == NULL || sx[i] < *xp)
		xp = &sx[i];
	    if (yp == NULL || sy[i] < *yp)
		yp = &sy[i];
	    if (xep == NULL || sx[i] > *xep)
		xep = &sx[i];
	    if (yep == NULL || sy[i] > *yep)
		yep = &sy[i];
	}
	blankRect(surface, *xp-1, *yp-1, *xep-*xp+2, *yep-*yp+2);
	ret = true;
    }
    else
	ret = ( filled ? filledPolygonColor :
		useAA ? aapolygonColor : polygonColor )(
		    surface, sx, sy, n, blank ? 0x000000ff : colour);

    delete[] sx;
    delete[] sy;
    return ret;
}

int Pixel::draw(SDL_Surface* surface, const View& view, View* boundView, bool blank, bool noAA)
{
    if (boundView && !boundView->inView(point))
	return 0;
    else
    {
	ScreenCoord c = view.coord(point);
	return pixelColor(surface, c.x, c.y, blank ? 0x000000ff : colour);
    }
}

void blankRect(SDL_Surface* surface, int x, int y, int w, int h)
{
    SDL_Rect rect;
    rect.x = x;
    rect.y = y;
    rect.w = w;
    rect.h = h;
    SDL_FillRect(surface, &rect, SDL_MapRGB(surface->format, 0x00, 0x00, 0x00));
}
