----------------------------------------------------------------------------------
-- Company: 
-- Engineer: 
-- 
-- Create Date:    22:42:00 01/03/2010 
-- Design Name: 
-- Module Name:    flickerfixer_top - Behavioral 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description: 
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
--library UNISIM;
--use UNISIM.VComponents.all;

entity flickerfixer_top is
	Port (
		-- Amiga video signals
		am_red		: in  STD_LOGIC_VECTOR (3 downto 0);
		am_green		: in  STD_LOGIC_VECTOR (3 downto 0);
		am_blue		: in  STD_LOGIC_VECTOR (3 downto 0);
		am_hsync_n	: in  STD_LOGIC;
		am_vsync_n	: in  STD_LOGIC;
		-- VGA video signals
		vga_red		: out  STD_LOGIC_VECTOR (3 downto 0);
		vga_green	: out  STD_LOGIC_VECTOR (3 downto 0);
		vga_blue		: out  STD_LOGIC_VECTOR (3 downto 0);
		vga_hsync_n	: out  STD_LOGIC;
		vga_vsync_n	: out  STD_LOGIC;
		-- Memory interface
		dl				: inout  STD_LOGIC_VECTOR (11 downto 0);
		dh				: inout  STD_LOGIC_VECTOR (11 downto 0);
		a				: out  STD_LOGIC_VECTOR (17 downto 0);
		cs_n			: out  STD_LOGIC_VECTOR (1 downto 0);
		we_n			: out  STD_LOGIC;
		oe_n			: out  STD_LOGIC;
		-- Test output
		test			: out STD_LOGIC;
		-- clock enable output
		clocken_n	: out STD_LOGIC;
		-- clock inputs
		clock28M		: in  STD_LOGIC;
		am_cdac		: in  STD_LOGIC;
		am_c1_n		: in	STD_LOGIC;
		am_c3_n		: in	STD_LOGIC;
		am_c4_n		: in	STD_LOGIC
	);
end flickerfixer_top;

architecture Behavioral of flickerfixer_top is
	-- Amiga: duration of an entire scanline in 7 MHz cycles (= pixel pairs)
	-- 908 is the number of pixels at 14 MHz per Amiga Scanline (= within 64ms) (orig. 904 if 0 is executed 2 times)
	constant columns						: integer := 908/2;
	
	-- Amiga: back porch length in 7 MHz clock cycles (= pixel pairs)
	-- Number of pixel pairs to wait, until sampling starts
	--constant am_back_porch_length		: integer := 168/2;  -- 168/2 orig
	constant am_back_porch_length		: integer := 168/2;
	-- The number of pixel pairs (= memory addresses within one single scan line)
	constant am_scanline_length		: integer := (columns - am_back_porch_length);

	-- VGA: length of the horizontal front porch period in 7 MHz Cycles
	constant vga_front_porch_length	: integer := 38; -- 36
	
	-- VGA: length of the back porch period in 7 MHz Cycles
--	constant vga_back_porch_length	: integer := 20;

	-- VGA: hsync duration in 7 MHz Cycles
	constant vga_hsync_length			: integer := 27;

--	constant vga_video_start			: integer := vga_hsync_length + vga_back_porch_length;

	-- The number of rows for the vsync_period clock cycles
	constant vga_vsync_rows				: integer := 2; --2
	
	-- The number of black top rows to be skipped
	constant vga_start_row_lores		: integer := 0; -- 40

	constant vga_start_row_hires		: integer := 32; -- 40
	
	-- total number of VGA rows to be displayed
	constant vga_use_row_hires			: integer := 525; --580; -- 504;
	-- 449 ends up in 720x400 (official VESA)
	-- 464 ends up in 720x400
	-- 490 ends up in 720x400
	-- 510 ends up in 640x400
	-- 520 ends up in 720x400
	-- 525 ends up in 640x480 with bar on the bottom
	-- 530 ends up in 640x480 with bar on the bottom
	-- 532 ends up in 640x480 with bar on the bottom *
	-- 533 ends up in 640x480 with bar on the bottom
	-- 540 ends up in 848x480 with bar on the bottom
	-- 560 ends up in 848x480 with bar on the bottom
	constant vga_use_row_lores			: integer := 449;

	-- type denoting current vga_state
	type vga_state_t is ( vga_hsync, vga_fporch, vga_video, vga_bporch );

	-- the 28 MHz main system clock
	signal clock			: std_logic;
	-- the 28 Mhz cycle in which we are
	signal cycle			: std_logic_vector(1 downto 0);

	-- registered copy of cdac
	signal am_cdac_r		: std_logic;

	-- registered copy of hsync, vsync
	signal am_hsync_r		: std_logic;
	signal am_vsync_r		: std_logic;

	-- internal output memory data bus
	signal d_out			: std_logic_vector(11 downto 0);
	-- internal input memory data bus
	signal d_in				: std_logic_vector(23 downto 0);
	-- output enable for data bus driver, drive d_out to the databus when high
	signal d_oe				: std_logic;

	-- the address to which to write the next entry
	--signal wr_addr			: std_logic_vector(18 downto 0);
	-- the address from which to read the next entry
	--signal rd_addr			: std_logic_vector(18 downto 1);
	
	signal front_addr			: std_logic_vector(18 downto 0);
	signal back_addr			: std_logic_vector(18 downto 0);

	-- Field type detection circuitry
	signal field_cnt		: std_logic_vector(1 downto 0);
	-- indicates, which field we have (odd, even)
	signal this_field		: std_logic;
	signal last_field		: std_logic;
	signal interlaced		: std_logic;

	-- This one is high for one 7 MHz cycle, to indicate that we left vsync period
	-- in this case interlace and xxx_field are valid and updated
	-- this can be used to reload the address counter
	signal vsync_left		: std_logic;

	-- this is a shortcut to indicate that we entered hsync
	signal hsync_entering : std_logic;

	-- this is a shortcut to indicate that we entered hsync
	signal hsync_leaving	: std_logic;

	signal amiga_wren		: std_logic;
	
--	signal back_porch_ind : std_logic;
	
	-- don't record data
	signal vert_inhibit	: std_logic;
	
	-- number of 7 MHz cycles within this period (one row contains at most 448 7MHz clock cycles)
	signal cycle_count	: std_logic_vector(7 downto 0);

	signal vga_readen		: std_logic;
	signal vga_readen_r	: std_logic;
	attribute KEEP : string;
--	attribute KEEP of vga_readen : signal is "TRUE";

	signal vga_end			: std_logic;
--	attribute KEEP of vga_end : signal is "TRUE";

	-- VGA signals
	signal vga_state		: vga_state_t;

	-- this signal is asserted for one cacle to start the VGA readout
--	signal vga_start		: std_logic;

	signal vga_vsync_cnt : std_logic_vector(1 downto 0);

	signal vga_cnt			: std_logic_vector(7 downto 0);

	signal vga_red_i		: std_logic_vector(3 downto 0);
	signal vga_green_i	: std_logic_vector(3 downto 0);
	signal vga_blue_i		: std_logic_vector(3 downto 0);

	signal vga_vsync_ni	: std_logic;
	
	signal we_nr			: std_logic;
	signal we_ni			: std_logic;
	
	signal cs_nr			: std_logic_vector(1 downto 0);
	signal cs_ni			: std_logic_vector(1 downto 0);
	
begin
	-- just rename the 28 Mhz clock to clock.
	-- As cdac is in-pase with 28 MHz, we better invert the clock and use falling clock edges for capturing
	clock <= not clock28M;

	-- cycle counter
	process(clock)
	begin
		if(clock'event and clock='1') then
			if(am_cdac_r='0' and am_cdac='1') then
				cycle <= (others => '0');
			else
				cycle <= cycle + 1;
			end if;
			am_cdac_r <= am_cdac;
		end if;
	end process;
	
	-- Show, when cycle is 0
	--test <= '1' when cycle=0 else '0';
	--test <= vga_readen;
	--test <= vsync_left;
	--test <= vert_inhibit;
	--test <=  this_field;
	--test <= '1' when vga_row = 0 else '0';

	-- foward internal data bus to pins, when output is enabled, otherwise high-z
	dl <= d_out when d_oe='1' else (others => 'Z');
	dh <= d_out when d_oe='1' else (others => 'Z');

	-- Amiga video sampler
	process(clock)
	begin
		if(clock'event and clock='1') then
			-- latch sample, if we are on an even cycle
			-- the left pixel is stored in the MSB the right pixel is in the LSB
			if(cycle="-1") then
				d_out <= am_red & am_green & am_blue;
				--d_out <= vga_readen & "00" & vga_readen & cycle_count;
				--d_out <= x"0DD";
				--d_out <= amiga_row & cycle_count;
				--d_out <= front_addr(11 downto 0);
			end if;
		end if;
	end process;

	-- VGA output driver
	process(clock)
	begin
		if(clock'event and clock='1') then
			if(cycle="-0") then
				-- memorize data from SRAM, when we are on an even clock cycle
				if(vga_readen='1') then
					d_in(11 downto 0) <= dl;
					d_in(23 downto 12) <= dh;
				else
					d_in <= (others => '0');
				end if;
				--d_in <= vga_row & cycle_count & vga_row & cycle_count;
			else
				-- shift right pixel to left pixel
				d_in <= d_in(11 downto 0) & x"000";
			end if;
		end if;
	end process;

	-- the pixel data is taken from the data input register
	vga_red_i	<= d_in(23 downto 20);
	vga_green_i	<= d_in(19 downto 16);
	vga_blue_i	<= d_in(15 downto 12);
	
	vga_drv_ts_inst : for i in 0 to 3 generate
	begin
		vga_red(i) <= '1' when vga_red_i(i)='1' else 'Z';
		vga_green(i) <= '1' when vga_green_i(i)='1' else 'Z';
		vga_blue(i) <= '1' when vga_blue_i(i)='1' else 'Z';
	end generate;

	-- Amiga vsync evaluation
	vsync_proc : process(clock)
	begin
		if(clock'event and clock='1') then
			if(cycle=0) then
				-- Check for vsync
				if(am_vsync_n='0') then
					-- we are in vertical sync state
					if(am_vsync_r='1') then
						-- we just entered v_sync, so reset the field detector
						field_cnt <= (others => '0');
					elsif(hsync_leaving='1') then
						-- rising edge on hsync detected
						field_cnt <= field_cnt +1;
					end if;
				else
					-- we are out of sync state
					if(am_vsync_r='0') then
						-- we just left the vsync phase
						-- memorize last field, to determine whether we are in interlaced mode
						last_field <= this_field;

						-- detect in which field we are currently
						if(field_cnt > 2) then
							-- we encountered 3 vsyncs (so we are in lores or interlace mode)
							--this_field <= not this_field;
							this_field <= '1';
						else
							this_field <= '0';
						end if;

						-- indicate vsync for one 7 MHz Cycle
						vsync_left <= '1';
					else
						vsync_left <= '0';
					end if;			
				end if;

				-- make registered copy of vsync for edge detection
				am_vsync_r <= am_vsync_n;
			end if;
		end if;
	end process;

	-- the frames are interlaced, when this field is different to the last field
	interlaced <= this_field xor last_field;

	-- hsync evaluation
	-- cycle_count starts counting with the falling hsync_edge and counts until columns/2 is reached.
	-- So the counter is reset after 32ms (= one VGA line or half of the PAL line)
	hsync_proc : process(clock)
	begin
		if(clock'event and clock='1') then
			if(cycle=0) then
				-- reset counters when entering hsync or reaching mid of the line
				if(hsync_entering='1' or cycle_count=0) then
					-- hsync fallig edge, reset counter
					-- as zero is included in counting, set the counter to one less than the duration
					cycle_count <= CONV_STD_LOGIC_VECTOR(columns/2-1, cycle_count'length);
				else
					cycle_count <= cycle_count -1;
				end if;

				if(am_vsync_n='0') then
					-- stop recording
					vert_inhibit <= '1';
				elsif(hsync_leaving='1') then
					-- restart recording
					vert_inhibit <= '0';
				end if;

				-- start writing of screen content
				if(cycle_count = columns/2-am_back_porch_length) then
					amiga_wren <= '1';
				elsif(hsync_entering='1') then
					amiga_wren <= '0';
				end if;

				if(cycle_count = columns/2-vga_front_porch_length) then
					vga_readen <= '1';
				elsif(cycle_count = columns/2-vga_front_porch_length-am_scanline_length/2) then
					vga_readen <= '0';
				end if;
				vga_readen_r <= vga_readen;

				-- make registered copy of hsync for edge detection
				am_hsync_r <= am_hsync_n;
			end if;
		end if;
	end process;

	-- make vga hsync signal
	-- generate vga outputs based on state mashine
	vga_hsync_n <= '0' when cycle_count >= columns/2-vga_hsync_length else '1';

	-- this is high, when we just entered hsync
	hsync_entering	<= am_hsync_r and not am_hsync_n;
	hsync_leaving	<= am_hsync_n and not am_hsync_r;

	-- memory control signal generator
	addrgen : process(clock)
	begin
		if(clock'event and clock='1') then
			-- get fron address at the pins from back
			--front_addr <= back_addr;
			if(cycle="-0") then
				-- even cycles initiate write cycles
				--a <= wr_addr(18 downto 1);
				oe_n <= '1';
				d_oe <= '1';
				we_ni <= '0';
				
				if(back_addr(0)='0') then
					-- the left/earlier pixel has the lower address and is part of the MSB
					cs_ni <= "01";
				else
					cs_ni <= "10";
				end if;
			else
				-- odd cycles initiate read cycles, we always read both words
				--a <= rd_addr;
				oe_n <= '0';
				d_oe <= '0';
				cs_ni <= "00";
				we_ni <= '1';
			end if;
		end if;
	end process;
	
	delay : process(clock)
	begin
		if(clock'event and clock='0') then
			we_nr <= we_ni;
			--cs_nr	<= cs_ni;
		end if;
	end process;

	we_n <= we_ni or not we_nr;
	cs_n <= cs_ni; --or not cs_nr;

	-- the front address is the one to be seen at the address pins
	a <= front_addr(18 downto 1);

	addr_gen_proc : process(clock)
	begin
		if(clock'event and clock='1') then
			if(cycle="-1") then
				-- perform counting on odd cycles only

				-- write address is in the front and has to be pushed back
				if(vert_inhibit='1') then
					--back_addr <= (others => '0');
					-- reset write address
					if(this_field='1') then
						-- will be set two times
						back_addr <= CONV_STD_LOGIC_VECTOR(am_scanline_length*2, back_addr'length);
					else
						back_addr <= (others => '0');
					end if;
				elsif(hsync_entering='1' and interlaced='1') then
					-- in lores don't skip lines
					-- will be added two times
					back_addr <= front_addr + CONV_STD_LOGIC_VECTOR(am_scanline_length*2+1, back_addr'length);
				elsif(amiga_wren='1') then
					back_addr <= front_addr +1;
				else
					back_addr <= front_addr;
				end if;

				-- read address is in the back and has to be pushed back
				if(back_addr=vga_use_row_hires*am_scanline_length*2) then
					front_addr <= CONV_STD_LOGIC_VECTOR(am_scanline_length*2*vga_start_row_hires, front_addr'length);
				elsif(back_addr=vga_use_row_lores*am_scanline_length*2 and interlaced='0') then
					front_addr <= CONV_STD_LOGIC_VECTOR(am_scanline_length*2*vga_start_row_lores, front_addr'length);
--					vga_vsync_ni <= '0';
--				elsif(vga_vsync_cnt > 0) then
--					-- do not increment read address during vsync
--					if(vga_readen='0' and vga_readen_r='1') then
--						vga_vsync_cnt <= vga_vsync_cnt -1;
--					end if;
--					front_addr <= back_addr;
				elsif(vga_readen='1') then
					front_addr <= back_addr +2;
				else
					front_addr <= back_addr;
				end if;

				if(back_addr < (vga_vsync_rows+vga_start_row_lores)*am_scanline_length*2) then
					vga_vsync_ni <= '0';
				elsif(back_addr < (vga_vsync_rows+vga_start_row_hires)*am_scanline_length*2 and interlaced='1') then
					vga_vsync_ni <= '0';
				else
					vga_vsync_ni <= '1';
				end if;

			else
				-- just flip rd and wr addresses
				back_addr <= front_addr;	-- read address is in the front and is pushed back
				front_addr <= back_addr;	-- write address is in the back and has to be pushed to the front
			end if;
		end if;
	end process;

	vga_vsync_n <= vga_vsync_ni;

--	oe_n <= cycle(0);
--	d_oe <= cycle(0);
--	we_n <= not cycle(0);

--	a <=	wr_addr(18 downto 1) when (cycle="-1") else
--			rd_addr(18 downto 1);

--	-- write address generation
--	wr_addr_proc : process(clock)
--	begin
--		if(clock'event and clock='1') then
--			if(cycle="-0") then
--				if(vsync_left='1') then
--					-- reset write address
--					if(this_field='1') then
--						-- will be set two times
--						wr_addr <= CONV_STD_LOGIC_VECTOR(am_scanline_length, wr_addr'length);
--					else
--						wr_addr <= (others => '0');
--					end if;
--				elsif(hsync_leaving='1') then
--					-- will be added two times
--					wr_addr <= wr_addr + CONV_STD_LOGIC_VECTOR(am_scanline_length/2, wr_addr'length);
--				elsif(back_porch_ind='0') then
--					wr_addr <= wr_addr +1;
--				end if;
--			end if;
--		end if;
--	end process;
--
--	-- Read address generator
--	rd_addr_proc : process(clock)
--	begin
--		if(clock'event and clock='1') then
--			if(cycle="-0") then
--				if(rd_addr >= vga_use_row*am_scanline_length) then
--					rd_addr <= (others => '0');
--				elsif(cycle_count>=vga_video_start and cycle_count<vga_video_start+am_scanline_length) then
--					rd_addr <= rd_addr +1;
--				end if;
--			end if;
--		end if;
--	end process;
--
--	vga_vsync_n <= '1' when rd_addr < vga_vsync_rows*am_scanline_length else '0';

--	-- vga state maschine
--	vga_engine : process(clock)
--	begin
--		if(clock'event and clock='1') then
--			-- look at every fourth cycle only
--			if(cycle=0) then
----				if(vga_start='1') then
----					-- if vga_start is active, then reset frame drawing
----					vga_state <= vga_fporch;
----					vga_row <= (others => '0');
----					vga_cnt <= CONV_STD_LOGIC_VECTOR(vga_front_porch_length, vga_cnt'length);
--
--				if(vga_cnt=0) then
--					-- if counter reached zero, then switch state
--					case vga_state is
--						when vga_hsync =>
--							vga_cnt <= CONV_STD_LOGIC_VECTOR(vga_back_porch_length, vga_cnt'length);
--							vga_state <= vga_bporch;
--					
--						when vga_bporch =>
--							-- goto video state
--							vga_state <= vga_video;
--							-- reading data takes half the time of writing
--							vga_cnt <= CONV_STD_LOGIC_VECTOR(am_scanline_length/2, vga_cnt'length);
--
--						when vga_video =>
--							-- goto front porch
--							vga_state <= vga_fporch;
--							vga_cnt <= CONV_STD_LOGIC_VECTOR(vga_front_porch_length, vga_cnt'length);
--							vga_row <= vga_row +1;
--
--						when vga_fporch =>
--							vga_state <= vga_hsync;
--							vga_cnt <= CONV_STD_LOGIC_VECTOR(vga_hsync_length, vga_cnt'length);
--
--						when others =>
--							null;
--					end case;
--				else
--					vga_cnt <= vga_cnt -1;
--				end if;
--			end if;
--
--		end if;
--	end process;

	-- disable output clock
	clocken_n <= 'Z';

end Behavioral;
