
/*
 * FS1541 - volume, BAM and lock handling
 *
 * Copyright (C) 1996 - 1999 Michael Krause
 * Copyright (C) 1998        John "Graham" Selck (portions)
 * Copyright (C) 2000 - 2008 Helmut Schoettner (many portions)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h> /* contains decl of sprintf, but will be linked from amiga.lib! */
#include <string.h>

#include <exec/types.h>
#include <exec/execbase.h>
#include <exec/memory.h>
#include <dos/dosextens.h>
#include <dos/filehandler.h>
#include <devices/trackdisk.h>
#include <devices/timer.h>

#include <proto/exec.h>
#include <proto/dos.h>
#include <clib/alib_protos.h>
#include <proto/utility.h>

#include "main.h"
#include "volume.h"
#include "disk.h"
#include "packet.h"
#include "support.h"

BYTE diskchgintbit= -1;
struct VolumeNode *curvolumenode= NULL;
struct DosList *curdoslist= NULL;

int disk_inserted= FALSE;

struct BAM *bam;
struct DirEntry directory[144];
int dirsize;

UBYTE interleave= 6;

static struct MinList volumelist;

static struct IOExtTD *diskchgint_req;

#ifdef LATTICE
__interrupt __saveds static void diskchginthandler(void)
#else
static void diskchginthandler(void)
#endif
{
	Signal (ourtask, 1<< diskchgintbit);
}
static struct Interrupt diskchgint =
{
	{ NULL, NULL, NT_INTERRUPT, 0, "FS1541" },
	NULL,
	(APTR)&diskchginthandler
};

static struct MsgPort *UDStimerport;
struct timerequest *UDStimer;
static int timeropen= 0;

static void CreateDollar (struct VolumeNode *node);

/*-------------------------------------------------------------------------*/

LONG InitVolumeSS (void)
{
	LONG error= 0;

	if ((UDStimerport= CreateMsgPort()))
	{
		if ((UDStimer= CreateIORequest(UDStimerport, sizeof(struct timerequest))))
		{
			if ((timeropen= (!OpenDevice(TIMERNAME,UNIT_VBLANK,(struct IORequest *)UDStimer, 0))))
			{
				if ((diskchgintbit= AllocSignal(-1))>= 0)
				{
					if ((diskchgint_req= AllocVec(sizeof(struct IOExtTD), MEMF_PUBLIC)))
					{
						NewList((struct List*)&volumelist);

						CopyMem(SExtDiskReq, diskchgint_req, sizeof(struct IOExtTD));
						diskchgint_req->iotd_Req.io_Command= TD_ADDCHANGEINT;
						diskchgint_req->iotd_Req.io_Data= &diskchgint;
						diskchgint_req->iotd_Req.io_Length= sizeof(struct Interrupt);
						diskchgint_req->iotd_Req.io_Flags= 0;
						SendIO((struct IORequest*)diskchgint_req);

						return(0);

					}
					else
					{
						error= ERROR_NO_FREE_STORE;
					}
				}
				else
				{
					error= ERROR_NO_FREE_STORE;
				}
			}
			else
			{
				error= ERROR_DEVICE_NOT_MOUNTED;
			}
		}
		else
		{
			error= ERROR_NO_FREE_STORE;
		}
	}
	else
	{
		error= ERROR_NO_FREE_STORE;
	}

	QuitVolumeSS();
	return(error);
}

void QuitVolumeSS (void)
{
	if (diskchgint_req)
	{
		diskchgint_req->iotd_Req.io_Command= TD_REMCHANGEINT;
		diskchgint_req->iotd_Req.io_Data= &diskchgint;
		diskchgint_req->iotd_Req.io_Length= sizeof(struct Interrupt);
		diskchgint_req->iotd_Req.io_Flags= 0;
		DoIO((struct IORequest*)diskchgint_req);

		FreeVec(diskchgint_req);
	}

	if (diskchgintbit >= 0)
	{
		FreeSignal(diskchgintbit);
	}

	if (timeropen)
	{
		StopUDSTimer();
		CloseDevice((struct IORequest*)UDStimer);
	}

	if (UDStimer)
	{
		DeleteIORequest(UDStimer);
	}

	if (UDStimerport)
	{
		DeleteMsgPort(UDStimerport);
	}
}

/*-------------------------------------------------------------------------*/

static ULONG bam_secmask[21]=
{
	1<<16, 1<<17, 1<<18, 1<<19, 1<<20, 1<<21, 1<<22, 1<<23,
	1<<8,  1<<9,  1<<10, 1<<11, 1<<12, 1<<13, 1<<14, 1<<15,
	1<<0,  1<<1,  1<<2,  1<<3,  1<<4
};

static int CountBits (ULONG l)
{
	BYTE i, n= 0;

	for (i= 0; i< 32; i++)
	{
		if (l& 1) n++;
		l= l>> 1;
	}

	return (n);
}

/* Fix the track allocation info so we can rely on the upper byte without danger */

static void BamFixTracks (struct BAM *b)
{
	int i;

	for (i= 0; i < 35; i++)
	{
		ULONG p= bam->tracks[i];
		int track= i+ 1;

		if (track<= 17)
		{
			p &= 0x00ffff1f;
		}
		else if (track<= 24)
		{
			p &= 0x00ffff07;
		}
		else if (track<= 30)
		{
			p &= 0x00ffff03;
		}
		else
		{
			p &= 0x00ffff01;
		}

		p |= CountBits(p)<< 24;

		bam->tracks[i]= p;
	}
}

void DoDiskInsert (BOOL bForce)
{
	struct DateStamp datestamp;
	SExtDiskReq->iotd_Req.io_Command= TD_CHANGESTATE;
	SExtDiskReq->iotd_Req.io_Flags= IOF_QUICK;
	DoIO((struct IORequest*)SExtDiskReq);

	if (SExtDiskReq->iotd_Req.io_Actual && !bForce)
	{
		/* Switch off the motor. */

		SExtDiskReq->iotd_Req.io_Command= TD_MOTOR;
		SExtDiskReq->iotd_Req.io_Flags= 0;
		SExtDiskReq->iotd_Req.io_Length= 0;
		DoIO((struct IORequest*)SExtDiskReq);
	}
	else
	{
		/* Disk has been inserted. */

		int i, t, s;
		UBYTE diskname[20];
		UBYTE diskid[5];
		struct VolumeNode *node;

		disk_inserted= TRUE;

		ResetDisk();
		iWriteProtect= iHardWriteProtect;

		/* Read Block Allocation Map */

		if (!(bam= (struct BAM*)GetBlockTS(18, 0)))
		{
			return;
		}

		BamFixTracks(bam);

		if (bam->id != 'A')
			iWriteProtect= TRUE;

		/* Read directory */

		/* Clear directory structure in memory */

		memset(directory, 0, sizeof(directory));

		/* 1541 always uses 18, 1	*/
		/* t= 18 bam->dirt			*/
		/* s= 1 bam->dirs				*/

		for (i= 0, dirsize= 0, t= 18, s= 1; i< 18; t= directory[8* i].t, s= directory[8* i].s, i++)
		{
			struct DataBlock *block;

			if (!(block= GetBlockTS(t, s)))
			{
				return;
			}

			CopyMem(block, &directory[8* i], 256);

			dirsize += 8;

			if (!directory[8* i].t)
			{
				break;
			}
		}

		/* Strip trailing type 0x00 entries */

		for (i= dirsize; --i>= 0 && directory[i].type == 0x00; dirsize--);

		/* Check if this is a volume we know */

		Cbm2Ascii(diskname, bam->name, 16, CHR_LEVEL_ALPHA);
		Cbm2Ascii(diskid, bam->ubDiskIdAr, 2, CHR_LEVEL_ALPHA);
		sprintf(diskname, "%s,%s", diskname, diskid);

		for (node= (struct VolumeNode*)volumelist.mlh_Head; node->node.mln_Succ; node=(struct VolumeNode*)(node->node.mln_Succ))
		{
			if (!Stricmp(diskname, &node->name[1]))
			{
				while(!AttemptLockDosList(LDF_VOLUMES|LDF_WRITE))
				{
					DoPackets();
				}

				UnLockDosList(LDF_VOLUMES|LDF_WRITE);

				curvolumenode= node;
				curdoslist= node->volnode;
				curdoslist->dol_Task= ourport;
				curdoslist->dol_misc.dol_volume.dol_LockList= NULL;

				SendEvent(TRUE);
				return;
			}
		}

		/* Create a new volume node */

		if ((node= AllocVec(sizeof(struct VolumeNode), MEMF_PUBLIC|MEMF_CLEAR)))
		{
			struct DosList *newvol;

			if ((newvol= AllocVec(sizeof(struct DosList), MEMF_PUBLIC|MEMF_CLEAR)))
			{
				/* Generate DosList entry (Volume) */

				LONG rc;

				newvol->dol_Type= DLT_VOLUME;
				newvol->dol_Task= ourport;
				newvol->dol_misc.dol_volume.dol_LockList= NULL;
				newvol->dol_misc.dol_volume.dol_DiskType= ID_CBM_DISK;
				newvol->dol_Name= (BSTR)MKBADDR(&node->name);

				/* set the volume creation date to the disk insertion date */

				DateStamp(&datestamp);
				newvol->dol_misc.dol_volume.dol_VolumeDate.ds_Days= datestamp.ds_Days;
				newvol->dol_misc.dol_volume.dol_VolumeDate.ds_Minute= datestamp.ds_Minute;
				newvol->dol_misc.dol_volume.dol_VolumeDate.ds_Tick= datestamp.ds_Tick;

				Cbm2Ascii(&node->name[1], bam->name, 16, CHR_LEVEL_ALPHA);
				Cbm2Ascii(diskid, bam->ubDiskIdAr, 2, CHR_LEVEL_ALPHA);
				sprintf(&node->name[1], "%s,%s", &node->name[1], diskid);

				node->name[0]= strlen(&node->name[1]);

				node->volnode= newvol;
				AddHead((struct List*)&volumelist, (struct Node*)node);

				while(!AttemptLockDosList(LDF_VOLUMES|LDF_WRITE))
				{
					DoPackets();
				}

				UnLockDosList(LDF_VOLUMES|LDF_WRITE);

				rc= AddDosEntry(newvol);

				if (rc)
				{
					curvolumenode= node;
					curdoslist= newvol;

					CreateDollar(node);

					SendEvent(TRUE);
					MotorOff();

					return;
				}

				FreeVec(newvol);
			}
			FreeVec(node);
		}
	}

	MotorOff();
}

void DoDiskRemove (BOOL bForce)
{
	disk_inserted= FALSE;

	if (curvolumenode)
	{
		if (!curvolumenode->locklist || bForce)
		{
			/* No locks -> remove completely */

			while(!AttemptLockDosList(LDF_VOLUMES|LDF_WRITE))
			{
				DoPackets();
			}

			RemDosEntry(curdoslist);
			Remove((struct Node*)curvolumenode);
			FreeVec(curvolumenode);
			UnLockDosList(LDF_VOLUMES|LDF_WRITE);
			SendEvent(FALSE);
		}
		else
		{
			/* Do not remove completely, leave disk icon on Workbench. */

			while(!AttemptLockDosList(LDF_VOLUMES|LDF_WRITE))
			{
				DoPackets();
			}

			curdoslist->dol_Task= NULL;
			curdoslist->dol_misc.dol_volume.dol_LockList= (BPTR)curvolumenode->locklist;
			UnLockDosList(LDF_VOLUMES|LDF_WRITE);
			SendEvent(FALSE);
		}

		curvolumenode= NULL;
		curdoslist= NULL;
	}
}

/*-------------------------------------------------------------------------*/

static void CreateDollar (struct VolumeNode *node)
{
	UBYTE buf[32], buf2[32], idbuf[5], *p= node->dollarbuf;
	int i, j;
	char types[27]= "DSPUREERSELQGRLh@@?\\?\\qrhu";

	strncpy(&buf2[0], bam->ubDiskIdAr,  2);
	strncpy(&buf2[2], bam->pad3,    1);
	strncpy(&buf2[3], bam->dostype, 2);

	/* convert all shift spaces (0xA0) in normal spaces (0x20) */

	for (i= 0; i< 5; i++)
	{
	  if (buf2[i] == 0xA0)
	  {
	      buf2[i]= 0x20;
	  }
	}

	/* get format id and dos id */

	Cbm2Ascii(&buf[0], &buf2[0], 5, CHR_LEVEL_PRINT);

	/* get true disk id of sector label of first bam sector */

	Cbm2Ascii(&idbuf[0], &ubDiskIdAr[357][0], 2, CHR_LEVEL_PRINT);

	/* get true disk name */

	Cbm2Ascii(&buf2[0], bam->name, 16, CHR_LEVEL_PRINT);

	sprintf(p, "0 \033[7m\"%-16s\" %-5s\033[0m\n", buf2, buf);
	p += strlen(p);

	for (i=0; i < dirsize; i++)
	{
		UBYTE type, typ;

		/* hide type DEL and open */

		if ((type= directory[i].type) == 0x00)
		{
			continue;
		}

		Cbm2Ascii(buf, directory[i].name, 16, CHR_LEVEL_PRINT);
		sprintf(p, "%-5ld\"%s\"", (ULONG)(directory[i].lengthh<< 8)| directory[i].lengthl, buf);
		p += strlen(p);

		for (j= 0; j< (16- strlen(buf)); j++)
		{
			*p++= ' ';
		}

		typ= type & 0x0F;
		sprintf(p, "%lc%lc%lc%lc%lc\n", (type & 0x80 ? ' ' : '*'), types[typ], types[typ+ 5], types[typ+ 10], (type & 0x40 ? '<' : ' '));
		p += strlen(p);
	}

	sprintf(p, "%ld BLOCKS FREE.\n", (LONG)(683-UsedBlocks()));
	p += strlen(p);

	sprintf(p, "\033[7m%lc%lc\033[0m FORMAT IDENTIFIER.\n", idbuf[0], idbuf[1]);
	p += strlen(p);

	/*
		sprintf(p, "\nSecLabel (16 Bytes) of Block 357:\n");
		p += strlen(p);

		for (i= 0; i< 16; i++)
		{
			sprintf(p, "Byte #%02d: 0x%02X [%c]\n", i, ubDiskIdAr[357][i], (ubDiskIdAr[357][i]) ? (ubDiskIdAr[357][i]) : (0x20));
			p += strlen(p);
		}
	*/

	if (iDiskScanned)
	{
		/* check for different format identifiers */

		for (i= 0, j= 0; i< 683; i++)
		{
			if (ubDiskIdAr[i][0] != ubDiskIdAr[357][0] || ubDiskIdAr[i][1] != ubDiskIdAr[357][1])
			{
				j++;
			}
		}

		if (j)
		{
			sprintf(p, "%ld BLOCK(S) DO NOT", (LONG)j);
			p += strlen(p);
		}
		else
		{
			sprintf(p, "ALL BLOCKS");
			p += strlen(p);
		}

		sprintf(p, " MATCH DIRECTORY SECTOR'S FORMAT IDENTIFIER.\n");
		p += strlen(p);

	}

/*
	if (curvolumenode->locklist)
	{
		struct FileLock *fl;
		int i;

		i= 0;
		fl= curvolumenode->locklist;
		while (fl)
		{
			sprintf(p, "#%03d fl_Key: %ld\n", i, fl->fl_Key);
			p += strlen(p);

			sprintf(p, "#%03d fl_Access: %ld\n", i, fl->fl_Access);
			p += strlen(p);

			sprintf(p, "#%03d fl_Task: 0x%X\n", i, fl->fl_Task);
			p += strlen(p);

			sprintf(p, "#%03d fl_Volume: 0x%X\n", i, fl->fl_Volume);
			p += strlen(p);

			fl= (struct FileLock *)fl->fl_Link;

			sprintf(p, "#%03d fl_Link: 0x%X\n\n", i, fl);
			p += strlen(p);

			i++;
		}

		sprintf(p, "VOLUME HAS LOCK LIST AT 0x%X WITH %d ENTRIES.\n", curvolumenode->locklist, i);
		p += strlen(p);

		fl= curvolumenode->locklist;
		while (fl)
		{
			FreeLock(fl);
			fl= (struct FileLock *)fl->fl_Link;
		}

	}
*/

	/* change output text style to normal */

	sprintf(p, "\033[0m");
	p += strlen(p);

	node->dollarlen= (ULONG)(p - node->dollarbuf);

	/* suppress control characters except EOS, ESC, LF and CR */

	while (p >= node->dollarbuf)
	{
		if (*p > 0x00 && *p < 0x20 && *p != 0x1B && *p != 0x0A && *p != 0x0D)
		{
			*p= '.';
		}

		p--;
	}
}

/*-------------------------------------------------------------------------*/

static BYTE SearchFreeBlockOnTrack (UBYTE track, UBYTE froms)
{
	UBYTE i,s,numsecs;
	ULONG mask, plan=bam->tracks[track-1];

	if (plan>> 24 > 0)
	{
		numsecs= SectorsOnTrack(track);

		if (track == 18)
		{
			s= froms+ 3;
		}
		else
		{
			s= froms+ interleave;
		}

		for (i= 0; i< numsecs; i++, s++)
		{
			s= s % numsecs;
			mask= bam_secmask[s];

			if (plan & mask)
			{
				/* Free sector found. */
				plan &= ~mask;
				plan -= (1<<24);
				bam->tracks[track- 1]= plan;

				return((BYTE)s);
			}
		}

		/* If we reach this point, the BAM is corrupted... */
	}

	return(-1);
}

UWORD AllocBlock (UBYTE fromt, UBYTE froms)
{
	UBYTE track= fromt;
	BYTE sector;
	BOOL flag= 0;
	BYTE dir= track < 18 ? -1 : 1;

	while(1)
	{
		if (track == 18)
		{
			track += dir;
		}

		if ((sector= SearchFreeBlockOnTrack(track, track==fromt ? froms : 0)) >= 0)
		{
			return ((UWORD)((track<<8)|sector));
		}

		track += dir;

		if (track == 0)
		{
			if (flag)
			{
				return(0);
			}

			flag++;
			dir= 1;
			track= fromt+ 1;
		}

		if (track == 36)
		{
			if (flag)
			{
				return(0);
			}

			flag++;
			dir= -1;
			track= fromt- 1;
		}
	}
}

void FreeBlock (UBYTE t, UBYTE s)
{
	if (t>=1 && t<=35)
	{
		if (s < SectorsOnTrack(t))
		{
			ULONG plan= bam->tracks[t- 1];

			plan |= bam_secmask[s];
			plan += (1<< 24);
			bam->tracks[t- 1]= plan;
		}
	}
}

UWORD UsedBlocks (void)
{
	int i;
	UWORD r= 683;

	if (curvolumenode)
	{
		for (i= 1; i<= 17; i++)
		{
			r -= bam->tracks[i- 1]>> 24;
		}

		for (i= 19; i<= 35;i++)
		{
			r -= bam->tracks[i- 1]>> 24;
		}
	}

	return(r);
}

/*-------------------------------------------------------------------------*/

void OptimizeDirectory (void)
{
	int i, newdirsize= 0;

	/* Throw out all DEL files */
	for (i= 0; i< dirsize; i++)
	{
		if (((directory[i].type) & 0x0F) != 0x00)
		{
			memmove(&directory[newdirsize++], &directory[i], sizeof(struct DirEntry));
		}
	}

	dirsize= newdirsize;
	memset(&directory[dirsize], 0, (144- newdirsize)* sizeof(struct DirEntry));

	/* Clear all Track/Sector fields */
	for (i= 0; i< dirsize; i++)
	{
		directory[i].t= 0;
		directory[i].s= 0;
	}
}

/*-------------------------------------------------------------------------*/

void StopUDSTimer (void)
{
	if (!CheckIO((struct IORequest*)UDStimer))
	{
		AbortIO((struct IORequest*)UDStimer);
		WaitIO((struct IORequest*)UDStimer);
	}

	SetSignal(0, 1<< (UDStimerport->mp_SigBit));
}

void StartUDSTimer (void)
{
	StopUDSTimer();

	UDStimer->tr_time.tv_secs= 1;
	UDStimer->tr_time.tv_micro= 0;
	UDStimer->tr_node.io_Command= TR_ADDREQUEST;
	SendIO((struct IORequest*)UDStimer);
}

void UpdateDiskStructure (void)
{
	int i,n;
	UBYTE s,s2;

	if (!CheckIO((struct IORequest*)UDStimer))
	{
		return;
	}

	StopUDSTimer();

	/* Strip trailing type 0x00 entries */

	for (i= dirsize; --i>= 0 && directory[i].type == 0x00; dirsize--);

	/* Write BAM and directory blocks */
	bam->dirt= 18;
	bam->dirs= 1;
	bam->tracks[18- 1]= 0x11fcff07; /* Block 0 is BAM and Block 1 is first directory block */
	i= dirsize>> 3;
	s2= 1;

	for (n= 0; n<= i; n++)
	{
		s= s2;

		if (n< i)
		{
			s2= SearchFreeBlockOnTrack(18, s); /* will always succeed */
			directory[n<< 3].t= 18;
			directory[n<< 3].s= s2;
		}
		else
		{
			directory[n<< 3].t= 0;
			directory[n<< 3].s= 0xff;
		}

		PutBlockTS(18, s, &directory[n<< 3]);
	}

	PutBlockTS(18, 0, bam);
	MotorOff();

	CreateDollar(curvolumenode);
}

/*-------------------------------------------------------------------------*/

BPTR MakeLock (LONG flkey, LONG axs)
{
	struct FileLock *fl;

	if ((fl= AllocVec(sizeof(struct FileLock), MEMF_PUBLIC|MEMF_CLEAR)))
	{
		fl->fl_Key= flkey;
		fl->fl_Access= axs;
		fl->fl_Task= ourport;
		fl->fl_Volume= MKBADDR(curdoslist);
		fl->fl_Link= (BPTR)curvolumenode->locklist;
		curvolumenode->locklist= fl;
	}

	return(MKBADDR(fl));
}

/* currently not used - leads to crashes */
int FreeAllLocks (BOOL bForce)
{
	struct FileLock *fl;
	int i;

	i= 0;

	if (curvolumenode && inhibited)
	{
		fl= curvolumenode->locklist;
		while (fl)
		{
			if (fl->fl_Access != EXCLUSIVE_LOCK || bForce)
			{
				FreeLock(fl);
				fl= (struct FileLock *)fl->fl_Link;
				i++;
			}
		}
	}

	return (i);
}

void FreeLock (struct FileLock *fl)
{
	struct VolumeNode *node;

	for (node=(struct VolumeNode*)volumelist.mlh_Head; node->node.mln_Succ; node=(struct VolumeNode*)(node->node.mln_Succ))
	{
		struct FileLock *cur,*last;

		for (cur= node->locklist, last= (struct FileLock*)&node->locklist; cur; last= cur, cur= (struct FileLock*)last->fl_Link)
		{
			if (cur == fl)
			{
				last->fl_Link= cur->fl_Link;
				fl->fl_Task= NULL;
				FreeVec(fl);

				if (!node->locklist && !node->volnode->dol_Task)
				{
					/* An unmounted volume does not have any locks open
					   any more, so we can safely remove the volume node. */

					while(!AttemptLockDosList(LDF_VOLUMES|LDF_WRITE))
					{
						DoPackets();
					}

					RemDosEntry((struct DosList*)node->volnode);
					Remove((struct Node*)node);
					FreeVec(node);
					UnLockDosList(LDF_VOLUMES|LDF_WRITE);
					SendEvent(FALSE);
				}

				return;
			}
		}
	}
}

/* Is an object still Lockable with the respective access mode? */

BOOL Lockable (UBYTE t, UBYTE s, LONG mode)
{
	struct FileLock *fl;
	LONG searchkey= t<< 8| s;

	for (fl= curvolumenode->locklist; fl; fl= (struct FileLock*)fl->fl_Link)
	{
		if (fl->fl_Key == searchkey)
		{
			if (mode == EXCLUSIVE_LOCK)
			{
				return(FALSE);
			}
			else
			{
				if (fl->fl_Access == EXCLUSIVE_LOCK)
				{
					return(FALSE);
				}
			}
		}
	}

	return(TRUE);
}
