
/*
 * FS1541 - basic disk interaction routines
 *
 * Copyright (C) 1996 - 1999 Michael Krause
 * Copyright (C) 1998        John "Graham" Selck (portions)
 * Copyright (C) 2000 - 2008 Helmut Schoettner (many portions)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <exec/types.h>
#include <exec/execbase.h>
#include <exec/memory.h>
#include <exec/ports.h>
#include <dos/dos.h>
#include <intuition/intuition.h>
#include <devices/timer.h>

#include <proto/exec.h>
#include <proto/intuition.h>

#include "disk.h"
#include "volume.h"

LONG lNumSoftErrors= 0;
static struct MsgPort *diskport;
struct IOExtTD *SExtDiskReq;
BOOL bAutoscan= TRUE;
BOOL bSectorLabel= FALSE;

struct MsgPort *SDpSender;

static UBYTE *diskimage;
// static int iDiskScanned;
int iDiskScanned;
static BYTE sectab[683];
UBYTE ubDiskIdAr[683][16];

static int curblk= 357; /* Preloading starts at 18, 0= BAM */
static int devopen= 0;
ULONG ulDiskChangeCount;

static BOOL ignoreall, abortall;

struct IntuitionBase *IntuitionBase;

int iWriteProtect= FALSE, iHardWriteProtect= FALSE;

static void CacheBlock(ULONG n, BOOL bWrite);

/* ErrorReq() return codes */
#define REQ_RETRY				1
#define REQ_IGNORE			2
#define REQ_IGNORE_ALL		3
#define REQ_ABORT_ALL		0

/*-------------------------------------------------------------------------*/

LONG InitDiskSS (STRPTR pszDevice, ULONG ulUnit, ULONG ulFlags)
{
	LONG error= 0;

	if ((diskimage= AllocVec(D64_SIZE, MEMF_PUBLIC)))
	{
		if ((diskport= CreateMsgPort()))
		{
			if ((SExtDiskReq= CreateIORequest(diskport, sizeof(struct IOExtTD))))
			{
				if ((devopen= (!OpenDevice(pszDevice, ulUnit, (struct IORequest *)SExtDiskReq,ulFlags))))
				{
					return (0);
				}
				else
				{
					error= ERROR_DEVICE_NOT_MOUNTED;
				}
			}
			else
			{
				error= ERROR_NO_FREE_STORE;
			}
		}
		else
		{
			error= ERROR_NO_FREE_STORE;
		}
	}
	else
	{
		error= ERROR_NO_FREE_STORE;
	}

	QuitDiskSS();
	
	return (error);
}

void QuitDiskSS (void)
{
	if (devopen)
	{
		CloseDevice((struct IORequest*)SExtDiskReq);
	}

	if (SExtDiskReq)
	{
		DeleteIORequest(SExtDiskReq);
	}

	if (diskport)
	{
		DeleteMsgPort(diskport);
	}

	if (diskimage)
	{
		FreeVec(diskimage);
	}
}

/*-------------------------------------------------------------------------*/

void ResetDisk (void)
{
	iDiskScanned= !bAutoscan;

	memset(sectab, SEC_NOT_LOADED, 683);
	memset(ubDiskIdAr, 0, (683*16));

	lNumSoftErrors= 0;
	curblk= 357;

	abortall= FALSE;
	ignoreall= FALSE;

	SExtDiskReq->iotd_Req.io_Command= TD_CHANGENUM;
	SExtDiskReq->iotd_Req.io_Flags= IOF_QUICK;
	DoIO((struct IORequest*)SExtDiskReq);
	ulDiskChangeCount= SExtDiskReq->iotd_Req.io_Actual;

	SExtDiskReq->iotd_Req.io_Command= ETD_CLEAR;
	SExtDiskReq->iotd_Count= ulDiskChangeCount;
	SExtDiskReq->iotd_Req.io_Flags= IOF_QUICK;
	DoIO((struct IORequest*)SExtDiskReq);

	SExtDiskReq->iotd_Req.io_Command= TD_PROTSTATUS;
	SExtDiskReq->iotd_Req.io_Flags= IOF_QUICK;
	DoIO((struct IORequest*)SExtDiskReq);
	iHardWriteProtect= SExtDiskReq->iotd_Req.io_Actual;
}

/* Asynchronous disk preloader */
BOOL LoadDisk (void)
{
	if (!iDiskScanned && curvolumenode && bAutoscan)
	{
		int readblk;

		for (readblk= curblk; readblk< 683; readblk++)
		{
			if (sectab[readblk]== SEC_NOT_LOADED)
			{
				break;
			}
		}

		if (readblk == 683)
		{
			if (curblk == 0)
			{
				/* Loading finished - switch off motor. */
				
				SExtDiskReq->iotd_Req.io_Command= TD_MOTOR;
				SExtDiskReq->iotd_Req.io_Flags= 0;
				SExtDiskReq->iotd_Req.io_Length= 0;
				DoIO((struct IORequest*)SExtDiskReq);
				iDiskScanned= TRUE;
				
				/* Recreate virtual file '$' again to check format id of all blocks */
				UpdateDiskStructure();
				
				return (TRUE);
			}
			else
			{
				curblk= 0;
			}

			return (FALSE);
		}

		curblk= readblk;
		CacheBlock(readblk, FALSE);

		return (FALSE);

	}
	else
	{
		return (TRUE);
	}
}

void MotorOff (void)
{
	if (iDiskScanned && CheckIO((struct IORequest*)UDStimer))
	{
		SExtDiskReq->iotd_Req.io_Command= ETD_UPDATE;
		SExtDiskReq->iotd_Req.io_Flags= 0;
		DoIO((struct IORequest*)SExtDiskReq);

		SExtDiskReq->iotd_Req.io_Command= TD_MOTOR;
		SExtDiskReq->iotd_Req.io_Flags= 0;
		SExtDiskReq->iotd_Req.io_Length= 0;
		DoIO((struct IORequest*)SExtDiskReq);
	}
}

/*-------------------------------------------------------------------------*/

static LONG TS2Block (UBYTE ubTrack, UBYTE ubSector)
{
	if (ubTrack< 1)
	{
		return (-1);
	}
	else if (ubTrack<= 17)
	{
		if (ubSector> 21)
		{
			return (-1);
		}
		else
		{
			return (21* (ubTrack- 1)+ ubSector);
		}
	}
	else if (ubTrack<= 24)
	{
		if (ubSector> 19)
		{
			return (-1);
		}
		else
		{
			return (21* 17+ 19* (ubTrack- 18)+ ubSector);
		}
	}
	else if (ubTrack<= 30)
	{
		if (ubSector> 18)
		{
			return (-1);
		}
		else
		{
			return (21* 17+ 19* 7+ 18* (ubTrack- 25)+ ubSector);
		}
	}
	else if (ubTrack<= 35)
	{
		if (ubSector> 17)
		{
			return (-1);
		}
		else
		{
			return (21* 17+ 19* 7+ 18* 6+ 17* (ubTrack- 31)+ ubSector);
		}
	}
	else
	{
		return (-1);
	}
}

int SectorsOnTrack (BYTE sbTrack)
{
	if (sbTrack< 1)
	{
		return (0);
	}
	
	if (sbTrack< 18)
	{
		return (21);
	}
	
	if (sbTrack< 25)
	{
		return (19);
	}
	
	if (sbTrack< 31)
	{
		return (18);
	}
	
	if (sbTrack< 36)
	{
		return (17);
	}
	
	return (0);
}

UWORD Block2TS (UWORD uwBlock)
{
	UWORD uwTrack, uwSector, uwI, uwB= 0;

	for (uwTrack= 1; uwTrack< 36; uwTrack++)
	{
		uwI= SectorsOnTrack(uwTrack);
		for (uwSector= 0; uwSector< uwI; uwSector++)
		{
			if (uwB == uwBlock)
			{
				return ((UWORD)((uwTrack<< 8)| uwSector));
			}
			
			uwB++;
		}
	}
	return (0);
}

struct DataBlock *GetBlockTS (UBYTE ubTrack, UBYTE ubSector)
{
	LONG ulBlock= TS2Block(ubTrack, ubSector);

	if (ulBlock>= 0)
	{
		return (GetPutBlock(ulBlock, FALSE));
	}
	else
	{
		return (NULL);
	}
}

struct DataBlock *PutBlockTS (UBYTE ubTrack, UBYTE ubSector, APTR apData)
{
	LONG ulBlock= TS2Block(ubTrack, ubSector);

	if (ulBlock>= 0)
	{
		CopyMem(apData, diskimage+ (ulBlock* 256), 256);
		
		return (GetPutBlock(ulBlock, TRUE));
	}
	else
	{
		return (NULL);
	}
}

static int ErrorReq (ULONG ulBlock)
{
	static struct EasyStruct req= {
		sizeof(struct EasyStruct),
		0,
		"FS1541 Error",
		"Volume %ubSector has a\nread/bWrite error on\ntrack %ld, sector %ld.",
		"Retry|Ignore|Ignore All|Abort All" /* see REQ_ defines at the top of the file */
	};
	struct Process *pr= SDpSender->mp_SigTask;
	int x;

	if (!curvolumenode)
	{
		return -1;
	}

	if (abortall)
	{
		return REQ_ABORT_ALL;
	}
	
	if (ignoreall)
	{
		return REQ_IGNORE_ALL;
	}
	
	if (!(IntuitionBase= (struct IntuitionBase*)OpenLibrary("intuition.library", 36)))
	{
		return -1;
	}

	if ((SDpSender->mp_Flags & PF_ACTION) != PA_SIGNAL || pr->pr_Task.tc_Node.ln_Type != NT_PROCESS || pr->pr_WindowPtr != (APTR)-1)
	{
		UWORD ts= Block2TS(ulBlock);
		/* ULONG args[3]= { */
		ULONG args[3];
		args[0]=	(ULONG)&curvolumenode->name[1];
		args[1]=	(ULONG)(ts>> 8);
		args[2]=	(ULONG)(ts& 0xFF);
		/* }; */
		
		x= EasyRequestArgs(NULL, &req, NULL, args);
	}
	else
	{
		x= REQ_ABORT_ALL;
	}
	
	CloseLibrary((struct Library*)IntuitionBase);
	return x;
}

struct DataBlock *GetPutBlock (ULONG ulBlock, BOOL bWrite)
{
	int retry;
	int num= REQ_IGNORE;

	if (ulBlock>= 683)
	{
		return NULL;
	}

	if (bWrite)
	{
		CacheBlock(ulBlock, TRUE);
	}

	switch(sectab[ulBlock])
	{
		case SEC_NOT_LOADED:
			do
			{
				retry= 5;
			
				while (retry)
				{
					curblk= ulBlock;
					CacheBlock(ulBlock, FALSE);
				
					if (sectab[ulBlock]== SEC_OK)
					{
						return (struct DataBlock*)(diskimage+ (ulBlock* 256));
					}
					
					retry--;
				}
				
			} while((num= ErrorReq(ulBlock))== REQ_RETRY);
			/* fall through */
	
		case SEC_ERROR:
		default:
			switch(num)
			{
			case REQ_IGNORE_ALL:
				ignoreall= TRUE;
				/* fall through */
			case REQ_IGNORE:
				return (struct DataBlock*)(diskimage+ (ulBlock* 256));
	
			case REQ_ABORT_ALL:
			default:
				abortall= TRUE;
				return NULL;
			}
			
			return NULL;
			
		case SEC_OK:
			return (struct DataBlock*)(diskimage+ (ulBlock* 256));
	}
}

static void CacheBlock (ULONG n, BOOL bWrite)
{
	SExtDiskReq->iotd_Count= ulDiskChangeCount;
	SExtDiskReq->iotd_Req.io_Command= bWrite ? ETD_WRITE : ETD_READ;
		
	/* special Data for cbm sector'ubSector disk id */
	
	if (bSectorLabel)
	{
		SExtDiskReq->iotd_SecLabel= (ULONG)&(ubDiskIdAr[n][0]);		
	}
	else
	{
		SExtDiskReq->iotd_SecLabel= NULL;
	}
		
	SExtDiskReq->iotd_Req.io_Flags= 0;
	SExtDiskReq->iotd_Req.io_Data= diskimage+ (n* 256);
	SExtDiskReq->iotd_Req.io_Offset= 256* n;
	SExtDiskReq->iotd_Req.io_Length= 256;

	if (!DoIO((struct IORequest*)SExtDiskReq))
	{
		sectab[n]= SEC_OK;
	}
	else
	{
		sectab[n]= SEC_ERROR;
		lNumSoftErrors++;

		if (!curvolumenode)
		{
			/* Switch off the motor, unread/writable disk. */
			
			SExtDiskReq->iotd_Req.io_Command= TD_MOTOR;
			SExtDiskReq->iotd_Req.io_Flags= 0;
			SExtDiskReq->iotd_Req.io_Length= 0;
			DoIO((struct IORequest*)SExtDiskReq);
		}
	}
}
