/* FFT_Math_ADE.cpp */

/* *********************************************************************** */
/* Author:- $VER FFT_Math_ADE.cpp_Version_1.00.00_(C)2023,_B.Walker,_G0LCU._Issued_as_GPL3. */
/* Using ADE the *NIX emulator suite for the AMIGA A1200(HD). */
/* Compiler called as:- */
/* g++ -Wall -Os -ansi -o FFT_Math_ADE FFT_Math_ADE.cpp<CR> */
/* OR... */
/* c++ -Wall -Os -o FFT_Math_ADE FFT_Math_ADE.cpp<CR> */
/* In AMIGA ADE mode the executable will need the 'strip' utility to make the code much smaller. */
/* Prompt> strip --strip-all FFT_Math_ADE<CR> */
/* *********************************************************************** */
/* Using g++ from GNU/Linux Mint 20.3. */
/* g++ -Wall -Wextra -pedantic -std=c++98 -Os -o FFT_Math_ADE FFT_Math_ADE.cpp<CR> */
/* *********************************************************************** */
/* Allow for major 'clang' fatal error bug in GNU/Linux Mint 20.3, (and MAYBE above)... */
/*
The error generated:-
amiga@AMIGA:~/Desktop/Workspace/C_CPP$ clang -Wall -std=c++98 -Os -o FFT_Math_ADE FFT_Math_ADE.cpp
FFT_Complex.cpp:18:10: fatal error: 'iostream' file not found
#include <iostream>
         ^~~~~~~~~~
1 error generated.
*/
/* The cure for the error above, assuming clang Version 9:- */
/* clang++-9 -I /usr/include/c++/9 -I /usr/include/x86_64-linux-gnu/c++/9 -L /usr/lib/gcc/x86_64-linux-gnu/9 -Wall -Wextra -pedantic -pedantic-errors -std=c++98 -Os -o FFT_Math_ADE FFT_Math_ADE.cpp<CR> */
/* *********************************************************************** */

#include <iostream>
#include <iomanip>
#include <cmath>
#include <cstdlib>

/* PI to 15 decimal places. */
#define PI 3.141592653589793

typedef struct Complex{double REAL;double IMAG;} complex_pair;

/* Recursive method. */
void FFT(complex_pair COMPLEX_ARRAY[], complex_pair COMPLEX_OUT[], int N, int STEP)
{
	int i=0;
	int JUMP=(STEP*2);
	complex_pair RI_PAIR;

	if(STEP<N)
	{
		/* Divide. */
		/* Even elements. */
		FFT(COMPLEX_OUT, COMPLEX_ARRAY, N, JUMP);
		/* Odd elements. */
		FFT((COMPLEX_OUT+STEP), (COMPLEX_ARRAY+STEP), N, JUMP);

		/* Conquer. */
		while(i<N)
		{
			RI_PAIR.REAL=std::cos(-PI*i/N)*(COMPLEX_OUT[i+STEP].REAL)-std::sin(-PI*i/N)*(COMPLEX_OUT[i+STEP].IMAG);
			RI_PAIR.IMAG=std::sin(-PI*i/N)*(COMPLEX_OUT[i+STEP].REAL)+std::cos(-PI*i/N)*(COMPLEX_OUT[i+STEP].IMAG);
			COMPLEX_ARRAY[i/2].REAL=COMPLEX_OUT[i].REAL+RI_PAIR.REAL;
			COMPLEX_ARRAY[i/2].IMAG=COMPLEX_OUT[i].IMAG+RI_PAIR.IMAG;
			COMPLEX_ARRAY[(i+N)/2].REAL=COMPLEX_OUT[i].REAL-RI_PAIR.REAL;
			COMPLEX_ARRAY[(i+N)/2].IMAG=COMPLEX_OUT[i].IMAG-RI_PAIR.IMAG;
			i=(i+(STEP*2));
		}
	}
	return;
}

int Fast_Fourier_Transform(complex_pair COMPLEX_ARRAY[], int N)
{
	int i=0;
	/* Allocate the required memory... */
	/* 'malloc()' is technically deprecated as of c++14. */
	/* This is compiling to c++98 for the AMIGA however! */
	/* It is by far easier to use in this context also! */
	complex_pair *COMPLEX_OUT=(complex_pair *)std::malloc(N*sizeof(complex_pair));

	if(COMPLEX_OUT==NULL)
	{
		return 0;
	}
	for(i=0;i<N;i++)
	{
		COMPLEX_OUT[i].REAL=COMPLEX_ARRAY[i].REAL;
		COMPLEX_OUT[i].IMAG=COMPLEX_ARRAY[i].IMAG;
	}

	FFT(COMPLEX_ARRAY, COMPLEX_OUT, N, 1);

	/* Free up used memory... */
	std::free(COMPLEX_OUT);
	COMPLEX_OUT=NULL;
	return 1;
}

int main(void)
{
	int i=0;
	/* 13 sample pairs with 3 off REAL padding of 0.5 to make a RADIX 2 array of arrays. */
	complex_pair COMPLEX_ARRAY[]={{1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0} ,{0.0, 0.0}, {0.0, 0.0} ,{0.0, 0.0}, {1.0, 0.0}, {0.5, 0.0}, {0.5, 0.0}, {0.5, 0.0}};
	const int LENGTH=sizeof(COMPLEX_ARRAY)/sizeof(COMPLEX_ARRAY[0]);
	/* Exit with a return code of one if not a power of 2! */
	if((LENGTH&(LENGTH-1))!=0)
	{
		std::cout << std::endl << "Power of 2 ERROR!" << std::endl;
		std::cout << "Padding or cropping is required to the nearest power of 2 elements." << std::endl;
		std::cout << "Exiting with a return code of 1!" << std::endl << std::endl;
		std::exit(1);
	}

	Fast_Fourier_Transform(COMPLEX_ARRAY, LENGTH);

	/* Display the results. */
	std::cout << std::setprecision(10) << std::endl << "FFT complex values:-" << std::endl;
	for(i=0;i<LENGTH;i++)
	{
		/* Use 'j' instead of 'i' in the imaginary part to avoid confusion. */
		if(COMPLEX_ARRAY[i].IMAG>=0.0)
		{
			std::cout << "(" << COMPLEX_ARRAY[i].REAL << "+" << COMPLEX_ARRAY[i].IMAG << "j)  ";
		}
		/* Remove the 'double negative' displayed on screen. */
		if(COMPLEX_ARRAY[i].IMAG<0.0)
		{
			std::cout << "(" << COMPLEX_ARRAY[i].REAL << "-" << std::fabs(COMPLEX_ARRAY[i].IMAG) << "j)  ";
		}
	}
	std::cout << std::endl;
	/* Display the absolute results... */
	std::cout << std::endl << std::setprecision(5) << "FFT absolute values:-" << std::endl;
	for(i=0;i<LENGTH;i++)
	{
		std::cout
		<< std::abs(std::sqrt(((COMPLEX_ARRAY[i].REAL)*(COMPLEX_ARRAY[i].REAL))+((COMPLEX_ARRAY[i].IMAG)*(COMPLEX_ARRAY[i].IMAG))))
		<< " ";
	}
	std::cout << std::endl << std::endl;

	/* Using exit(0) by choice. */
	std::exit(0);
}

/*
# What to expect for the above 16 sample data.
# NOTE: 'j' used instead of 'i' to remove any ambiguity.
#
# FFT complex values:-
# (8.5+0j)  (3.637821187-1.930714406j)  (-1.707106781-0.5j)
# (0.05142521051-0.2414680083j)  (1.5-1j)  (-0.1727455541+0.4656387729j)
# (-0.2928932188+0.5j)  (0.4834991568-1.223607624j)  (0.5+0j)
# (0.4834991568+1.223607624j)  (-0.2928932188-0.5j)
# (-0.1727455541-0.4656387729j)  (1.5+1j)  (0.05142521051+0.2414680083j)
# (-1.707106781+0.5j)  (3.637821187+1.930714406j)
#
# FFT absolute values:-
# 8.5 4.1184 1.7788 0.24688 1.8028 0.49665 0.57947 1.3157 0.5 1.3157 0.57947
# 0.49665 1.8028 0.24688 1.7788 4.1184
*/

