/* FFT_VBCC_GCC.c */

/* Author:- B.Walker, G0LCU, Issued as CC0, Public Domain... */
/* ********************************************************** */
/* Written for the AMIGA A1200, ADE *NIX emulator. */
/* gcc version 2.95.3 20010315 (release) */
/* gcc -Wall -o FFT_VBCC_GCC FFT_VBCC_GCC.c -lm<CR> */
/* ********************************************************** */
/* Using the VBCC C compiler for the AMIGA. */
/* VBCC Version:- $VER: vbcc 0.810 (05.06.2006) */
/* vc FFT_VBCC_GCC.c -o FFT_VBCC_GCC -lmieee<CR> */
/* ********************************************************** */
/* IMPORTANT NOTE! There is no 'complex.h' in the above gcc and VBCC suites for the AMIGA. */
/* Hence this is a 'math.h' version instead. */
/* It might well compile using Dice-C but not tried. */

/* Apple gcc --version brings up...... */
/* Apple LLVM version 10.0.0 (clang-1000.11.45.5) */
/* Target: x86_64-apple-darwin19.6.0 */
/* Thread model: posix */
/* clang -Wall -pedantic -ansi -std=c99 -o FFT_VBCC_GCC FFT_VBCC_GCC.c -lm<CR> */

/* gcc (Ubuntu 9.3.0-17ubuntu1~20.04) 9.3.0 */
/* Copyright (C) 2019 Free Software Foundation, Inc. */
/* This is free software; see the source for copying conditions.  There is NO */
/* warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. */
/* gcc -Wall -pedantic -ansi -std=c99 -o FFT_VBCC_GCC FFT_VBCC_GCC.c -lm<CR> */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define PI 3.141592653589793

typedef struct Complex
{
	double REAL;
	double IMAG;
} complex_pair;

/* Recursive method. */
void FFT(complex_pair COMPLEX_ARRAY[], complex_pair COMPLEX_OUT[], int N, int STEP)
{
	int i=0;
	complex_pair RI_PAIR;

	if (STEP<N)
	{
		/* Even elements. */
		FFT(COMPLEX_OUT, COMPLEX_ARRAY, N, (STEP*2));
		/* Odd elements. */
		FFT((COMPLEX_OUT+STEP), (COMPLEX_ARRAY+STEP), N, (STEP*2));
		while (i<N)
		{
			RI_PAIR.REAL=cos(-PI*i/N)*(COMPLEX_OUT[i+STEP].REAL)-sin(-PI*i/N)*(COMPLEX_OUT[i+STEP].IMAG);
			RI_PAIR.IMAG=sin(-PI*i/N)*(COMPLEX_OUT[i+STEP].REAL)+cos(-PI*i/N)*(COMPLEX_OUT[i+STEP].IMAG);
			COMPLEX_ARRAY[i/2].REAL=COMPLEX_OUT[i].REAL+RI_PAIR.REAL;
			COMPLEX_ARRAY[i/2].IMAG=COMPLEX_OUT[i].IMAG+RI_PAIR.IMAG;
			COMPLEX_ARRAY[(i+N)/2].REAL=COMPLEX_OUT[i].REAL-RI_PAIR.REAL;
			COMPLEX_ARRAY[(i+N)/2].IMAG=COMPLEX_OUT[i].IMAG-RI_PAIR.IMAG;
			i=(i+(STEP*2));
		}
	}
}

int Fast_Fourier_Transform(complex_pair COMPLEX_ARRAY[], int N)
{
	int i=0;
	/* Allocate he required memory... */
	complex_pair *COMPLEX_OUT=(complex_pair *)malloc(N*sizeof(complex_pair));

	if (COMPLEX_OUT==NULL)
	{
		return 0;
	}
	for (i=0; i<N; i++)
	{
		COMPLEX_OUT[i].REAL=COMPLEX_ARRAY[i].REAL;
		COMPLEX_OUT[i].IMAG=COMPLEX_ARRAY[i].IMAG;
	}

	FFT(COMPLEX_ARRAY, COMPLEX_OUT, N, 1);

	/* Free up used memory... */
	free(COMPLEX_OUT);

	return 0;
}

int main()
{
	/* 13 sample pairs with 3 off REAL padding of 0.5 to make a RADIX 2 array of arrays. */
	complex_pair COMPLEX_ARRAY[]={{1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0} ,{0.0, 0.0}, {0.0, 0.0} ,{0.0, 0.0}, {1.0, 0.0}, {0.5, 0.0}, {0.5, 0.0}, {0.5, 0.0}};
	/* The standard test of 8 sample pairs to make a RADIX 2 array of arrays. */
	/* complex_pair COMPLEX_ARRAY[]={{1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {1.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0} ,{0.0, 0.0}}; */
	/* See the results below... */
	int LENGTH=sizeof(COMPLEX_ARRAY)/sizeof(COMPLEX_ARRAY[0]);
	int i=0;
	if ((LENGTH&(LENGTH-1))!=0)
	{
		printf("\nPower of 2 ERROR!\n");
		printf("Padding or cropping is required to the nearest power of 2 elements.\n");
		printf("Exiting with a return code of 1!\n\n");
		exit(1);
	}

	Fast_Fourier_Transform(COMPLEX_ARRAY, LENGTH);

	/* Display the results. */
	printf("\nFFT array values:-\n      Real:     Imaginary:\n");
	for (i=0; i<LENGTH; i++)
	{
		if (COMPLEX_ARRAY[i].IMAG>=0.0)
		{
			printf("(%.10f+%.10fj)\n", COMPLEX_ARRAY[i].REAL, COMPLEX_ARRAY[i].IMAG);
		}
		/* Remove the 'double negative' displayed on screen. */
		if (COMPLEX_ARRAY[i].IMAG<0.0)
		{
			printf("(%.10f-%.10fj)\n", COMPLEX_ARRAY[i].REAL, fabs(COMPLEX_ARRAY[i].IMAG));
		}
	}
	printf("\n");

	exit(0);
}

/*
# What to expect for the above 16 sample data.
# NOTE: 'j' used instead of'i' to remove any ambiguity.
#
# (8.5000000000+0.0000000000j)
# (3.6378211867-1.9307144055j)
# (-1.7071067812-0.5000000000j)
# (0.0514252105-0.2414680083j)
# (1.5000000000-1.0000000000j)
# (-0.1727455541+0.4656387729j)
# (-0.2928932188+0.5000000000j)
# (0.4834991568-1.2236076244j)
# (0.5000000000+0.0000000000j)
# (0.4834991568+1.2236076244j)
# (-0.2928932188-0.5000000000j)
# (-0.1727455541-0.4656387729j)
# (1.5000000000+1.0000000000j)
# (0.0514252105+0.2414680083j)
# (-1.7071067812+0.5000000000j)
# (3.6378211867+1.9307144055j)
#
# *****************************
#
# The standard 8 sample test results.
#
# (4.0000000000+0.0000000000j)
# (1.0000000000-2.4142135624j)
# (-0.0000000000-0.0000000000j)
# (1.0000000000-0.4142135624j)
# (0.0000000000-0.0000000000j)
# (1.0000000000+0.4142135624j)
# (0.0000000000-0.0000000000j)
# (1.0000000000+2.4142135624j)
#
*/

