﻿/* 
 * IFF-ILBM library v1.0 (2016) by Ilkka Lehtoranta (ilkleht (at) yahoo.com)
 *
 * Public Domain
 */

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Text;

namespace IFF
{
	public class ILBM
	{
		#region IFF-ILBM IDs
		static byte[] FORM_ID = new byte[4] { (byte)'F', (byte)'O', (byte)'R', (byte)'M' };
		static byte[] PBM_ID = new byte[4] { (byte)'P', (byte)'B', (byte)'M', (byte)' ' };
		static byte[] ILBM_ID = new byte[4] { (byte)'I', (byte)'L', (byte)'B', (byte)'M' };
		static byte[] BMHD_ID = new byte[4] { (byte)'B', (byte)'M', (byte)'H', (byte)'D' };
		static byte[] BODY_ID = new byte[4] { (byte)'B', (byte)'O', (byte)'D', (byte)'Y' };
		static byte[] CMAP_ID = new byte[4] { (byte)'C', (byte)'M', (byte)'A', (byte)'P' };
		static byte[] CRGN_ID = new byte[4] { (byte)'C', (byte)'R', (byte)'G', (byte)'N' };
		#endregion

		#region Enumerations
		public enum Mask
		{
			None = 0,
			Masked,
			Transparent,
			Lasso
		}
		#endregion

		#region Utility methods
		private static int Read32(BinaryReader r)
		{
			var arr = r.ReadBytes(4);
			return arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3];
		}

		private static int Read16(BinaryReader r)
		{
			var arr = r.ReadBytes(2);
			return arr[0] << 8 | arr[1];
		}
		#endregion

		#region ILBMdata
		protected internal class ILBMdata
		{
			int Width, Height;
			int BitsPerPixel;
			bool IsCompressed;

			Mask MaskType;

			internal byte[] colormap;
			private readonly uint[] palette = new uint[256];

			#region Decompress
			/// <summary>
			/// Byterun decompression
			/// </summary>
			/// <param name="length"></param>
			/// <param name="r"></param>
			/// <returns></returns>
			private byte[] Decompress(BinaryReader r, int size, byte[] body)
			{
				var decomp = new byte[size];
				int srcidx = 0;
				int dstidx = 0;

				try
				{
					while (srcidx < body.Length)
					{
						int byteval = body[srcidx++];

						if (byteval <= 0x7f)
						{
							byteval++;
							Array.Copy(body, srcidx, decomp, dstidx, byteval);

							srcidx += byteval;
							dstidx += byteval;
						}
						else if (byteval > 0x80)
						{
							byteval = 257 - byteval;
							byte data = body[srcidx++];

							for (int i = 0; i < byteval; i++)
							{
								decomp[dstidx++] = data;
							}
						}
					}
				}
				catch
				{
					throw;
				}

				return decomp;
			}
			#endregion

			#region ReadBMHD
			internal void ReadBMHD(BinaryReader r, int size)
			{
				Width = Read16(r);
				Height = Read16(r);

				r.BaseStream.Seek(4, SeekOrigin.Current);
				BitsPerPixel = r.ReadByte();
				MaskType = (Mask)r.ReadByte();

				IsCompressed = r.ReadByte() == 0 ? false : true;
			}
			#endregion

			#region ReadBody
			internal Bitmap ReadBody(BinaryReader br, int size)
			{
				if (colormap != null)
				{
					int i = 0;

					while (i * 3 + 3 <= colormap.Length)
					{
						uint r = colormap[i * 3];
						uint g = colormap[i * 3 + 1];
						uint b = colormap[i * 3 + 2];

						palette[i] = (uint)(0xff000000 | (r << 16) | (g << 8) | b);
						i++;
					}
				}

				var bm = new Bitmap(Width, Height, System.Drawing.Imaging.PixelFormat.Format8bppIndexed);

				ColorPalette pal = bm.Palette;

				for (int i = 0; i < 256; i++)
				{
					pal.Entries[i] = Color.FromArgb((int)palette[i]);
				}

				bm.Palette = pal;

				var body = br.ReadBytes(size);
				ReadPixels(br, body, bm);

				return bm;
			}

			private unsafe void ReadPixels(BinaryReader br, byte[] body, Bitmap bm)
			{
				int bpr = (Width + 7) / 8 * BitsPerPixel;

				if (bpr % 2 != 0)
					bpr++;

				var data = IsCompressed ? Decompress(br, bpr * Height, body) : body;
				int rot = 0, tmp = 0;

				var lockdata = bm.LockBits(new Rectangle(0, 0, Width, Height), ImageLockMode.WriteOnly, bm.PixelFormat);

				try
				{
					byte* dst = (byte*)lockdata.Scan0.ToPointer();

					for (int y = 0; y < Height; y++)
					{
						rot = 0; tmp = y * bpr * (MaskType == Mask.Masked ? 2 : 1);

						if (BitsPerPixel == 8)
						{
							System.Runtime.InteropServices.Marshal.Copy(data, tmp, lockdata.Scan0 + y * lockdata.Stride, Width);
						}
						else
						{
							for (int x = 0; x < Width; x++)
							{
								int idx = (Int16)(data[tmp + x / BitsPerPixel] << 8 | data[tmp + x / BitsPerPixel + 1]);
								idx <<= rot;
								idx &= 0xffff;
								idx >>= 16 - BitsPerPixel;
								rot += BitsPerPixel;
								rot %= 8;

								dst[y * lockdata.Stride + x] = (byte)idx;
							}
						}
					}
				}
				catch
				{
					throw;
				}
				finally
				{
					bm.UnlockBits(lockdata);
				}
			}
			#endregion
		}
		#endregion

		#region CheckID
		private static bool CheckID(byte[] id1, byte[] id2)
		{
			bool rc = true;

			for (int i = 0; i < 4; i++)
			{
				if (id1[i] != id2[i])
				{
					rc = false;
					break;
				}
			}

			return rc;
		}
		#endregion

		#region Load
		/// <summary>
		/// Load ILBM to bitmap.
		/// </summary>
		/// <param name="filename"></param>
		/// <returns></returns>
		public static Bitmap Load(string filename)
		{
			Bitmap bm = null;

			using (var strm = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{
				using (var r = new BinaryReader(strm))
				{
					var ilbm = new ILBMdata();

					var id = r.ReadBytes(4);

					if (CheckID(id, FORM_ID))
					{
						r.BaseStream.Seek(4, SeekOrigin.Current); // This four bytes contain size but we are not interested...
						var tp = r.ReadBytes(4);

						if (CheckID(tp, PBM_ID) || CheckID(tp, ILBM_ID))
						{
							while (r.BaseStream.Position < r.BaseStream.Length - 1)
							{
								id = r.ReadBytes(4);
								int sz = Read32(r);
								long pos = r.BaseStream.Position;

								// DPPS - Deluxe Paint Page Size
								// CRNG - Color Range
								if (CheckID(id, BMHD_ID))
								{
									ilbm.ReadBMHD(r, sz);
								}
								else if (CheckID(id, BODY_ID))
								{
									bm = ilbm.ReadBody(r, sz);
								}
								else if (CheckID(id, CMAP_ID))
								{
									ilbm.colormap = r.ReadBytes(sz);
								}

								r.BaseStream.Seek(pos + sz, SeekOrigin.Begin);
							}
						}
					}
				}
			}

			return bm;
		}
		#endregion
	}
}
