//----------------------------------------------------------------------------
// Anti-Grain Geometry - Version 2.4
// Copyright (C) 2002-2005 Maxim Shemanarev (http://www.antigrain.com)
//
// Permission to copy, use, modify, sell and distribute this software 
// is granted provided this copyright notice appears in all copies. 
// This software is provided "as is" without express or implied
// warranty, and with no claim as to its suitability for any purpose.
//
//----------------------------------------------------------------------------
// Contact: mcseem@antigrain.com
//          mcseemagg@yahoo.com
//          http://www.antigrain.com
//----------------------------------------------------------------------------
//
// class platform_support
//
//----------------------------------------------------------------------------

#include "platform/agg_platform_support.h"
#include "util/agg_color_conv_rgb8.h"

#include <sys/time.h>
#include <cstring>

#include <datatypes/pictureclass.h>
#include <proto/exec.h>
#include <proto/dos.h>
#include <proto/datatypes.h>
#include <proto/graphics.h>
#include <proto/intuition.h>
#include <proto/keymap.h>
#include <libraries/cybergraphics.h>
#include <proto/cybergraphics.h>
#include <proto/utility.h>

#define FMT_UNKNOWN (99)

namespace agg
{
	typedef int32_t RGBFTYPE;

	//------------------------------------------------------------------------
	class platform_specific
	{
	public:
		platform_specific(platform_support& support, pix_format_e format,
			bool flip_y);
		~platform_specific();
		bool handle_input();
		bool load_img(const char* file, unsigned idx, rendering_buffer* rbuf);
		bool create_img(unsigned idx, rendering_buffer* rbuf, unsigned width, unsigned height);
		bool make_bitmap();
	public:
		platform_support& m_support;
		RGBFTYPE m_ftype;	// cgx RECTFMT_...
		pix_format_e m_format;	// AGG format
		unsigned m_bpp;		// bits per pixel
		int8u* m_buf_window;	// byte array for window
		bool m_flip_y;
		uint16_t m_width;
		uint16_t m_height;
		Window* m_window;
		Screen* m_screen;
		unsigned m_input_flags;
		bool m_dragging;
		double m_start_time;
		uint16_t m_last_key;
		int8u* m_buf_img[platform_support::max_images];
	};

	//------------------------------------------------------------------------
	platform_specific::platform_specific(platform_support& support,
		pix_format_e format, bool flip_y) :
		m_support(support),
		m_ftype(FMT_UNKNOWN),
		m_format(format),
		m_bpp(0),
		m_buf_window(0),
		m_flip_y(flip_y),
		m_width(0),
		m_height(0),
		m_window(0),
		m_input_flags(0),
		m_dragging(false),
		m_start_time(0.0),
		m_last_key(0)
	{
		switch ( format )
		{
		case pix_format_gray8:
			m_ftype = RECTFMT_GREY8;
			m_bpp = 8;
			break;
		case pix_format_rgb555:
			m_ftype = RECTFMT_RGB15;
			m_bpp = 15;
			break;
		case pix_format_rgb565:
			m_ftype = RECTFMT_RGB16;
			m_bpp = 16;
			break;
		case pix_format_rgb24:
			m_ftype = RECTFMT_RGB24;
			m_bpp = 24;
			break;
		case pix_format_bgr24:
			m_ftype = RECTFMT_BGR24;
			m_bpp = 24;
			break;
		case pix_format_bgra32:
			m_ftype = RECTFMT_BGRA32;
			m_bpp = 32;
			break;
		case pix_format_abgr32:
			m_ftype = RECTFMT_ABGR32;
			m_bpp = 32;
			break;
		case pix_format_argb32:
			m_ftype = RECTFMT_ARGB32;
			m_bpp = 32;
			break;
		case pix_format_rgba32:
			m_ftype = RECTFMT_RGBA32;
			m_bpp = 32;
			break;
		}

		for ( unsigned i = 0; i < platform_support::max_images; ++i )
		{
			m_buf_img[i] = 0;
		}

		m_screen = LockPubScreen(NULL);
	}

	//------------------------------------------------------------------------
	platform_specific::~platform_specific()
	{
		UnlockPubScreen(NULL, m_screen);
		CloseWindow(m_window);

		FreeVec(m_buf_window);

		for ( unsigned i = 0; i < platform_support::max_images; ++i )
		{
			FreeVec(m_buf_img[i]);
		}
	}

	//------------------------------------------------------------------------
	bool platform_specific::handle_input()
	{
		struct IntuiMessage *message;
		platform_support* app = &m_support;
		Window* window = app->m_specific->m_window;

		while (NULL != (message = (struct IntuiMessage *)GetMsg(m_window->UserPort)))
		{
			int16 x = message->MouseX - window->BorderLeft;
			int16 y = 0;
			if ( app->flip_y() )
			{
				y = window->Height - window->BorderBottom - message->MouseY;
			}
			else
			{
				y = message->MouseY - window->BorderTop;
			}

			switch ( message->Class )
			{
			case IDCMP_CLOSEWINDOW:
				return true;
				break;
			case IDCMP_INTUITICKS:
				if ( !m_support.wait_mode() )
				{
					m_support.on_idle();
				}
				break;
			case IDCMP_NEWSIZE:
				if ( make_bitmap() )
				{
					m_support.trans_affine_resizing(m_width, m_height);
					m_support.on_resize(m_width, m_height);
					m_support.force_redraw();
				}
				break;
			case IDCMP_MOUSEBUTTONS:
				if ( message->Code & IECODE_UP_PREFIX )
				{
					if ( message->Code == SELECTUP )
					{
						app->m_specific->m_input_flags = mouse_left;
						app->m_specific->m_dragging = false;
					}
					else if ( message->Code == MENUUP )
					{
						app->m_specific->m_input_flags = mouse_right;
						app->m_specific->m_dragging = false;
					}
					else
					{
						break;
					}

					if ( app->m_ctrls.on_mouse_button_up(x, y) )
					{
						app->on_ctrl_change();
						app->force_redraw();
					}

					app->on_mouse_button_up(x, y, app->m_specific->m_input_flags);
				}
				else
				{
					if ( message->Code == SELECTDOWN )
					{
						app->m_specific->m_input_flags = mouse_left;
						app->m_specific->m_dragging = true;
					}
					else if ( message->Code == MENUDOWN )
					{
						app->m_specific->m_input_flags = mouse_right;
						app->m_specific->m_dragging = true;
					}
					else
					{
						break;
					}

					app->m_ctrls.set_cur(x, y);
					if ( app->m_ctrls.on_mouse_button_down(x, y) )
					{
						app->on_ctrl_change();
						app->force_redraw();
					}
					else
					{
						if ( app->m_ctrls.in_rect(x, y) )
						{
							if ( app->m_ctrls.set_cur(x, y) )
							{
								app->on_ctrl_change();
								app->force_redraw();
							}
						}
						else
						{
							app->on_mouse_button_down(x, y,
								app->m_specific->m_input_flags);
						}
					}
				}
				break;
			case IDCMP_MOUSEMOVE:
				if ( app->m_specific->m_dragging )  {
					if ( app->m_ctrls.on_mouse_move(x, y,
						app->m_specific->m_input_flags & mouse_left) != 0 )
					{
						app->on_ctrl_change();
						app->force_redraw();
					}
					else
					{
						if ( !app->m_ctrls.in_rect(x, y) )
						{
							app->on_mouse_move(x, y,
								app->m_specific->m_input_flags);
						}
					}
				}
				break;
			case IDCMP_RAWKEY:
			{
				static InputEvent ie = { 0 };
				ie.ie_Class = IECLASS_RAWKEY;
				ie.ie_Code = message->Code;
				ie.ie_Qualifier = message->Qualifier;

				static const unsigned BUF_SIZE = 16;
				static char key_buf[BUF_SIZE];
				int16 num_chars = MapRawKey(&ie, key_buf, BUF_SIZE, 0);

				uint32_t code = 0x00000000;
				switch ( num_chars )
				{
				case 1:
					code = key_buf[0];
					break;
				case 2:
					code = key_buf[0]<<8 | key_buf[1];
					break;
				case 3:
					code = key_buf[0]<<16 | key_buf[1]<<8 | key_buf[2];
					break;
				}

				uint16_t key_code = 0;

				if ( num_chars == 1 )
				{
					if ( code >= IECODE_ASCII_FIRST && code <= IECODE_ASCII_LAST )
					{
						key_code = code;
					}
				}

				if ( key_code == 0 )
				{
					switch ( code )
					{
					case 0x00000008: key_code = key_backspace;	break;
					case 0x00000009: key_code = key_tab;		break;
					case 0x0000000D: key_code = key_return;		break;
					case 0x0000001B: key_code = key_escape;		break;
					case 0x0000007F: key_code = key_delete;		break;
					case 0x00009B41:
					case 0x00009B54: key_code = key_up;		break;
					case 0x00009B42:
					case 0x00009B53: key_code = key_down;		break;
					case 0x00009B43:
					case 0x009B2040: key_code = key_right;		break;
					case 0x00009B44:
					case 0x009B2041: key_code = key_left;		break;
					case 0x009B307E: key_code = key_f1;		break;
					case 0x009B317E: key_code = key_f2;		break;
					case 0x009B327E: key_code = key_f3;		break;
					case 0x009B337E: key_code = key_f4;		break;
					case 0x009B347E: key_code = key_f5;		break;
					case 0x009B357E: key_code = key_f6;		break;
					case 0x009B367E: key_code = key_f7;		break;
					case 0x009B377E: key_code = key_f8;		break;
					case 0x009B387E: key_code = key_f9;		break;
					case 0x009B397E: key_code = key_f10;		break;
					case 0x009B3F7E: key_code = key_scrollock;	break;
					}
				}

				if ( ie.ie_Code & IECODE_UP_PREFIX )
				{
					if ( app->m_specific->m_last_key != 0 )
					{
						bool left = (key_code == key_left) ? true : false;
						bool right = (key_code == key_right) ? true : false;
						bool down = (key_code == key_down) ? true : false;
						bool up = (key_code == key_up) ? true : false;

						if ( app->m_ctrls.on_arrow_keys(left, right, down, up) )
						{
							app->on_ctrl_change();
							app->force_redraw();
						}
						else
						{
							app->on_key(x, y, app->m_specific->m_last_key, 0);
						}

						app->m_specific->m_last_key = 0;
					}
				}
				else
				{
					app->m_specific->m_last_key = key_code;
				}
				break;
			}
			default:
				break;
			}
			ReplyMsg((struct Message*)message);
		}

		return false;
	}

	//------------------------------------------------------------------------
	bool platform_specific::load_img(const char* file, unsigned idx,
		rendering_buffer* rbuf)
	{
		if ( m_buf_img[idx] != 0 )
		{
			FreeVec(m_buf_img[idx]);
			m_buf_img[idx] = 0;
		}

		bool result = false;
		Object* picture = NewDTObject(const_cast<STRPTR>(file),
			DTA_GroupID, GID_PICTURE,
			PDTA_DestMode, PMODE_V43,
			PDTA_Remap, TRUE,
			PDTA_Screen, m_screen,
			TAG_END);
		if ( picture != 0 )
		{
			gpLayout layout;
			layout.MethodID = DTM_PROCLAYOUT;
			layout.gpl_GInfo = 0;
			layout.gpl_Initial = 1;
			ULONG loaded = DoDTMethodA(picture, 0, 0, reinterpret_cast<Msg>(&layout));
			if ( 1 /* loaded != 0 */)
			{
				BitMapHeader *bmh;
				BitMap *bm;
				GetDTAttrs(picture,
					PDTA_BitMapHeader, &bmh,
					PDTA_DestBitMap, &bm,
					TAG_END);

				RastPort *rp = CreateRastPort();
				rp->BitMap = bm;

				uint16_t width = bmh->bmh_Width;
				uint16_t height = bmh->bmh_Height;
				int bpr = width * 4;
				m_buf_img[idx] = reinterpret_cast<int8u*>AllocVec(width * height * 4, MEMF_ANY);
				if (m_buf_img[idx])
				{
					ReadPixelArray
					(
						m_buf_img[idx],
						0, 0,
						bpr,
						rp,
						0, 0,
						width, height,
						RECTFMT_ARGB
					);
					int8u* buf = m_buf_img[idx];
					int stride = (m_flip_y) ? -bpr : bpr;
					rbuf->attach(buf, width, height, stride);
					result = true;
				}
				FreeRastPort(rp);
			}
		}

		DisposeDTObject(picture);

		return result;
	}

	//------------------------------------------------------------------------
	bool platform_specific::create_img(unsigned idx, rendering_buffer* rbuf,
		unsigned width, unsigned height)
	{
		if ( m_buf_img[idx] != 0 )
		{
			FreeVec(m_buf_img[idx]);
			m_buf_img[idx] = 0;
		}

		m_buf_img[idx] = reinterpret_cast<int8u*>AllocVec(width * height * (m_bpp/8), MEMF_CLEAR);
		if ( m_buf_img[idx] != 0 )
		{
			int8u* buf = m_buf_img[idx];
			int bpr = width * (m_bpp/8);
			int stride = (m_flip_y) ? -bpr : bpr;

			rbuf->attach(buf, width, height, stride);

			return true;
		}

		return false;
	}

	//------------------------------------------------------------------------
	bool platform_specific::make_bitmap()
	{
		uint32_t width  = m_window->Width - m_window->BorderLeft - m_window->BorderRight;
		uint32_t height = m_window->Height - m_window->BorderTop - m_window->BorderBottom;

		int8u* buf = reinterpret_cast<int8u*>(AllocVec(width * height * (m_bpp/8), MEMF_CLEAR));
		if ( buf == 0 )
		{
			return false;
		}

		int bpr = width * (m_bpp/8);
		int stride = (m_flip_y) ? -bpr : bpr;

		m_support.rbuf_window().attach(buf, width, height, stride);

		m_buf_window = buf;
		m_width = width;
		m_height = height;

		return true;
	}

	//------------------------------------------------------------------------
	platform_support::platform_support(pix_format_e format, bool flip_y) :
		m_specific(new platform_specific(*this, format, flip_y)),
		m_format(format),
		m_bpp(m_specific->m_bpp),
		m_window_flags(0),
		m_wait_mode(true),
		m_flip_y(flip_y),
		m_initial_width(10),
		m_initial_height(10)
	{
		std::strncpy(m_caption, "Anti-Grain Geometry", 256);
	}

	//------------------------------------------------------------------------
	platform_support::~platform_support()
	{
		delete m_specific;
	}

	//------------------------------------------------------------------------
	void platform_support::caption(const char* cap)
	{
		std::strncpy(m_caption, cap, 256);
		if ( m_specific->m_window != 0 )
		{
			SetWindowTitles(m_specific->m_window, m_caption, (const char*)-1);
		}
	}

	//------------------------------------------------------------------------
	void platform_support::start_timer()
	{
		timeval tv;
		gettimeofday(&tv, 0);
		m_specific->m_start_time = tv.tv_secs + tv.tv_micro/1e6;
	}

	//------------------------------------------------------------------------
	double platform_support::elapsed_time() const
	{
		timeval tv;
		gettimeofday(&tv, 0);
		double end_time = tv.tv_secs + tv.tv_micro/1e6;

		double elasped_seconds = end_time - m_specific->m_start_time;
		double elasped_millis = elasped_seconds*1e3;

		return elasped_millis;
	}

	//------------------------------------------------------------------------
	void* platform_support::raw_display_handler()
	{
		return 0;	// Not available.
	}

	//------------------------------------------------------------------------
	void platform_support::message(const char* msg)
	{
		struct EasyStruct es = {
			sizeof(struct EasyStruct),
			0,
			"Anti-Grain Geometry",
			msg,
			"_OK"
		};

		EasyRequestArgs( NULL, &es, 0, 0 );
	}

	//------------------------------------------------------------------------
	bool platform_support::init(unsigned width, unsigned height,
		unsigned flags)
	{
		if( m_specific->m_ftype == FMT_UNKNOWN )
		{
			message("Unsupported mode requested.");
			return false;
		}

		m_window_flags = flags;

		m_specific->m_window = OpenWindowTags(NULL,
				WA_Title, m_caption,
				//WA_AutoAdjustDClip, TRUE,
				WA_InnerWidth, width,
				WA_InnerHeight, height,
				WA_Activate, TRUE,
				WA_SmartRefresh, TRUE,
				WA_NoCareRefresh, TRUE,
				WA_CloseGadget, TRUE,
				WA_DepthGadget, TRUE,
				WA_SizeGadget, (flags & agg::window_resize) ? TRUE : FALSE,
				WA_MaxWidth, -1,
				WA_MaxHeight, -1,
				WA_DragBar, TRUE,
				WA_AutoAdjust, TRUE,
				WA_ReportMouse, TRUE,
				WA_RMBTrap, TRUE,
				WA_MouseQueue, 1,
				WA_IDCMP,
					IDCMP_CLOSEWINDOW |
					IDCMP_NEWSIZE |
					IDCMP_MOUSEBUTTONS |
					IDCMP_MOUSEMOVE |
					IDCMP_RAWKEY |
					IDCMP_INTUITICKS,
				TAG_END);

		if ( m_specific->m_window == 0 )
		{
			return false;
		}

		if ( !m_specific->make_bitmap() )
		{
			return false;
		}

		m_initial_width = width;
		m_initial_height = height;

		on_init();
		on_resize(width, height);
		force_redraw();

		return true;
	}

	//------------------------------------------------------------------------
	int platform_support::run()
	{
		uint32_t window_mask = 1L << m_specific->m_window->UserPort->mp_SigBit;
		uint32_t wait_mask = window_mask | SIGBREAKF_CTRL_C;

		bool done = false;

		while ( !done )
		{
			uint32_t sig_mask = Wait(wait_mask);
			if ( sig_mask & SIGBREAKF_CTRL_C )
			{
				done = true;
			}
			else
			{
				done = m_specific->handle_input();
			}
		}

		return 0;
	}

	//------------------------------------------------------------------------
	const char* platform_support::img_ext() const
	{
		return ".bmp";
	}

	//------------------------------------------------------------------------
	const char* platform_support::full_file_name(const char* file_name)
	{
		return file_name;
	}

	//------------------------------------------------------------------------
	bool platform_support::load_img(unsigned idx, const char* file)
	{
		if ( idx < max_images )
		{
			static char fn[1024];
			std::strncpy(fn, file, 1024);
			int len = std::strlen(fn);
			if ( len < 4 || std::strcmp(fn + len - 4, ".bmp") != 0 )
			{
				std::strncat(fn, ".bmp", 1024);
			}

			return m_specific->load_img(fn, idx, &m_rbuf_img[idx]);
		}

		return false;
	}

	//------------------------------------------------------------------------
	bool platform_support::save_img(unsigned idx, const char* file)
	{
		message("Not supported");
		return false;
	}

	//------------------------------------------------------------------------
	bool platform_support::create_img(unsigned idx, unsigned width,
		unsigned height)
	{
		if ( idx < max_images )
		{
			if ( width == 0 )
			{
				width = m_specific->m_width;
			}

			if ( height == 0 )
			{
				height = m_specific->m_height;
			}

			return m_specific->create_img(idx, &m_rbuf_img[idx], width,
				height);
		}

		return false;
	}

	//------------------------------------------------------------------------
	void platform_support::force_redraw()
	{
		on_draw();
		update_window();
	}

	//------------------------------------------------------------------------
	void platform_support::update_window()
	{
		// Note this function does automatic color conversion.
		WritePixelArray
		(
			m_specific->m_buf_window,
			0, 0,
			m_specific->m_width * m_specific->m_bpp/8,
			m_specific->m_window->RPort,
			m_specific->m_window->BorderLeft,
			m_specific->m_window->BorderTop,
			m_specific->m_width,
			m_specific->m_height,
			m_specific->m_ftype
		);
	}

	//------------------------------------------------------------------------
	void platform_support::on_init() {}
	void platform_support::on_resize(int sx, int sy) {}
	void platform_support::on_idle() {}
	void platform_support::on_mouse_move(int x, int y, unsigned flags) {}
	void platform_support::on_mouse_button_down(int x, int y, unsigned flags) {}
	void platform_support::on_mouse_button_up(int x, int y, unsigned flags) {}
	void platform_support::on_key(int x, int y, unsigned key, unsigned flags) {}
	void platform_support::on_ctrl_change() {}
	void platform_support::on_draw() {}
	void platform_support::on_post_draw(void* raw_handler) {}
}

//----------------------------------------------------------------------------
int agg_main(int argc, char* argv[]);


//----------------------------------------------------------------------------
int main(int argc, char* argv[])
{
	return agg_main(argc, argv);
}
